/*
 * Copyright (C) 2008 the VideoLAN team
 *
 * This file is part of VLMa.
 *
 * VLMa is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * VLMa is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VLMa. If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.videolan.vlma.order;

import java.io.IOException;

import org.apache.log4j.Logger;
import org.videolan.vlma.exception.NoAvailableOrderSenderException;
import org.videolan.vlma.model.FilesChannel;
import org.videolan.vlma.model.Order;
import org.videolan.vlma.model.SatChannel;
import org.videolan.vlma.model.StreamChannel;
import org.videolan.vlma.model.DTTChannel;

/**
 * Dispatches the order to the appropriate OrderSender.
 *
 * @author Adrien Grand <jpountz at videolan.org>
 */
public class OrderSenderDispatcher extends OrderSender {

    private static final Logger logger = Logger.getLogger(OrderSenderDispatcher.class);

    /**
     * Returns the order sender associated to the provided media class.
     *
     * @param mediaClass the media class
     * @return the appropriate order sender
     */
    private OrderSender getOrderSenderForClass(Class clazz) throws NoAvailableOrderSenderException {
        OrderSender result;
        if (SatChannel.class.isAssignableFrom(clazz)) {
            result = new SatOrderSender();
        } else if (DTTChannel.class.isAssignableFrom(clazz)) {
            result = new DTTOrderSender();
        } else if (FilesChannel.class.isAssignableFrom(clazz)) {
            result = new FilesOrderSender();
        } else if (StreamChannel.class.isAssignableFrom(clazz)) {
            result = new StreamOrderSender();
        } else {
            throw new NoAvailableOrderSenderException("No available OrderSender for class " + clazz);
        }
        result.setCommandLogger(commandLogger);
        result.setConfiguration(configuration);
        return result;
    }


    @Override
    public void start(Order order) throws IOException {
        Class clazz = order.getMedias().getMediaClass();
        try {
            OrderSender orderSender = getOrderSenderForClass(clazz);
            orderSender.start(order);
        } catch (NoAvailableOrderSenderException e) {
            logger.warn("Couldn't find an appropriate OrderSender for the media class: " + clazz);
        }
    }

    @Override
    public void stop(Order order) throws IOException {
        Class clazz = order.getMedias().getMediaClass();
        try {
            OrderSender orderSender = getOrderSenderForClass(clazz);
            orderSender.stop(order);
        } catch (NoAvailableOrderSenderException e) {
            logger.warn("Couldn't find an appropriate OrderSender for the media class: " + clazz);
        }
    }

}
