/*
 * Copyright (C) 2006-2008 the VideoLAN team
 *
 * This file is part of VLMa.
 *
 * VLMa is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * VLMa is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VLMa. If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.videolan.vlma.notifier;

import java.io.IOException;
import java.util.List;

import org.apache.commons.configuration.Configuration;
import org.apache.log4j.Logger;

import rath.msnm.MSNMessenger;
import rath.msnm.SwitchboardSession;
import rath.msnm.UserStatus;
import rath.msnm.msg.MimeMessage;

/**
 * A MSN notifier.
 *
 * @author Adrien Grand <jpountz at videolan.org>
 */
public class MSNNotifier extends Notifier {

    private static final Logger logger = Logger.getLogger(IRCNotifier.class);

    private Configuration configuration;
    private MSNMessenger msnm;
    private String login;
    private String pass;
    private List<String> recipients;

    /**
     * Log into MSN.
     */
    @SuppressWarnings("unchecked")
    private void connect() {
        login = configuration.getString("vlma.notification.msn.login");
        pass = configuration.getString("vlma.notification.msn.pass");
        recipients = configuration.getList("vlma.notification.msn.recipients");
        msnm = new MSNMessenger(login, pass);
        msnm.setInitialStatus(UserStatus.ONLINE);
        msnm.login();
    }

    @Override
    public void init() {
        if(isMSNEnabled()) {
            connect();
        }
    }

    /**
     * Does the user want to be notified by MSN?
     *
     * @return true if there is one recipient or more
     */
    @SuppressWarnings("unchecked")
    private boolean isMSNEnabled() {
        List<String> tmp = configuration.getList("vlma.notification.msn.recipients");
        return (tmp != null && tmp.size() > 0);
    }

    @Override
    public void sendNotification(String message) {
        if (isMSNEnabled()) {
            if (msnm == null || !msnm.isLoggedIn()) {
                connect();
            }
            MimeMessage msg = new MimeMessage();
            msg.setKind(MimeMessage.KIND_MESSAGE);
            msg.setMessage(message);
            SwitchboardSession ss = null;
            for (String recipient : recipients) {
                ss = null;
                try {
                    while (ss == null) {
                       ss = msnm.doCallWait(recipient);
                    }
                    ss.sendInstantMessage(msg);
                } catch (IOException e) {
                    logger.error("Could not send message", e);
                } catch (InterruptedException e) {
                    logger.error("Could not send message", e);
                }
            }
        }
    }

    @Override
    public void destroy() {
        if (msnm != null && msnm.isLoggedIn())
            msnm.logout();
    }

    /**
     * Sets the configuration.
     *
     * @param configuration the configuration to set
     */
    public void setConfiguration(Configuration configuration) {
        this.configuration = configuration;
    }

}
