/*
 * Copyright (C) 2006-2008 the VideoLAN team
 *
 * This file is part of VLMa.
 *
 * VLMa is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * VLMa is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VLMa. If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.videolan.vlma.model;

import java.util.ArrayList;


/**
 * A media group. A media group is a set of media which are all available using
 * the same adapter.
 *
 * @author Sylvain Cadilhac <sylv at videolan.org>
 *
 */
public class MediaGroup extends ArrayList<Media> implements Comparable<MediaGroup> {

    private static final long serialVersionUID = 1123115519448157592L;

    /**
     * Compares groups according to their priority.
     *
     * @param o the object to compare to
     * @return 1 if the group has a higher priority, 0 if both groups have the
     *         same priority and -1 otherwise.
     */
    public int compareTo(MediaGroup o) {
        return o.getPriority().compareTo(getPriority());
    }

    /**
     * Compares the media group to the specified object.
     *
     * @return true if the media group are equal; false otherwise.
     */
    @Override
    public boolean equals(Object o) {
    if (o == null || !(o instanceof MediaGroup)) return false;
        MediaGroup g = (MediaGroup) o;
        if (g.size() != this.size())
            return false;
        for (Media media : this) {
            if (!g.contains(media))
                return false;
        }
        return true;
    }

    /**
     * Computes the MediaGroup hashCode.
     *
     * This implementation does not take into account the position of the media
     * in the group.
     */
    @Override
    public int hashCode() {
        int result = 0;
        for(Media media : this) {
            result += media.hashCode();
        }
        return result;
    }

    /**
     * Gives the media group priority. It is the sum of the priorities of all
     * media that belong to this group.
     *
     * @return the group priority
     */
    public Integer getPriority() {
        int p = 0;
        for (Media m : this) {
            p += m.getProgram().getPriority();
        }
        return p;
    }

    /**
     * Gives the true class of all Media instances in the group.
     *
     * @return the class of the Media instances
     */
    public Class getMediaClass() {
        if (!isEmpty()) {
            return get(0).getClass();
        }
        return null;
    }

}
