/*
 * Copyright (C) 2006-2008 the VideoLAN team
 *
 * This file is part of VLMa.
 * 
 * VLMa is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * 
 * VLMa is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with VLMa. If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.videolan.vlma.common.programs;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.videolan.vlma.common.IVlData;
import org.videolan.vlma.common.medias.IVlMedia;

/**
 * A set of multicast IP addresses. This set is used to give addresses to start
 * a new programmation.
 * 
 * @author SylV
 */
public class VlIpBank {
    private static final Logger logger = Logger.getLogger(VlIpBank.class);

    private HashMap<InetAddress, Boolean> usedIps;

    private IVlData data;

    /**
     * Initializes IP addresses according to the data interface.
     * 
     * @throws UnknownHostException
     *             address is invalid
     */
    public void initIps(boolean IPsDevel) throws UnknownHostException {
        usedIps = new HashMap<InetAddress, Boolean>();
        usedIps.clear();
        int iMin, iMax;
        if (IPsDevel == false) {
            iMin = 100;
            iMax = 120;
        } else {
            iMin = 200;
            iMax = 220;
        }
        for (int i = iMin; i < iMax; i++) {
            for (int j = 1; j < 255; j++) {
                byte[] b = { (byte) 239, (byte) 255, (byte) i, (byte) j };
                usedIps.put(InetAddress.getByAddress(b), Boolean.FALSE);
            }
        }
        for (IVlMedia m : data.getMedias()) {
            IVlProgram p = m.getProgram();
            if (p == null) {
                continue;
            }
            logger.log(Level.DEBUG, "The media " + m.getName()
                    + " has a prohram");
            if (usedIps.containsKey(p.getIp())) {
                usedIps.put(p.getIp(), Boolean.TRUE);
            }
        }
    }

    /**
     * Gives a new IP address to the programmation, the first which is not used.
     * 
     * @return the IP address
     */
    public InetAddress getIp() {
        Iterator<Map.Entry<InetAddress, Boolean>> k = usedIps.entrySet()
                .iterator();
        while (k.hasNext()) {
            Map.Entry<InetAddress, Boolean> i = k.next();
            if (!i.getValue().booleanValue()) {
                i.setValue(Boolean.TRUE);
                return i.getKey();
            }
        }
        return null;
    }

    /**
     * Constructs a new IP bank and sets the data interface.
     * 
     * @param data
     *            the data interface to set
     */
    public VlIpBank(IVlData data) {
        this.data = data;
    }

}
