/* SPDX-License-Identifier: GPL-2.0+ */
/*
 * Internal header file for scenes
 *
 * Copyright 2022 Google LLC
 * Written by Simon Glass <sjg@chromium.org>
 */

#ifndef __SCENE_INTERNAL_H
#define __SCENE_INTERNAL_H

/**
 * expo_lookup_scene_id() - Look up a scene ID
 *
 * @exp: Expo to use
 * @id: scene ID to look up
 * Returns: Scene for that ID, or NULL if none
 */
struct scene *expo_lookup_scene_id(struct expo *exp, uint scene_id);

/**
 * resolve_id() - Automatically allocate an ID if needed
 *
 * @exp: Expo to use
 * @id: ID to use, or 0 to auto-allocate one
 * Returns: Either @id, or the auto-allocated ID
 */
uint resolve_id(struct expo *exp, uint id);

/**
 * scene_obj_find() - Find an object in a scene
 *
 * Note that @type is used to restrict the search when the object type is known.
 * If any type is acceptable, set @type to SCENEOBJT_NONE
 *
 * @scn: Scene to search
 * @id: ID of object to find
 * @type: Type of the object, or SCENEOBJT_NONE to match any type
 * Returns: Object found, or NULL if not found
 */
void *scene_obj_find(struct scene *scn, uint id, enum scene_obj_t type);

/**
 * scene_obj_find_by_name() - Find an object in a scene by name
 *
 * @scn: Scene to search
 * @name: Name to search for
 */
void *scene_obj_find_by_name(struct scene *scn, const char *name);

/**
 * scene_obj_add() - Add a new object to a scene
 *
 * @scn: Scene to update
 * @name: Name to use (this is allocated by this call)
 * @id: ID to use for the new object (0 to allocate one)
 * @type: Type of object to add
 * @size: Size to allocate for the object, in bytes
 * @objp: Returns a pointer to the new object (must not be NULL)
 * Returns: ID number for the object (generally @id), or -ve on error
 */
int scene_obj_add(struct scene *scn, const char *name, uint id,
		  enum scene_obj_t type, uint size, struct scene_obj **objp);

/**
 * scene_obj_flag_clrset() - Adjust object flags
 *
 * @scn: Scene to update
 * @id: ID of object to update
 * @clr: Bits to clear in the object's flags
 * @set: Bits to set in the object's flags
 * Returns 0 if OK, -ENOENT if the object was not found
 */
int scene_obj_flag_clrset(struct scene *scn, uint id, uint clr, uint set);

/**
 * scene_calc_dims() - Calculate the dimensions of the scene objects
 *
 * Updates the width and height of all objects based on their contents
 *
 * @scn: Scene to update
 * @do_menus: true to calculate only menus, false to calculate everything else
 * Returns 0 if OK, -ENOTSUPP if there is no graphical console
 */
int scene_calc_dims(struct scene *scn, bool do_menus);

/**
 * scene_menu_arrange() - Set the position of things in the menu
 *
 * This updates any items associated with a menu to make sure they are
 * positioned correctly relative to the menu. It also selects the first item
 * if not already done
 *
 * @scn: Scene to update
 * @menu: Menu to process
 * Returns: 0 if OK, -ve on error
 */
int scene_menu_arrange(struct scene *scn, struct scene_obj_menu *menu);

/**
 * scene_apply_theme() - Apply a theme to a scene
 *
 * @scn: Scene to update
 * @theme: Theme to apply
 * Returns: 0 if OK, -ve on error
 */
int scene_apply_theme(struct scene *scn, struct expo_theme *theme);

/**
 * scene_menu_send_key() - Send a key to a menu for processing
 *
 * @scn: Scene to use
 * @menu: Menu to use
 * @key: Key code to send (KEY_...)
 * @event: Place to put any event which is generated by the key
 * Returns: 0 if OK, -ENOTTY if there is no current menu item, other -ve on other
 *	error
 */
int scene_menu_send_key(struct scene *scn, struct scene_obj_menu *menu, int key,
			struct expo_action *event);

/**
 * scene_menu_destroy() - Destroy a menu in a scene
 *
 * @scn: Scene to destroy
 */
void scene_menu_destroy(struct scene_obj_menu *menu);

/**
 * scene_menu_display() - Display a menu as text
 *
 * @menu: Menu to display
 * Returns: 0 if OK, -ENOENT if @id is invalid
 */
int scene_menu_display(struct scene_obj_menu *menu);

/**
 * scene_destroy() - Destroy a scene and all its memory
 *
 * @scn: Scene to destroy
 */
void scene_destroy(struct scene *scn);

/**
 * scene_render() - Render a scene
 *
 * This is called from expo_render()
 *
 * @scn: Scene to render
 * Returns: 0 if OK, -ve on error
 */
int scene_render(struct scene *scn);

/**
 * scene_send_key() - set a keypress to a scene
 *
 * @scn: Scene to receive the key
 * @key: Key to send (KEYCODE_UP)
 * @event: Returns resulting event from this keypress
 * Returns: 0 if OK, -ve on error
 */
int scene_send_key(struct scene *scn, int key, struct expo_action *event);

/**
 * scene_menu_render() - Render the background behind a menu
 *
 * @menu: Menu to render
 */
void scene_menu_render(struct scene_obj_menu *menu);

/**
 * scene_render_deps() - Render an object and its dependencies
 *
 * @scn: Scene to render
 * @id: Object ID to render (or 0 for none)
 * Returns: 0 if OK, -ve on error
 */
int scene_render_deps(struct scene *scn, uint id);

/**
 * scene_menu_render_deps() - Render a menu and its dependencies
 *
 * Renders the menu and all of its attached objects
 *
 * @scn: Scene to render
 * @menu: Menu render
 * Returns: 0 if OK, -ve on error
 */
int scene_menu_render_deps(struct scene *scn, struct scene_obj_menu *menu);

/**
 * scene_menu_calc_dims() - Calculate the dimensions of a menu
 *
 * Updates the width and height of the menu based on its contents
 *
 * @menu: Menu to update
 * Returns 0 if OK, -ENOTSUPP if there is no graphical console
 */
int scene_menu_calc_dims(struct scene_obj_menu *menu);

#endif /* __SCENE_INTERNAL_H */
