/*
 * Copyright (C) 2022-2023 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "GeneratedSerializers.h"
#include "GeneratedWebKitSecureCoding.h"

#include "ArgumentCoders.h"
#include "ConnectionHandle.h"
#if USE(SKIA)
#include "CoreIPCSkColorSpace.h"
#endif
#include "CoreIPCSkData.h"
#include "IPCEvent.h"
#include "IPCSemaphore.h"
#include "MessageFlags.h"
#include "StreamConnectionEncoder.h"
#include "WebPushDaemonConnectionConfiguration.h"
#include "WebPushMessage.h"
#include <WebCore/Color.h>
#include <WebCore/ColorTypes.h>

template<size_t...> struct MembersInCorrectOrder;
template<size_t onlyOffset> struct MembersInCorrectOrder<onlyOffset> {
    static constexpr bool value = true;
};
template<size_t firstOffset, size_t secondOffset, size_t... remainingOffsets> struct MembersInCorrectOrder<firstOffset, secondOffset, remainingOffsets...> {
    static constexpr bool value = firstOffset > secondOffset ? false : MembersInCorrectOrder<secondOffset, remainingOffsets...>::value;
};

template<uint64_t...> struct BitsInIncreasingOrder;
template<uint64_t onlyBit> struct BitsInIncreasingOrder<onlyBit> {
    static constexpr bool value = true;
};
template<uint64_t firstBit, uint64_t secondBit, uint64_t... remainingBits> struct BitsInIncreasingOrder<firstBit, secondBit, remainingBits...> {
    static constexpr bool value = firstBit == secondBit >> 1 && BitsInIncreasingOrder<secondBit, remainingBits...>::value;
};

template<bool, bool> struct VirtualTableAndRefCountOverhead;
template<> struct VirtualTableAndRefCountOverhead<true, true> : public RefCounted<VirtualTableAndRefCountOverhead<true, true>> {
    virtual ~VirtualTableAndRefCountOverhead() { }
};
template<> struct VirtualTableAndRefCountOverhead<false, true> : public RefCounted<VirtualTableAndRefCountOverhead<false, true>> { };
template<> struct VirtualTableAndRefCountOverhead<true, false> {
    virtual ~VirtualTableAndRefCountOverhead() { }
};
template<> struct VirtualTableAndRefCountOverhead<false, false> { };

IGNORE_WARNINGS_BEGIN("invalid-offsetof")

namespace IPC {

void ArgumentCoder<IPC::ConnectionHandle>::encode(Encoder& encoder, IPC::ConnectionHandle&& instance)
{
#if USE(UNIX_DOMAIN_SOCKETS)
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.m_handle)>, UnixFileDescriptor>);
#endif
#if OS(WINDOWS)
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.m_handle)>, Win32Handle>);
#endif
#if OS(DARWIN)
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.m_handle)>, MachSendRight>);
#endif
    struct ShouldBeSameSizeAsConnectionHandle : public VirtualTableAndRefCountOverhead<std::is_polymorphic_v<IPC::ConnectionHandle>, false> {
#if USE(UNIX_DOMAIN_SOCKETS)
        UnixFileDescriptor m_handle;
#endif
#if OS(WINDOWS)
        Win32Handle m_handle;
#endif
#if OS(DARWIN)
        MachSendRight m_handle;
#endif
    };
    static_assert(sizeof(ShouldBeSameSizeAsConnectionHandle) == sizeof(IPC::ConnectionHandle));
    static_assert(MembersInCorrectOrder < 0
#if USE(UNIX_DOMAIN_SOCKETS)
        , offsetof(IPC::ConnectionHandle, m_handle)
#endif
#if OS(WINDOWS)
        , offsetof(IPC::ConnectionHandle, m_handle)
#endif
#if OS(DARWIN)
        , offsetof(IPC::ConnectionHandle, m_handle)
#endif
    >::value);

#if USE(UNIX_DOMAIN_SOCKETS)
    encoder << WTFMove(instance.m_handle);
#endif
#if OS(WINDOWS)
    encoder << WTFMove(instance.m_handle);
#endif
#if OS(DARWIN)
    encoder << WTFMove(instance.m_handle);
#endif
}

std::optional<IPC::ConnectionHandle> ArgumentCoder<IPC::ConnectionHandle>::decode(Decoder& decoder)
{
#if USE(UNIX_DOMAIN_SOCKETS)
    auto m_handle = decoder.decode<UnixFileDescriptor>();
#endif
#if OS(WINDOWS)
    auto m_handle = decoder.decode<Win32Handle>();
#endif
#if OS(DARWIN)
    auto m_handle = decoder.decode<MachSendRight>();
#endif
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        IPC::ConnectionHandle {
#if USE(UNIX_DOMAIN_SOCKETS)
            WTFMove(*m_handle),
#endif
#if OS(WINDOWS)
            WTFMove(*m_handle),
#endif
#if OS(DARWIN)
            WTFMove(*m_handle)
#endif
        }
    };
}

void ArgumentCoder<IPC::Signal>::encode(Encoder& encoder, IPC::Signal&& instance)
{
#if PLATFORM(COCOA)
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.takeSendRight())>, MachSendRight>);
#endif
#if !PLATFORM(COCOA)
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.semaphore())>, IPC::Semaphore>);
#endif

#if PLATFORM(COCOA)
    encoder << instance.takeSendRight();
#endif
#if !PLATFORM(COCOA)
    encoder << instance.semaphore();
#endif
}

std::optional<IPC::Signal> ArgumentCoder<IPC::Signal>::decode(Decoder& decoder)
{
#if PLATFORM(COCOA)
    auto takeSendRight = decoder.decode<MachSendRight>();
#endif
#if !PLATFORM(COCOA)
    auto semaphore = decoder.decode<IPC::Semaphore>();
#endif
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        IPC::Signal {
#if PLATFORM(COCOA)
            WTFMove(*takeSendRight),
#endif
#if !PLATFORM(COCOA)
            WTFMove(*semaphore)
#endif
        }
    };
}

void ArgumentCoder<IPC::Semaphore>::encode(Encoder& encoder, const IPC::Semaphore& instance)
{
#if PLATFORM(COCOA)
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.createSendRight())>, MachSendRight>);
#endif
#if OS(WINDOWS)
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.win32Handle())>, Win32Handle>);
#endif
#if USE(UNIX_DOMAIN_SOCKETS)
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.duplicateDescriptor())>, UnixFileDescriptor>);
#endif

#if PLATFORM(COCOA)
    encoder << instance.createSendRight();
#endif
#if OS(WINDOWS)
    encoder << instance.win32Handle();
#endif
#if USE(UNIX_DOMAIN_SOCKETS)
    encoder << instance.duplicateDescriptor();
#endif
}

std::optional<IPC::Semaphore> ArgumentCoder<IPC::Semaphore>::decode(Decoder& decoder)
{
#if PLATFORM(COCOA)
    auto createSendRight = decoder.decode<MachSendRight>();
#endif
#if OS(WINDOWS)
    auto win32Handle = decoder.decode<Win32Handle>();
#endif
#if USE(UNIX_DOMAIN_SOCKETS)
    auto duplicateDescriptor = decoder.decode<UnixFileDescriptor>();
#endif
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        IPC::Semaphore {
#if PLATFORM(COCOA)
            WTFMove(*createSendRight),
#endif
#if OS(WINDOWS)
            WTFMove(*win32Handle),
#endif
#if USE(UNIX_DOMAIN_SOCKETS)
            WTFMove(*duplicateDescriptor)
#endif
        }
    };
}

void ArgumentCoder<WTF::URL>::encode(Encoder& encoder, const WTF::URL& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.string())>, String>);

    encoder << instance.string();
}

void ArgumentCoder<WTF::URL>::encode(StreamConnectionEncoder& encoder, const WTF::URL& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.string())>, String>);

    encoder << instance.string();
}

std::optional<WTF::URL> ArgumentCoder<WTF::URL>::decode(Decoder& decoder)
{
    auto string = decoder.decode<String>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::URL {
            WTFMove(*string)
        }
    };
}

void ArgumentCoder<WTF::UUID>::encode(Encoder& encoder, const WTF::UUID& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.high())>, uint64_t>);
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.low())>, uint64_t>);

    encoder << instance.high();
    encoder << instance.low();
}

std::optional<WTF::UUID> ArgumentCoder<WTF::UUID>::decode(Decoder& decoder)
{
    auto high = decoder.decode<uint64_t>();
    auto low = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(((static_cast<UInt128>(*high) << 64) | *low) != WTF::UUID::deletedValue))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::UUID {
            WTFMove(*high),
            WTFMove(*low)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::AXIDType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::AXIDType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::AXIDType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::AXIDType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebCore::AXIDType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::AXIDType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebCore::AXIDType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::FetchIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::FetchIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::FetchIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::FetchIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebCore::FetchIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::FetchIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebCore::FetchIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::LayerHostingContextIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::LayerHostingContextIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::LayerHostingContextIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::LayerHostingContextIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebCore::LayerHostingContextIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::LayerHostingContextIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebCore::LayerHostingContextIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::MediaSessionGroupIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::MediaSessionGroupIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::MediaSessionGroupIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::MediaSessionGroupIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebCore::MediaSessionGroupIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::MediaSessionGroupIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebCore::MediaSessionGroupIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::MediaUniqueIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::MediaUniqueIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::MediaUniqueIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::MediaUniqueIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebCore::MediaUniqueIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::MediaUniqueIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebCore::MediaUniqueIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::ProcessIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::ProcessIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::ProcessIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::ProcessIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebCore::ProcessIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::ProcessIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebCore::ProcessIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::PushSubscriptionIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::PushSubscriptionIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::PushSubscriptionIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::PushSubscriptionIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebCore::PushSubscriptionIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::PushSubscriptionIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebCore::PushSubscriptionIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::RealtimeMediaSourceIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::RealtimeMediaSourceIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::RealtimeMediaSourceIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::RealtimeMediaSourceIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebCore::RealtimeMediaSourceIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::RealtimeMediaSourceIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebCore::RealtimeMediaSourceIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::SWServerToContextConnectionIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::SWServerToContextConnectionIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::SWServerToContextConnectionIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::SWServerToContextConnectionIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebCore::SWServerToContextConnectionIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::SWServerToContextConnectionIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebCore::SWServerToContextConnectionIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::SharedWorkerIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::SharedWorkerIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::SharedWorkerIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::SharedWorkerIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebCore::SharedWorkerIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::SharedWorkerIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebCore::SharedWorkerIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::SpeechRecognitionConnectionClientIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::SpeechRecognitionConnectionClientIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::SpeechRecognitionConnectionClientIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::SpeechRecognitionConnectionClientIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebCore::SpeechRecognitionConnectionClientIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::SpeechRecognitionConnectionClientIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebCore::SpeechRecognitionConnectionClientIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::TextCheckingRequestIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::TextCheckingRequestIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::TextCheckingRequestIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::TextCheckingRequestIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebCore::TextCheckingRequestIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::TextCheckingRequestIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebCore::TextCheckingRequestIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::TextManipulationItemIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::TextManipulationItemIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::TextManipulationItemIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::TextManipulationItemIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebCore::TextManipulationItemIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::TextManipulationItemIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebCore::TextManipulationItemIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::TextManipulationTokenIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::TextManipulationTokenIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::TextManipulationTokenIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::TextManipulationTokenIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebCore::TextManipulationTokenIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::TextManipulationTokenIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebCore::TextManipulationTokenIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::WindowIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::WindowIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::WindowIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::WindowIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebCore::WindowIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::WindowIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebCore::WindowIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::WorkletGlobalScopeIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::WorkletGlobalScopeIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::WorkletGlobalScopeIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::WorkletGlobalScopeIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebCore::WorkletGlobalScopeIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::WorkletGlobalScopeIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebCore::WorkletGlobalScopeIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::AudioMediaStreamTrackRendererInternalUnitIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::AudioMediaStreamTrackRendererInternalUnitIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::AudioMediaStreamTrackRendererInternalUnitIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::AudioMediaStreamTrackRendererInternalUnitIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::AudioMediaStreamTrackRendererInternalUnitIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::AudioMediaStreamTrackRendererInternalUnitIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::AudioMediaStreamTrackRendererInternalUnitIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::ContentWorldIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::ContentWorldIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::ContentWorldIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::ContentWorldIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::ContentWorldIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::ContentWorldIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::ContentWorldIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::DownloadIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::DownloadIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::DownloadIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::DownloadIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::DownloadIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::DownloadIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::DownloadIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::ImageAnalysisRequestIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::ImageAnalysisRequestIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::ImageAnalysisRequestIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::ImageAnalysisRequestIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::ImageAnalysisRequestIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::ImageAnalysisRequestIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::ImageAnalysisRequestIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::MediaDevicePermissionRequestIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::MediaDevicePermissionRequestIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::MediaDevicePermissionRequestIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::MediaDevicePermissionRequestIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::MediaDevicePermissionRequestIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::MediaDevicePermissionRequestIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::MediaDevicePermissionRequestIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::MediaRecorderIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::MediaRecorderIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::MediaRecorderIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::MediaRecorderIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::MediaRecorderIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::MediaRecorderIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::MediaRecorderIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteAudioHardwareListenerIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::RemoteAudioHardwareListenerIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteAudioHardwareListenerIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::RemoteAudioHardwareListenerIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteAudioHardwareListenerIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteAudioHardwareListenerIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::RemoteAudioHardwareListenerIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteAudioSessionIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::RemoteAudioSessionIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteAudioSessionIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::RemoteAudioSessionIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteAudioSessionIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteAudioSessionIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::RemoteAudioSessionIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteCDMIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::RemoteCDMIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteCDMIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::RemoteCDMIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteCDMIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteCDMIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::RemoteCDMIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteCDMInstanceIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::RemoteCDMInstanceIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteCDMInstanceIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::RemoteCDMInstanceIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteCDMInstanceIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteCDMInstanceIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::RemoteCDMInstanceIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteCDMInstanceSessionIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::RemoteCDMInstanceSessionIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteCDMInstanceSessionIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::RemoteCDMInstanceSessionIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteCDMInstanceSessionIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteCDMInstanceSessionIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::RemoteCDMInstanceSessionIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteLegacyCDMIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::RemoteLegacyCDMIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteLegacyCDMIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::RemoteLegacyCDMIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteLegacyCDMIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteLegacyCDMIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::RemoteLegacyCDMIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteLegacyCDMSessionIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::RemoteLegacyCDMSessionIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteLegacyCDMSessionIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::RemoteLegacyCDMSessionIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteLegacyCDMSessionIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteLegacyCDMSessionIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::RemoteLegacyCDMSessionIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteMediaResourceIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::RemoteMediaResourceIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteMediaResourceIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::RemoteMediaResourceIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteMediaResourceIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteMediaResourceIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::RemoteMediaResourceIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteMediaSourceIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::RemoteMediaSourceIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteMediaSourceIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::RemoteMediaSourceIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteMediaSourceIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteMediaSourceIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::RemoteMediaSourceIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteRemoteCommandListenerIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::RemoteRemoteCommandListenerIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteRemoteCommandListenerIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::RemoteRemoteCommandListenerIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteRemoteCommandListenerIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteRemoteCommandListenerIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::RemoteRemoteCommandListenerIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteSourceBufferIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::RemoteSourceBufferIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteSourceBufferIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::RemoteSourceBufferIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteSourceBufferIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteSourceBufferIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::RemoteSourceBufferIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RetrieveRecordResponseBodyCallbackIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::RetrieveRecordResponseBodyCallbackIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RetrieveRecordResponseBodyCallbackIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::RetrieveRecordResponseBodyCallbackIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::RetrieveRecordResponseBodyCallbackIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RetrieveRecordResponseBodyCallbackIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::RetrieveRecordResponseBodyCallbackIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::SampleBufferDisplayLayerIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::SampleBufferDisplayLayerIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::SampleBufferDisplayLayerIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::SampleBufferDisplayLayerIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::SampleBufferDisplayLayerIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::SampleBufferDisplayLayerIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::SampleBufferDisplayLayerIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::ScriptMessageHandlerIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::ScriptMessageHandlerIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::ScriptMessageHandlerIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::ScriptMessageHandlerIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::ScriptMessageHandlerIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::ScriptMessageHandlerIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::ScriptMessageHandlerIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::ShapeDetectionIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::ShapeDetectionIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::ShapeDetectionIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::ShapeDetectionIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::ShapeDetectionIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::ShapeDetectionIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::ShapeDetectionIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::StorageAreaImplIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::StorageAreaImplIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::StorageAreaImplIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::StorageAreaImplIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::StorageAreaImplIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::StorageAreaImplIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::StorageAreaImplIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::StorageAreaMapIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::StorageAreaMapIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::StorageAreaMapIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::StorageAreaMapIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::StorageAreaMapIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::StorageAreaMapIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::StorageAreaMapIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::StorageNamespaceIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::StorageNamespaceIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::StorageNamespaceIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::StorageNamespaceIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::StorageNamespaceIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::StorageNamespaceIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::StorageNamespaceIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::UserScriptIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::UserScriptIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::UserScriptIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::UserScriptIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::UserScriptIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::UserScriptIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::UserScriptIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::UserStyleSheetIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::UserStyleSheetIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::UserStyleSheetIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::UserStyleSheetIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::UserStyleSheetIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::UserStyleSheetIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::UserStyleSheetIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::VisitedLinkTableIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::VisitedLinkTableIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::VisitedLinkTableIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::VisitedLinkTableIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::VisitedLinkTableIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::VisitedLinkTableIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::VisitedLinkTableIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::WCLayerTreeHostIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::WCLayerTreeHostIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::WCLayerTreeHostIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::WCLayerTreeHostIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::WCLayerTreeHostIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::WCLayerTreeHostIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::WCLayerTreeHostIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionControllerIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionControllerIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionControllerIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionControllerIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionControllerIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionControllerIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionControllerIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::XRDeviceIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::XRDeviceIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::XRDeviceIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::XRDeviceIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::XRDeviceIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::XRDeviceIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::XRDeviceIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::AttributedStringTextListIDType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::AttributedStringTextListIDType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::AttributedStringTextListIDType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::AttributedStringTextListIDType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebCore::AttributedStringTextListIDType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::AttributedStringTextListIDType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebCore::AttributedStringTextListIDType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::AttributedStringTextTableBlockIDType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::AttributedStringTextTableBlockIDType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::AttributedStringTextTableBlockIDType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::AttributedStringTextTableBlockIDType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebCore::AttributedStringTextTableBlockIDType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::AttributedStringTextTableBlockIDType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebCore::AttributedStringTextTableBlockIDType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::AttributedStringTextTableIDType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::AttributedStringTextTableIDType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::AttributedStringTextTableIDType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::AttributedStringTextTableIDType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebCore::AttributedStringTextTableIDType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::AttributedStringTextTableIDType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebCore::AttributedStringTextTableIDType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::BackForwardItemIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::BackForwardItemIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::BackForwardItemIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::BackForwardItemIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebCore::BackForwardItemIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::BackForwardItemIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebCore::BackForwardItemIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::DictationContextType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::DictationContextType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::DictationContextType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::DictationContextType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebCore::DictationContextType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::DictationContextType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebCore::DictationContextType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::FrameIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::FrameIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::FrameIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::FrameIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebCore::FrameIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::FrameIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebCore::FrameIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::MediaKeySystemRequestIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::MediaKeySystemRequestIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::MediaKeySystemRequestIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::MediaKeySystemRequestIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebCore::MediaKeySystemRequestIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::MediaKeySystemRequestIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebCore::MediaKeySystemRequestIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::MediaPlayerIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::MediaPlayerIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::MediaPlayerIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::MediaPlayerIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebCore::MediaPlayerIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::MediaPlayerIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebCore::MediaPlayerIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::MediaPlayerClientIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::MediaPlayerClientIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::MediaPlayerClientIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::MediaPlayerClientIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebCore::MediaPlayerClientIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::MediaPlayerClientIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebCore::MediaPlayerClientIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::MediaSessionIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::MediaSessionIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::MediaSessionIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::MediaSessionIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebCore::MediaSessionIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::MediaSessionIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebCore::MediaSessionIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::ModelPlayerIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::ModelPlayerIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::ModelPlayerIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::ModelPlayerIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebCore::ModelPlayerIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::ModelPlayerIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebCore::ModelPlayerIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::PageIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::PageIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::PageIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::PageIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebCore::PageIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::PageIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebCore::PageIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::PlatformLayerIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::PlatformLayerIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::PlatformLayerIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::PlatformLayerIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebCore::PlatformLayerIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::PlatformLayerIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebCore::PlatformLayerIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::PlaybackTargetClientContextIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::PlaybackTargetClientContextIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::PlaybackTargetClientContextIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::PlaybackTargetClientContextIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebCore::PlaybackTargetClientContextIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::PlaybackTargetClientContextIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebCore::PlaybackTargetClientContextIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::SharedWorkerObjectIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::SharedWorkerObjectIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::SharedWorkerObjectIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::SharedWorkerObjectIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebCore::SharedWorkerObjectIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::SharedWorkerObjectIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebCore::SharedWorkerObjectIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::SleepDisablerIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::SleepDisablerIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::SleepDisablerIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::SleepDisablerIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebCore::SleepDisablerIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::SleepDisablerIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebCore::SleepDisablerIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::UserGestureTokenIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::UserGestureTokenIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::UserGestureTokenIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::UserGestureTokenIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebCore::UserGestureTokenIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::UserGestureTokenIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebCore::UserGestureTokenIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::UserMediaRequestIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::UserMediaRequestIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::UserMediaRequestIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::UserMediaRequestIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebCore::UserMediaRequestIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::UserMediaRequestIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebCore::UserMediaRequestIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::ScrollingNodeIDType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::ScrollingNodeIDType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::ScrollingNodeIDType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebCore::ScrollingNodeIDType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebCore::ScrollingNodeIDType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebCore::ScrollingNodeIDType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebCore::ScrollingNodeIDType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::AuthenticationChallengeIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::AuthenticationChallengeIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::AuthenticationChallengeIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::AuthenticationChallengeIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::AuthenticationChallengeIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::AuthenticationChallengeIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::AuthenticationChallengeIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::DataTaskIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::DataTaskIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::DataTaskIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::DataTaskIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::DataTaskIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::DataTaskIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::DataTaskIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::DisplayLinkObserverIDType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::DisplayLinkObserverIDType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::DisplayLinkObserverIDType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::DisplayLinkObserverIDType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::DisplayLinkObserverIDType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::DisplayLinkObserverIDType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::DisplayLinkObserverIDType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::DrawingAreaIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::DrawingAreaIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::DrawingAreaIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::DrawingAreaIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::DrawingAreaIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::DrawingAreaIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::DrawingAreaIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::GeolocationIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::GeolocationIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::GeolocationIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::GeolocationIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::GeolocationIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::GeolocationIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::GeolocationIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::IPCConnectionTesterIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::IPCConnectionTesterIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::IPCConnectionTesterIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::IPCConnectionTesterIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::IPCConnectionTesterIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::IPCConnectionTesterIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::IPCConnectionTesterIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::IPCStreamTesterIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::IPCStreamTesterIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::IPCStreamTesterIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::IPCStreamTesterIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::IPCStreamTesterIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::IPCStreamTesterIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::IPCStreamTesterIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::MarkSurfacesAsVolatileRequestIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::MarkSurfacesAsVolatileRequestIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::MarkSurfacesAsVolatileRequestIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::MarkSurfacesAsVolatileRequestIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::MarkSurfacesAsVolatileRequestIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::MarkSurfacesAsVolatileRequestIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::MarkSurfacesAsVolatileRequestIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::NetworkResourceLoadIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::NetworkResourceLoadIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::NetworkResourceLoadIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::NetworkResourceLoadIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::NetworkResourceLoadIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::NetworkResourceLoadIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::NetworkResourceLoadIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::PDFPluginIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::PDFPluginIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::PDFPluginIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::PDFPluginIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::PDFPluginIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::PDFPluginIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::PDFPluginIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::PageGroupIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::PageGroupIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::PageGroupIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::PageGroupIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::PageGroupIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::PageGroupIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::PageGroupIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteAudioDestinationIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::RemoteAudioDestinationIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteAudioDestinationIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::RemoteAudioDestinationIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteAudioDestinationIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteAudioDestinationIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::RemoteAudioDestinationIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteImageBufferSetIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::RemoteImageBufferSetIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteImageBufferSetIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::RemoteImageBufferSetIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteImageBufferSetIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::RemoteImageBufferSetIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::RemoteImageBufferSetIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::TapIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::TapIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::TapIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::TapIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::TapIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::TapIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::TapIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::TextCheckerRequestType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::TextCheckerRequestType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::TextCheckerRequestType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::TextCheckerRequestType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::TextCheckerRequestType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::TextCheckerRequestType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::TextCheckerRequestType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::UserContentControllerIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::UserContentControllerIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::UserContentControllerIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::UserContentControllerIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::UserContentControllerIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::UserContentControllerIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::UserContentControllerIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionContextIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionContextIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionContextIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionContextIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionContextIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionContextIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionContextIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionFrameIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionFrameIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionFrameIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionFrameIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionFrameIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionFrameIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionFrameIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionPortChannelIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionPortChannelIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionPortChannelIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionPortChannelIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionPortChannelIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionPortChannelIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionPortChannelIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionTabIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionTabIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionTabIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionTabIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionTabIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionTabIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionTabIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionWindowIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionWindowIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionWindowIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionWindowIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionWindowIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionWindowIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::WebExtensionWindowIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::WebPageProxyIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::WebPageProxyIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::WebPageProxyIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::WebPageProxyIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::WebPageProxyIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::WebPageProxyIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::WebPageProxyIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::WebTransportSessionIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::WebTransportSessionIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::WebTransportSessionIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::WebTransportSessionIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::WebTransportSessionIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::WebTransportSessionIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::WebTransportSessionIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::WebTransportStreamIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::WebTransportStreamIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::WebTransportStreamIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableObjectIdentifier<WebKit::WebTransportStreamIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableObjectIdentifier<WebKit::WebTransportStreamIdentifierType>> ArgumentCoder<WTF::LegacyNullableObjectIdentifier<WebKit::WebTransportStreamIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableObjectIdentifier<WebKit::WebTransportStreamIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::ObjectIdentifier<WebKit::WebURLSchemeHandler>>::encode(Encoder& encoder, const WTF::ObjectIdentifier<WebKit::WebURLSchemeHandler>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::ObjectIdentifier<WebKit::WebURLSchemeHandler>>::encode(StreamConnectionEncoder& encoder, const WTF::ObjectIdentifier<WebKit::WebURLSchemeHandler>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::ObjectIdentifier<WebKit::WebURLSchemeHandler>> ArgumentCoder<WTF::ObjectIdentifier<WebKit::WebURLSchemeHandler>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::ObjectIdentifier<WebKit::WebURLSchemeHandler> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::ObjectIdentifier<WebCore::BackgroundFetchRecordIdentifierType>>::encode(Encoder& encoder, const WTF::ObjectIdentifier<WebCore::BackgroundFetchRecordIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::ObjectIdentifier<WebCore::BackgroundFetchRecordIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::ObjectIdentifier<WebCore::BackgroundFetchRecordIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::ObjectIdentifier<WebCore::BackgroundFetchRecordIdentifierType>> ArgumentCoder<WTF::ObjectIdentifier<WebCore::BackgroundFetchRecordIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::ObjectIdentifier<WebCore::BackgroundFetchRecordIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::ObjectIdentifier<WebCore::ElementIdentifierType>>::encode(Encoder& encoder, const WTF::ObjectIdentifier<WebCore::ElementIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::ObjectIdentifier<WebCore::ElementIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::ObjectIdentifier<WebCore::ElementIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::ObjectIdentifier<WebCore::ElementIdentifierType>> ArgumentCoder<WTF::ObjectIdentifier<WebCore::ElementIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::ObjectIdentifier<WebCore::ElementIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::ObjectIdentifier<WebCore::ImageDecoderIdentifierType>>::encode(Encoder& encoder, const WTF::ObjectIdentifier<WebCore::ImageDecoderIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::ObjectIdentifier<WebCore::ImageDecoderIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::ObjectIdentifier<WebCore::ImageDecoderIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::ObjectIdentifier<WebCore::ImageDecoderIdentifierType>> ArgumentCoder<WTF::ObjectIdentifier<WebCore::ImageDecoderIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::ObjectIdentifier<WebCore::ImageDecoderIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::ObjectIdentifier<WebCore::ImageOverlayDataDetectionResultIdentifierType>>::encode(Encoder& encoder, const WTF::ObjectIdentifier<WebCore::ImageOverlayDataDetectionResultIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::ObjectIdentifier<WebCore::ImageOverlayDataDetectionResultIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::ObjectIdentifier<WebCore::ImageOverlayDataDetectionResultIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::ObjectIdentifier<WebCore::ImageOverlayDataDetectionResultIdentifierType>> ArgumentCoder<WTF::ObjectIdentifier<WebCore::ImageOverlayDataDetectionResultIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::ObjectIdentifier<WebCore::ImageOverlayDataDetectionResultIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::ObjectIdentifier<WebCore::InbandGenericCueIdentifierType>>::encode(Encoder& encoder, const WTF::ObjectIdentifier<WebCore::InbandGenericCueIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::ObjectIdentifier<WebCore::InbandGenericCueIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::ObjectIdentifier<WebCore::InbandGenericCueIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::ObjectIdentifier<WebCore::InbandGenericCueIdentifierType>> ArgumentCoder<WTF::ObjectIdentifier<WebCore::InbandGenericCueIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::ObjectIdentifier<WebCore::InbandGenericCueIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::ObjectIdentifier<WebCore::NavigationIdentifierType>>::encode(Encoder& encoder, const WTF::ObjectIdentifier<WebCore::NavigationIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::ObjectIdentifier<WebCore::NavigationIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::ObjectIdentifier<WebCore::NavigationIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::ObjectIdentifier<WebCore::NavigationIdentifierType>> ArgumentCoder<WTF::ObjectIdentifier<WebCore::NavigationIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::ObjectIdentifier<WebCore::NavigationIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::ResourceLoader>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::ResourceLoader>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::ResourceLoader>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::ResourceLoader>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::ResourceLoader>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::ResourceLoader>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebCore::ResourceLoader> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebSocketChannel>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebSocketChannel>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebSocketChannel>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebSocketChannel>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebSocketChannel>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebSocketChannel>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebSocketChannel> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebSocketChannelClient>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebSocketChannelClient>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebSocketChannelClient>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebSocketChannelClient>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebSocketChannelClient>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebSocketChannelClient>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebSocketChannelClient> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebSocketChannelInspector>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebSocketChannelInspector>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebSocketChannelInspector>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebSocketChannelInspector>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebSocketChannelInspector>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebSocketChannelInspector>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebSocketChannelInspector> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<IPC::ConnectionSyncRequestIDType>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<IPC::ConnectionSyncRequestIDType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<IPC::ConnectionSyncRequestIDType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<IPC::ConnectionSyncRequestIDType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<IPC::ConnectionSyncRequestIDType>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<IPC::ConnectionSyncRequestIDType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<IPC::ConnectionSyncRequestIDType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<JSC::MicrotaskIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<JSC::MicrotaskIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<JSC::MicrotaskIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<JSC::MicrotaskIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<JSC::MicrotaskIdentifierType>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<JSC::MicrotaskIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<JSC::MicrotaskIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::BroadcastChannelIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::BroadcastChannelIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::BroadcastChannelIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::BroadcastChannelIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::BroadcastChannelIdentifierType>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::BroadcastChannelIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebCore::BroadcastChannelIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::DOMCacheIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::DOMCacheIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::DOMCacheIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::DOMCacheIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::DOMCacheIdentifierType>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::DOMCacheIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebCore::DOMCacheIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::DisplayConfigurationClientIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::DisplayConfigurationClientIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::DisplayConfigurationClientIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::DisplayConfigurationClientIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::DisplayConfigurationClientIdentifierType>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::DisplayConfigurationClientIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebCore::DisplayConfigurationClientIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::FileSystemHandleIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::FileSystemHandleIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::FileSystemHandleIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::FileSystemHandleIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::FileSystemHandleIdentifierType>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::FileSystemHandleIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebCore::FileSystemHandleIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::FileSystemSyncAccessHandleIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::FileSystemSyncAccessHandleIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::FileSystemSyncAccessHandleIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::FileSystemSyncAccessHandleIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::FileSystemSyncAccessHandleIdentifierType>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::FileSystemSyncAccessHandleIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebCore::FileSystemSyncAccessHandleIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::IDBObjectStoreIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::IDBObjectStoreIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::IDBObjectStoreIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::IDBObjectStoreIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::IDBObjectStoreIdentifierType>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::IDBObjectStoreIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebCore::IDBObjectStoreIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::LibWebRTCSocketIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::LibWebRTCSocketIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::LibWebRTCSocketIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::LibWebRTCSocketIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::LibWebRTCSocketIdentifierType>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::LibWebRTCSocketIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebCore::LibWebRTCSocketIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::MainThreadPermissionObserverIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::MainThreadPermissionObserverIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::MainThreadPermissionObserverIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::MainThreadPermissionObserverIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::MainThreadPermissionObserverIdentifierType>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::MainThreadPermissionObserverIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebCore::MainThreadPermissionObserverIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::OpaqueOriginIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::OpaqueOriginIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::OpaqueOriginIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::OpaqueOriginIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::OpaqueOriginIdentifierType>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::OpaqueOriginIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebCore::OpaqueOriginIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::PortIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::PortIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::PortIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::PortIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::PortIdentifierType>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::PortIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebCore::PortIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::RTCDataChannelLocalIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::RTCDataChannelLocalIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::RTCDataChannelLocalIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::RTCDataChannelLocalIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::RTCDataChannelLocalIdentifierType>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::RTCDataChannelLocalIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebCore::RTCDataChannelLocalIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::RTCRtpScriptTransformerIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::RTCRtpScriptTransformerIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::RTCRtpScriptTransformerIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::RTCRtpScriptTransformerIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::RTCRtpScriptTransformerIdentifierType>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::RTCRtpScriptTransformerIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebCore::RTCRtpScriptTransformerIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::RenderingResourceIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::RenderingResourceIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::RenderingResourceIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::RenderingResourceIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::RenderingResourceIdentifierType>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::RenderingResourceIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebCore::RenderingResourceIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::ServiceWorkerIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::ServiceWorkerIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::ServiceWorkerIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::ServiceWorkerIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::ServiceWorkerIdentifierType>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::ServiceWorkerIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebCore::ServiceWorkerIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::ServiceWorkerJobIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::ServiceWorkerJobIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::ServiceWorkerJobIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::ServiceWorkerJobIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::ServiceWorkerJobIdentifierType>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::ServiceWorkerJobIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebCore::ServiceWorkerJobIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::ServiceWorkerRegistrationIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::ServiceWorkerRegistrationIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::ServiceWorkerRegistrationIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::ServiceWorkerRegistrationIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::ServiceWorkerRegistrationIdentifierType>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::ServiceWorkerRegistrationIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebCore::ServiceWorkerRegistrationIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::IDBDatabaseConnectionIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::IDBDatabaseConnectionIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::IDBDatabaseConnectionIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::IDBDatabaseConnectionIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::IDBDatabaseConnectionIdentifierType>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::IDBDatabaseConnectionIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebCore::IDBDatabaseConnectionIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebLockIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebLockIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebLockIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebLockIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebLockIdentifierType>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebLockIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebLockIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebSocketIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebSocketIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebSocketIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebSocketIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebSocketIdentifierType>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebSocketIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebSocketIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WorkerFileSystemStorageConnectionCallbackIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WorkerFileSystemStorageConnectionCallbackIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WorkerFileSystemStorageConnectionCallbackIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WorkerFileSystemStorageConnectionCallbackIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WorkerFileSystemStorageConnectionCallbackIdentifierType>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WorkerFileSystemStorageConnectionCallbackIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WorkerFileSystemStorageConnectionCallbackIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::GPUProcessConnectionIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebKit::GPUProcessConnectionIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::GPUProcessConnectionIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebKit::GPUProcessConnectionIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::GPUProcessConnectionIdentifierType>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::GPUProcessConnectionIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebKit::GPUProcessConnectionIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::GraphicsContextGLIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebKit::GraphicsContextGLIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::GraphicsContextGLIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebKit::GraphicsContextGLIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::GraphicsContextGLIdentifierType>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::GraphicsContextGLIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebKit::GraphicsContextGLIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::LegacyCustomProtocolIDType>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebKit::LegacyCustomProtocolIDType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::LegacyCustomProtocolIDType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebKit::LegacyCustomProtocolIDType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::LegacyCustomProtocolIDType>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::LegacyCustomProtocolIDType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebKit::LegacyCustomProtocolIDType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::LibWebRTCResolverIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebKit::LibWebRTCResolverIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::LibWebRTCResolverIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebKit::LibWebRTCResolverIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::LibWebRTCResolverIdentifierType>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::LibWebRTCResolverIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebKit::LibWebRTCResolverIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::QuotaIncreaseRequestIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebKit::QuotaIncreaseRequestIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::QuotaIncreaseRequestIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebKit::QuotaIncreaseRequestIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::QuotaIncreaseRequestIdentifierType>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::QuotaIncreaseRequestIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebKit::QuotaIncreaseRequestIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::RemoteVideoFrameIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebKit::RemoteVideoFrameIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::RemoteVideoFrameIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebKit::RemoteVideoFrameIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::RemoteVideoFrameIdentifierType>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::RemoteVideoFrameIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebKit::RemoteVideoFrameIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::RenderingBackendIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebKit::RenderingBackendIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::RenderingBackendIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebKit::RenderingBackendIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::RenderingBackendIdentifierType>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::RenderingBackendIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebKit::RenderingBackendIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::VideoDecoderIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebKit::VideoDecoderIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::VideoDecoderIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebKit::VideoDecoderIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::VideoDecoderIdentifierType>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::VideoDecoderIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebKit::VideoDecoderIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::VideoEncoderIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebKit::VideoEncoderIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::VideoEncoderIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebKit::VideoEncoderIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::VideoEncoderIdentifierType>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::VideoEncoderIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebKit::VideoEncoderIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::WCContentBufferIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebKit::WCContentBufferIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::WCContentBufferIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebKit::WCContentBufferIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::WCContentBufferIdentifierType>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::WCContentBufferIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebKit::WCContentBufferIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::WebGPUIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebKit::WebGPUIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::WebGPUIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebKit::WebGPUIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::WebGPUIdentifierType>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::WebGPUIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebKit::WebGPUIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<TestWebKitAPI::TestedObjectIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<TestWebKitAPI::TestedObjectIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<TestWebKitAPI::TestedObjectIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<TestWebKitAPI::TestedObjectIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<TestWebKitAPI::TestedObjectIdentifierType>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<TestWebKitAPI::TestedObjectIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<TestWebKitAPI::TestedObjectIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<IPC::AsyncReplyIDType>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<IPC::AsyncReplyIDType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<IPC::AsyncReplyIDType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<IPC::AsyncReplyIDType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<IPC::AsyncReplyIDType>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<IPC::AsyncReplyIDType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<IPC::AsyncReplyIDType> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebSocketFrame>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebSocketFrame>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebSocketFrame>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebSocketFrame>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebSocketFrame>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebSocketFrame>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebCore::WebSocketFrame> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::ConnectionAsyncReplyHandler>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebKit::ConnectionAsyncReplyHandler>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::ConnectionAsyncReplyHandler>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebKit::ConnectionAsyncReplyHandler>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::ConnectionAsyncReplyHandler>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::ConnectionAsyncReplyHandler>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebKit::ConnectionAsyncReplyHandler> {
            WTFMove(*toUInt64)
        }
    };
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::StorageAreaIdentifierType>>::encode(Encoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebKit::StorageAreaIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

void ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::StorageAreaIdentifierType>>::encode(StreamConnectionEncoder& encoder, const WTF::LegacyNullableAtomicObjectIdentifier<WebKit::StorageAreaIdentifierType>& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.toUInt64())>, uint64_t>);

    encoder << instance.toUInt64();
}

std::optional<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::StorageAreaIdentifierType>> ArgumentCoder<WTF::LegacyNullableAtomicObjectIdentifier<WebKit::StorageAreaIdentifierType>>::decode(Decoder& decoder)
{
    auto toUInt64 = decoder.decode<uint64_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;

    if (!(WTF::ObjectIdentifierGenericBase<uint64_t>::isValidIdentifier(*toUInt64)))
        return std::nullopt;
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WTF::LegacyNullableAtomicObjectIdentifier<WebKit::StorageAreaIdentifierType> {
            WTFMove(*toUInt64)
        }
    };
}

#if USE(CG)
void ArgumentCoder<CGSize>::encode(Encoder& encoder, const CGSize& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.width)>, CGFloat>);
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.height)>, CGFloat>);
    struct ShouldBeSameSizeAsCGSize : public VirtualTableAndRefCountOverhead<std::is_polymorphic_v<CGSize>, false> {
        CGFloat width;
        CGFloat height;
    };
    static_assert(sizeof(ShouldBeSameSizeAsCGSize) == sizeof(CGSize));
    static_assert(MembersInCorrectOrder < 0
        , offsetof(CGSize, width)
        , offsetof(CGSize, height)
    >::value);

    encoder << instance.width;
    encoder << instance.height;
}

void ArgumentCoder<CGSize>::encode(StreamConnectionEncoder& encoder, const CGSize& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.width)>, CGFloat>);
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.height)>, CGFloat>);
    struct ShouldBeSameSizeAsCGSize : public VirtualTableAndRefCountOverhead<std::is_polymorphic_v<CGSize>, false> {
        CGFloat width;
        CGFloat height;
    };
    static_assert(sizeof(ShouldBeSameSizeAsCGSize) == sizeof(CGSize));
    static_assert(MembersInCorrectOrder < 0
        , offsetof(CGSize, width)
        , offsetof(CGSize, height)
    >::value);

    encoder << instance.width;
    encoder << instance.height;
}

std::optional<CGSize> ArgumentCoder<CGSize>::decode(Decoder& decoder)
{
    auto width = decoder.decode<CGFloat>();
    auto height = decoder.decode<CGFloat>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        CGSize {
            WTFMove(*width),
            WTFMove(*height)
        }
    };
}

#endif

#if USE(CG)
void ArgumentCoder<CGPoint>::encode(Encoder& encoder, const CGPoint& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.x)>, CGFloat>);
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.y)>, CGFloat>);
    struct ShouldBeSameSizeAsCGPoint : public VirtualTableAndRefCountOverhead<std::is_polymorphic_v<CGPoint>, false> {
        CGFloat x;
        CGFloat y;
    };
    static_assert(sizeof(ShouldBeSameSizeAsCGPoint) == sizeof(CGPoint));
    static_assert(MembersInCorrectOrder < 0
        , offsetof(CGPoint, x)
        , offsetof(CGPoint, y)
    >::value);

    encoder << instance.x;
    encoder << instance.y;
}

std::optional<CGPoint> ArgumentCoder<CGPoint>::decode(Decoder& decoder)
{
    auto x = decoder.decode<CGFloat>();
    auto y = decoder.decode<CGFloat>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        CGPoint {
            WTFMove(*x),
            WTFMove(*y)
        }
    };
}

#endif

#if USE(CG)
void ArgumentCoder<CGRect>::encode(Encoder& encoder, const CGRect& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.origin)>, CGPoint>);
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.size)>, CGSize>);
    struct ShouldBeSameSizeAsCGRect : public VirtualTableAndRefCountOverhead<std::is_polymorphic_v<CGRect>, false> {
        CGPoint origin;
        CGSize size;
    };
    static_assert(sizeof(ShouldBeSameSizeAsCGRect) == sizeof(CGRect));
    static_assert(MembersInCorrectOrder < 0
        , offsetof(CGRect, origin)
        , offsetof(CGRect, size)
    >::value);

    encoder << instance.origin;
    encoder << instance.size;
}

std::optional<CGRect> ArgumentCoder<CGRect>::decode(Decoder& decoder)
{
    auto origin = decoder.decode<CGPoint>();
    auto size = decoder.decode<CGSize>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        CGRect {
            WTFMove(*origin),
            WTFMove(*size)
        }
    };
}

#endif

void ArgumentCoder<WebCore::PackedColor::RGBA>::encode(Encoder& encoder, const WebCore::PackedColor::RGBA& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.value)>, uint32_t>);
    struct ShouldBeSameSizeAsRGBA : public VirtualTableAndRefCountOverhead<std::is_polymorphic_v<WebCore::PackedColor::RGBA>, false> {
        uint32_t value;
    };
    static_assert(sizeof(ShouldBeSameSizeAsRGBA) == sizeof(WebCore::PackedColor::RGBA));
    static_assert(MembersInCorrectOrder < 0
        , offsetof(WebCore::PackedColor::RGBA, value)
    >::value);

    encoder << instance.value;
}

void ArgumentCoder<WebCore::PackedColor::RGBA>::encode(StreamConnectionEncoder& encoder, const WebCore::PackedColor::RGBA& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.value)>, uint32_t>);
    struct ShouldBeSameSizeAsRGBA : public VirtualTableAndRefCountOverhead<std::is_polymorphic_v<WebCore::PackedColor::RGBA>, false> {
        uint32_t value;
    };
    static_assert(sizeof(ShouldBeSameSizeAsRGBA) == sizeof(WebCore::PackedColor::RGBA));
    static_assert(MembersInCorrectOrder < 0
        , offsetof(WebCore::PackedColor::RGBA, value)
    >::value);

    encoder << instance.value;
}

std::optional<WebCore::PackedColor::RGBA> ArgumentCoder<WebCore::PackedColor::RGBA>::decode(Decoder& decoder)
{
    auto value = decoder.decode<uint32_t>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WebCore::PackedColor::RGBA {
            WTFMove(*value)
        }
    };
}

void ArgumentCoder<WebCore::OutOfLineColorDataForIPC>::encode(Encoder& encoder, const WebCore::OutOfLineColorDataForIPC& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.colorSpace)>, WebCore::ColorSpace>);
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.c1)>, float>);
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.c2)>, float>);
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.c3)>, float>);
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.alpha)>, float>);
    struct ShouldBeSameSizeAsOutOfLineColorDataForIPC : public VirtualTableAndRefCountOverhead<std::is_polymorphic_v<WebCore::OutOfLineColorDataForIPC>, false> {
        WebCore::ColorSpace colorSpace;
        float c1;
        float c2;
        float c3;
        float alpha;
    };
    static_assert(sizeof(ShouldBeSameSizeAsOutOfLineColorDataForIPC) == sizeof(WebCore::OutOfLineColorDataForIPC));
    static_assert(MembersInCorrectOrder < 0
        , offsetof(WebCore::OutOfLineColorDataForIPC, colorSpace)
        , offsetof(WebCore::OutOfLineColorDataForIPC, c1)
        , offsetof(WebCore::OutOfLineColorDataForIPC, c2)
        , offsetof(WebCore::OutOfLineColorDataForIPC, c3)
        , offsetof(WebCore::OutOfLineColorDataForIPC, alpha)
    >::value);

    encoder << instance.colorSpace;
    encoder << instance.c1;
    encoder << instance.c2;
    encoder << instance.c3;
    encoder << instance.alpha;
}

void ArgumentCoder<WebCore::OutOfLineColorDataForIPC>::encode(StreamConnectionEncoder& encoder, const WebCore::OutOfLineColorDataForIPC& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.colorSpace)>, WebCore::ColorSpace>);
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.c1)>, float>);
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.c2)>, float>);
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.c3)>, float>);
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.alpha)>, float>);
    struct ShouldBeSameSizeAsOutOfLineColorDataForIPC : public VirtualTableAndRefCountOverhead<std::is_polymorphic_v<WebCore::OutOfLineColorDataForIPC>, false> {
        WebCore::ColorSpace colorSpace;
        float c1;
        float c2;
        float c3;
        float alpha;
    };
    static_assert(sizeof(ShouldBeSameSizeAsOutOfLineColorDataForIPC) == sizeof(WebCore::OutOfLineColorDataForIPC));
    static_assert(MembersInCorrectOrder < 0
        , offsetof(WebCore::OutOfLineColorDataForIPC, colorSpace)
        , offsetof(WebCore::OutOfLineColorDataForIPC, c1)
        , offsetof(WebCore::OutOfLineColorDataForIPC, c2)
        , offsetof(WebCore::OutOfLineColorDataForIPC, c3)
        , offsetof(WebCore::OutOfLineColorDataForIPC, alpha)
    >::value);

    encoder << instance.colorSpace;
    encoder << instance.c1;
    encoder << instance.c2;
    encoder << instance.c3;
    encoder << instance.alpha;
}

std::optional<WebCore::OutOfLineColorDataForIPC> ArgumentCoder<WebCore::OutOfLineColorDataForIPC>::decode(Decoder& decoder)
{
    auto colorSpace = decoder.decode<WebCore::ColorSpace>();
    auto c1 = decoder.decode<float>();
    auto c2 = decoder.decode<float>();
    auto c3 = decoder.decode<float>();
    auto alpha = decoder.decode<float>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WebCore::OutOfLineColorDataForIPC {
            WTFMove(*colorSpace),
            WTFMove(*c1),
            WTFMove(*c2),
            WTFMove(*c3),
            WTFMove(*alpha)
        }
    };
}

void ArgumentCoder<WebCore::ColorDataForIPC>::encode(Encoder& encoder, const WebCore::ColorDataForIPC& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.isSemantic)>, bool>);
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.usesFunctionSerialization)>, bool>);
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.data)>, std::variant<WebCore::PackedColor::RGBA, WebCore::OutOfLineColorDataForIPC>>);
    struct ShouldBeSameSizeAsColorDataForIPC : public VirtualTableAndRefCountOverhead<std::is_polymorphic_v<WebCore::ColorDataForIPC>, false> {
        bool isSemantic;
        bool usesFunctionSerialization;
        std::variant<WebCore::PackedColor::RGBA, WebCore::OutOfLineColorDataForIPC> data;
    };
    static_assert(sizeof(ShouldBeSameSizeAsColorDataForIPC) == sizeof(WebCore::ColorDataForIPC));
    static_assert(MembersInCorrectOrder < 0
        , offsetof(WebCore::ColorDataForIPC, isSemantic)
        , offsetof(WebCore::ColorDataForIPC, usesFunctionSerialization)
        , offsetof(WebCore::ColorDataForIPC, data)
    >::value);

    encoder << instance.isSemantic;
    encoder << instance.usesFunctionSerialization;
    encoder << instance.data;
}

void ArgumentCoder<WebCore::ColorDataForIPC>::encode(StreamConnectionEncoder& encoder, const WebCore::ColorDataForIPC& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.isSemantic)>, bool>);
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.usesFunctionSerialization)>, bool>);
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.data)>, std::variant<WebCore::PackedColor::RGBA, WebCore::OutOfLineColorDataForIPC>>);
    struct ShouldBeSameSizeAsColorDataForIPC : public VirtualTableAndRefCountOverhead<std::is_polymorphic_v<WebCore::ColorDataForIPC>, false> {
        bool isSemantic;
        bool usesFunctionSerialization;
        std::variant<WebCore::PackedColor::RGBA, WebCore::OutOfLineColorDataForIPC> data;
    };
    static_assert(sizeof(ShouldBeSameSizeAsColorDataForIPC) == sizeof(WebCore::ColorDataForIPC));
    static_assert(MembersInCorrectOrder < 0
        , offsetof(WebCore::ColorDataForIPC, isSemantic)
        , offsetof(WebCore::ColorDataForIPC, usesFunctionSerialization)
        , offsetof(WebCore::ColorDataForIPC, data)
    >::value);

    encoder << instance.isSemantic;
    encoder << instance.usesFunctionSerialization;
    encoder << instance.data;
}

std::optional<WebCore::ColorDataForIPC> ArgumentCoder<WebCore::ColorDataForIPC>::decode(Decoder& decoder)
{
    auto isSemantic = decoder.decode<bool>();
    auto usesFunctionSerialization = decoder.decode<bool>();
    auto data = decoder.decode<std::variant<WebCore::PackedColor::RGBA, WebCore::OutOfLineColorDataForIPC>>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WebCore::ColorDataForIPC {
            WTFMove(*isSemantic),
            WTFMove(*usesFunctionSerialization),
            WTFMove(*data)
        }
    };
}

void ArgumentCoder<WebCore::Color>::encode(Encoder& encoder, const WebCore::Color& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.data())>, std::optional<WebCore::ColorDataForIPC>>);

    encoder << instance.data();
}

void ArgumentCoder<WebCore::Color>::encode(StreamConnectionEncoder& encoder, const WebCore::Color& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.data())>, std::optional<WebCore::ColorDataForIPC>>);

    encoder << instance.data();
}

std::optional<WebCore::Color> ArgumentCoder<WebCore::Color>::decode(Decoder& decoder)
{
    auto data = decoder.decode<std::optional<WebCore::ColorDataForIPC>>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WebCore::Color {
            WTFMove(*data)
        }
    };
}

void ArgumentCoder<WebCore::NotificationPayload>::encode(Encoder& encoder, const WebCore::NotificationPayload& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.defaultActionURL)>, URL>);
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.title)>, String>);
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.appBadge)>, std::optional<unsigned long long>>);
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.options)>, std::optional<WebCore::NotificationOptionsPayload>>);
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.isMutable)>, bool>);
    struct ShouldBeSameSizeAsNotificationPayload : public VirtualTableAndRefCountOverhead<std::is_polymorphic_v<WebCore::NotificationPayload>, false> {
        URL defaultActionURL;
        String title;
        std::optional<unsigned long long> appBadge;
        std::optional<WebCore::NotificationOptionsPayload> options;
        bool isMutable;
    };
    static_assert(sizeof(ShouldBeSameSizeAsNotificationPayload) == sizeof(WebCore::NotificationPayload));
    static_assert(MembersInCorrectOrder < 0
        , offsetof(WebCore::NotificationPayload, defaultActionURL)
        , offsetof(WebCore::NotificationPayload, title)
        , offsetof(WebCore::NotificationPayload, appBadge)
        , offsetof(WebCore::NotificationPayload, options)
        , offsetof(WebCore::NotificationPayload, isMutable)
    >::value);

    encoder << instance.defaultActionURL;
    encoder << instance.title;
    encoder << instance.appBadge;
    encoder << instance.options;
    encoder << instance.isMutable;
}

std::optional<WebCore::NotificationPayload> ArgumentCoder<WebCore::NotificationPayload>::decode(Decoder& decoder)
{
    auto defaultActionURL = decoder.decode<URL>();
    auto title = decoder.decode<String>();
    auto appBadge = decoder.decode<std::optional<unsigned long long>>();
    auto options = decoder.decode<std::optional<WebCore::NotificationOptionsPayload>>();
    auto isMutable = decoder.decode<bool>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WebCore::NotificationPayload {
            WTFMove(*defaultActionURL),
            WTFMove(*title),
            WTFMove(*appBadge),
            WTFMove(*options),
            WTFMove(*isMutable)
        }
    };
}

void ArgumentCoder<WebCore::NotificationOptionsPayload>::encode(Encoder& encoder, const WebCore::NotificationOptionsPayload& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.dir)>, WebCore::NotificationDirection>);
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.lang)>, String>);
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.body)>, String>);
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.tag)>, String>);
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.icon)>, String>);
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.dataJSONString)>, String>);
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.silent)>, std::optional<bool>>);
    struct ShouldBeSameSizeAsNotificationOptionsPayload : public VirtualTableAndRefCountOverhead<std::is_polymorphic_v<WebCore::NotificationOptionsPayload>, false> {
        WebCore::NotificationDirection dir;
        String lang;
        String body;
        String tag;
        String icon;
        String dataJSONString;
        std::optional<bool> silent;
    };
    static_assert(sizeof(ShouldBeSameSizeAsNotificationOptionsPayload) == sizeof(WebCore::NotificationOptionsPayload));
    static_assert(MembersInCorrectOrder < 0
        , offsetof(WebCore::NotificationOptionsPayload, dir)
        , offsetof(WebCore::NotificationOptionsPayload, lang)
        , offsetof(WebCore::NotificationOptionsPayload, body)
        , offsetof(WebCore::NotificationOptionsPayload, tag)
        , offsetof(WebCore::NotificationOptionsPayload, icon)
        , offsetof(WebCore::NotificationOptionsPayload, dataJSONString)
        , offsetof(WebCore::NotificationOptionsPayload, silent)
    >::value);

    encoder << instance.dir;
    encoder << instance.lang;
    encoder << instance.body;
    encoder << instance.tag;
    encoder << instance.icon;
    encoder << instance.dataJSONString;
    encoder << instance.silent;
}

std::optional<WebCore::NotificationOptionsPayload> ArgumentCoder<WebCore::NotificationOptionsPayload>::decode(Decoder& decoder)
{
    auto dir = decoder.decode<WebCore::NotificationDirection>();
    auto lang = decoder.decode<String>();
    auto body = decoder.decode<String>();
    auto tag = decoder.decode<String>();
    auto icon = decoder.decode<String>();
    auto dataJSONString = decoder.decode<String>();
    auto silent = decoder.decode<std::optional<bool>>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WebCore::NotificationOptionsPayload {
            WTFMove(*dir),
            WTFMove(*lang),
            WTFMove(*body),
            WTFMove(*tag),
            WTFMove(*icon),
            WTFMove(*dataJSONString),
            WTFMove(*silent)
        }
    };
}

void ArgumentCoder<WebKit::WebPushD::WebPushDaemonConnectionConfiguration>::encode(Encoder& encoder, const WebKit::WebPushD::WebPushDaemonConnectionConfiguration& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.hostAppAuditTokenData)>, Vector<uint8_t>>);
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.bundleIdentifierOverride)>, String>);
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.pushPartitionString)>, String>);
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.dataStoreIdentifier)>, std::optional<WTF::UUID>>);
    struct ShouldBeSameSizeAsWebPushDaemonConnectionConfiguration : public VirtualTableAndRefCountOverhead<std::is_polymorphic_v<WebKit::WebPushD::WebPushDaemonConnectionConfiguration>, false> {
        Vector<uint8_t> hostAppAuditTokenData;
        String bundleIdentifierOverride;
        String pushPartitionString;
        std::optional<WTF::UUID> dataStoreIdentifier;
    };
    static_assert(sizeof(ShouldBeSameSizeAsWebPushDaemonConnectionConfiguration) == sizeof(WebKit::WebPushD::WebPushDaemonConnectionConfiguration));
    static_assert(MembersInCorrectOrder < 0
        , offsetof(WebKit::WebPushD::WebPushDaemonConnectionConfiguration, hostAppAuditTokenData)
        , offsetof(WebKit::WebPushD::WebPushDaemonConnectionConfiguration, bundleIdentifierOverride)
        , offsetof(WebKit::WebPushD::WebPushDaemonConnectionConfiguration, pushPartitionString)
        , offsetof(WebKit::WebPushD::WebPushDaemonConnectionConfiguration, dataStoreIdentifier)
    >::value);

    encoder << instance.hostAppAuditTokenData;
    encoder << instance.bundleIdentifierOverride;
    encoder << instance.pushPartitionString;
    encoder << instance.dataStoreIdentifier;
}

std::optional<WebKit::WebPushD::WebPushDaemonConnectionConfiguration> ArgumentCoder<WebKit::WebPushD::WebPushDaemonConnectionConfiguration>::decode(Decoder& decoder)
{
    auto hostAppAuditTokenData = decoder.decode<Vector<uint8_t>>();
    auto bundleIdentifierOverride = decoder.decode<String>();
    auto pushPartitionString = decoder.decode<String>();
    auto dataStoreIdentifier = decoder.decode<std::optional<WTF::UUID>>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WebKit::WebPushD::WebPushDaemonConnectionConfiguration {
            WTFMove(*hostAppAuditTokenData),
            WTFMove(*bundleIdentifierOverride),
            WTFMove(*pushPartitionString),
            WTFMove(*dataStoreIdentifier)
        }
    };
}

void ArgumentCoder<WebKit::WebPushMessage>::encode(Encoder& encoder, const WebKit::WebPushMessage& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.pushData)>, std::optional<Vector<uint8_t>>>);
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.pushPartitionString)>, String>);
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.registrationURL)>, URL>);
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.notificationPayload)>, std::optional<WebCore::NotificationPayload>>);
    struct ShouldBeSameSizeAsWebPushMessage : public VirtualTableAndRefCountOverhead<std::is_polymorphic_v<WebKit::WebPushMessage>, false> {
        std::optional<Vector<uint8_t>> pushData;
        String pushPartitionString;
        URL registrationURL;
        std::optional<WebCore::NotificationPayload> notificationPayload;
    };
    static_assert(sizeof(ShouldBeSameSizeAsWebPushMessage) == sizeof(WebKit::WebPushMessage));
    static_assert(MembersInCorrectOrder < 0
        , offsetof(WebKit::WebPushMessage, pushData)
        , offsetof(WebKit::WebPushMessage, pushPartitionString)
        , offsetof(WebKit::WebPushMessage, registrationURL)
        , offsetof(WebKit::WebPushMessage, notificationPayload)
    >::value);

    encoder << instance.pushData;
    encoder << instance.pushPartitionString;
    encoder << instance.registrationURL;
    encoder << instance.notificationPayload;
}

std::optional<WebKit::WebPushMessage> ArgumentCoder<WebKit::WebPushMessage>::decode(Decoder& decoder)
{
    auto pushData = decoder.decode<std::optional<Vector<uint8_t>>>();
    auto pushPartitionString = decoder.decode<String>();
    auto registrationURL = decoder.decode<URL>();
    auto notificationPayload = decoder.decode<std::optional<WebCore::NotificationPayload>>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WebKit::WebPushMessage {
            WTFMove(*pushData),
            WTFMove(*pushPartitionString),
            WTFMove(*registrationURL),
            WTFMove(*notificationPayload)
        }
    };
}

#if USE(SKIA)
void ArgumentCoder<WebKit::CoreIPCSkColorSpace>::encode(Encoder& encoder, const WebKit::CoreIPCSkColorSpace& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.dataReference())>, std::span<const uint8_t>>);

    encoder << instance.dataReference();
}

void ArgumentCoder<WebKit::CoreIPCSkColorSpace>::encode(StreamConnectionEncoder& encoder, const WebKit::CoreIPCSkColorSpace& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.dataReference())>, std::span<const uint8_t>>);

    encoder << instance.dataReference();
}

std::optional<WebKit::CoreIPCSkColorSpace> ArgumentCoder<WebKit::CoreIPCSkColorSpace>::decode(Decoder& decoder)
{
    auto dataReference = decoder.decode<std::span<const uint8_t>>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WebKit::CoreIPCSkColorSpace {
            WTFMove(*dataReference)
        }
    };
}

#endif

void ArgumentCoder<WebKit::CoreIPCSkData>::encode(Encoder& encoder, const WebKit::CoreIPCSkData& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.dataReference())>, std::span<const uint8_t>>);

    encoder << instance.dataReference();
}

void ArgumentCoder<WebKit::CoreIPCSkData>::encode(StreamConnectionEncoder& encoder, const WebKit::CoreIPCSkData& instance)
{
    static_assert(std::is_same_v<std::remove_cvref_t<decltype(instance.dataReference())>, std::span<const uint8_t>>);

    encoder << instance.dataReference();
}

std::optional<WebKit::CoreIPCSkData> ArgumentCoder<WebKit::CoreIPCSkData>::decode(Decoder& decoder)
{
    auto dataReference = decoder.decode<std::span<const uint8_t>>();
    if (UNLIKELY(!decoder.isValid()))
        return std::nullopt;
    return {
        WebKit::CoreIPCSkData {
            WTFMove(*dataReference)
        }
    };
}

} // namespace IPC

namespace WTF {

template<> bool isValidOptionSet<IPC::MessageFlags>(OptionSet<IPC::MessageFlags> value)
{
    constexpr uint8_t allValidBitsValue = 0
        | static_cast<uint8_t>(IPC::MessageFlags::DispatchMessageWhenWaitingForSyncReply)
        | static_cast<uint8_t>(IPC::MessageFlags::DispatchMessageWhenWaitingForUnboundedSyncReply)
        | static_cast<uint8_t>(IPC::MessageFlags::UseFullySynchronousModeForTesting)
        | static_cast<uint8_t>(IPC::MessageFlags::MaintainOrderingWithAsyncMessages)
#if ENABLE(IPC_TESTING_API)
        | static_cast<uint8_t>(IPC::MessageFlags::SyncMessageDeserializationFailure)
#endif
        | 0;
    return (value.toRaw() | allValidBitsValue) == allValidBitsValue;
}

template<> bool isValidEnum<WebCore::ColorSpace, void>(uint8_t value)
{
    switch (static_cast<WebCore::ColorSpace>(value)) {
    case WebCore::ColorSpace::A98RGB:
    case WebCore::ColorSpace::DisplayP3:
    case WebCore::ColorSpace::ExtendedA98RGB:
    case WebCore::ColorSpace::ExtendedDisplayP3:
    case WebCore::ColorSpace::ExtendedLinearSRGB:
    case WebCore::ColorSpace::ExtendedProPhotoRGB:
    case WebCore::ColorSpace::ExtendedRec2020:
    case WebCore::ColorSpace::ExtendedSRGB:
    case WebCore::ColorSpace::HSL:
    case WebCore::ColorSpace::HWB:
    case WebCore::ColorSpace::LCH:
    case WebCore::ColorSpace::Lab:
    case WebCore::ColorSpace::LinearSRGB:
    case WebCore::ColorSpace::OKLCH:
    case WebCore::ColorSpace::OKLab:
    case WebCore::ColorSpace::ProPhotoRGB:
    case WebCore::ColorSpace::Rec2020:
    case WebCore::ColorSpace::SRGB:
    case WebCore::ColorSpace::XYZ_D50:
    case WebCore::ColorSpace::XYZ_D65:
        return true;
    default:
        return false;
    }
}

template<> bool isValidEnum<WebCore::NotificationDirection, void>(uint8_t value)
{
    switch (static_cast<WebCore::NotificationDirection>(value)) {
    case WebCore::NotificationDirection::Auto:
    case WebCore::NotificationDirection::Ltr:
    case WebCore::NotificationDirection::Rtl:
        return true;
    default:
        return false;
    }
}

} // namespace WTF

IGNORE_WARNINGS_END
