"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Thread = void 0;

var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));

var _events = require("events");

var _event = require("./event");

var _eventTimelineSet = require("./event-timeline-set");

/*
Copyright 2021 The Matrix.org Foundation C.I.C.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

/**
 * @experimental
 */
class Thread extends _events.EventEmitter {
  /**
   * A reference to the event ID at the top of the thread
   */

  /**
   * A reference to all the events ID at the bottom of the threads
   */
  constructor(events = [], room, client) {
    super();
    this.room = room;
    this.client = client;
    (0, _defineProperty2.default)(this, "root", void 0);
    (0, _defineProperty2.default)(this, "tail", new Set());
    (0, _defineProperty2.default)(this, "_timelineSet", void 0);
    this._timelineSet = new _eventTimelineSet.EventTimelineSet(room, {
      unstableClientRelationAggregation: true,
      timelineSupport: true
    });
    events.forEach(event => this.addEvent(event));
  }
  /**
   * Add an event to the thread and updates
   * the tail/root references if needed
   * Will fire "Thread.update"
   * @param event The event to add
   */


  async addEvent(event) {
    if (this._timelineSet.findEventById(event.getId()) || event.status !== null) {
      return;
    }

    if (this.tail.has(event.replyEventId)) {
      this.tail.delete(event.replyEventId);
    }

    this.tail.add(event.getId());

    if (!event.replyEventId || !this._timelineSet.findEventById(event.replyEventId)) {
      this.root = event.getId();
    }

    event.setThread(this);

    this._timelineSet.addLiveEvent(event);

    if (this.ready) {
      this.client.decryptEventIfNeeded(event, {});
      this.emit("Thread.update", this);
    } else {
      this.emit("Thread.update", this);
    }
  }
  /**
   * Completes the reply chain with all events
   * missing from the current sync data
   * Will fire "Thread.ready"
   */


  async fetchReplyChain() {
    if (!this.ready) {
      let mxEvent = this.room.findEventById(this.rootEvent.replyEventId);

      if (!mxEvent) {
        mxEvent = await this.fetchEventById(this.rootEvent.getRoomId(), this.rootEvent.replyEventId);
      }

      this.addEvent(mxEvent);

      if (mxEvent.replyEventId) {
        await this.fetchReplyChain();
      } else {
        await this.decryptEvents();
        this.emit("Thread.ready", this);
      }
    }
  }

  async decryptEvents() {
    await Promise.allSettled(Array.from(this._timelineSet.getLiveTimeline().getEvents()).map(event => {
      return this.client.decryptEventIfNeeded(event, {});
    }));
  }
  /**
   * Fetches an event over the network
   */


  async fetchEventById(roomId, eventId) {
    const response = await this.client.http.authedRequest(undefined, "GET", `/rooms/${roomId}/event/${eventId}`);
    return new _event.MatrixEvent(response);
  }
  /**
   * Finds an event by ID in the current thread
   */


  findEventById(eventId) {
    return this._timelineSet.findEventById(eventId);
  }
  /**
   * Determines thread's ready status
   */


  get ready() {
    return this.rootEvent.replyEventId === undefined;
  }
  /**
   * The thread ID, which is the same as the root event ID
   */


  get id() {
    return this.root;
  }
  /**
   * The thread root event
   */


  get rootEvent() {
    return this.findEventById(this.root);
  }
  /**
   * The number of messages in the thread
   */


  get length() {
    return this.events.length;
  }
  /**
   * A set of mxid participating to the thread
   */


  get participants() {
    const participants = new Set();
    this.events.forEach(event => {
      participants.add(event.getSender());
    });
    return participants;
  }
  /**
   * A read-only getter to access the timeline set
   */


  get timelineSet() {
    return this._timelineSet;
  }
  /**
   * A getter for the last event added to the thread
   */


  get replyToEvent() {
    const events = this.events;
    return events[events.length - 1];
  }

  get events() {
    return this._timelineSet.getLiveTimeline().getEvents();
  }

  merge(thread) {
    thread.events.forEach(event => {
      this.addEvent(event);
    });
    this.events.forEach(event => event.setThread(this));
  }

}

exports.Thread = Thread;