//------------------------------ $Keywords ----------------------------------
// GPGee - GNU Privacy Guard Explorer Extension
// GPGeeSignEncrypt.cpp - 'Main' form code
// Copyright 2005, Kurt Fitzner <kfitzner@excelcia.org>
//---------------------------------------------------------------------------
// This file is part of GPGee.
//
// GPGee is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License (Version 2) as
// published by the Free Software Foundation.
//
// GPGee is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// VCS: $Version: 1 $ $Revision: 16 $
/*
$History: **** V 1.0 by kfitzner ****
$History: * gpgeesignencrypt.ddp - 2005-05-13 5:47:58 PM - 2136 Bytes
$History: * gpgeesignencrypt.cpp - 2005-05-13 11:14:16 PM - 41433 Bytes
$History: * gpgeesignencrypt.dfm - 2005-05-07 4:19:36 PM - 23037 Bytes
$History: * gpgeesignencrypt.h - 2005-05-07 7:07:38 AM - 4746 Bytes
$History: * Initial check-in
$History: **** V 1.1 by kfitzner ****
$History: * gpgeesignencrypt.cpp - 2005-05-17 2:16:33 AM - 40960 Bytes
$History: * gpgeesignencrypt.dfm - 2005-05-07 4:19:36 PM - 23037 Bytes
$History: * gpgeesignencrypt.h - 2005-05-16 9:28:22 PM - 4705 Bytes
$History: * gpgeesignencrypt.ddp - 2005-05-17 1:19:02 AM - 33 Bytes
$History: * Add persistent key cache
$History: **** V 1.2 by kfitzner ****
$History: * gpgeesignencrypt.cpp - 2005-07-17 1:28:16 PM - 41649 Bytes
$History: * gpgeesignencrypt.dfm - 2005-05-07 4:19:36 PM - 23037 Bytes
$History: * gpgeesignencrypt.h - 2005-05-16 9:28:22 PM - 4705 Bytes
$History: * gpgeesignencrypt.ddp - 2005-07-16 3:02:48 AM - 33 Bytes
$History: * Add option to always show parent key id
$History: **** V 1.3 by kfitzner ****
$History: * gpgeesignencrypt.cpp - 2005-08-08 6:50:13 AM - 41976 Bytes
$History: * gpgeesignencrypt.dfm - 2005-05-07 4:19:36 PM - 23037 Bytes
$History: * gpgeesignencrypt.h - 2005-08-08 6:35:58 AM - 4641 Bytes
$History: * gpgeesignencrypt.ddp - 2005-08-08 6:35:58 AM - 33 Bytes
$History: * License change - remove option for later versions of GPL
$History: **** V 1.4 by kfitzner ****
$History: * gpgeesignencrypt.cpp - 2005-09-05 6:35:55 AM - 47792 Bytes
$History: * gpgeesignencrypt.dfm - 2005-09-05 6:15:28 AM - 23354 Bytes
$History: * gpgeesignencrypt.h - 2005-09-05 6:15:28 AM - 5056 Bytes
$History: * gpgeesignencrypt.ddp - 2005-09-05 6:19:02 AM - 33 Bytes
$History: * Add multiple-key signing and option to make source files read-only
$History: **** V 1.5 by kfitzner ****
$History: * gpgeesignencrypt.cpp - 2005-09-05 4:33:57 PM - 49661 Bytes
$History: * gpgeesignencrypt.dfm - 2005-09-05 6:15:28 AM - 23354 Bytes
$History: * gpgeesignencrypt.h - 2005-09-05 4:19:16 PM - 5127 Bytes
$History: * gpgeesignencrypt.ddp - 2005-09-05 4:32:36 PM - 33 Bytes
$History: * Fix bug that prevents successful sign/encrypr operation 
$History: * if the cancel button is pressed on the password dialog. 
$History: *  Turns out to be a bug in MyGPGME - workaround implemented.
$History: **** V 1.6 by kfitzner ****
$History: * gpgeesignencrypt.cpp - 2005-09-06 7:06:22 PM - 53758 Bytes
$History: * gpgeesignencrypt.dfm - 2005-09-06 6:00:08 PM - 23656 Bytes
$History: * gpgeesignencrypt.h - 2005-09-06 4:21:26 PM - 5196 Bytes
$History: * gpgeesignencrypt.ddp - 2005-09-06 7:06:22 PM - 1799 Bytes
$History: * Fix hang when sign+encrypt, code cleanup, and add 
$History: * signature appending feature
$History: **** V 1.7 by kfitzner ****
$History: * gpgeesignencrypt.cpp - 2005-10-06 8:28:34 PM - 56523 Bytes
$History: * gpgeesignencrypt.dfm - 2005-09-06 6:00:08 PM - 23656 Bytes
$History: * gpgeesignencrypt.h - 2005-09-06 4:21:26 PM - 5196 Bytes
$History: * gpgeesignencrypt.ddp - 2005-10-06 8:20:36 PM - 1799 Bytes
$History: * Add language support - now all forms' visual elements 
$History: * are stored in the string table (strings.rc)
$History: **** V 1.8 by kfitzner ****
$History: * gpgeesignencrypt.cpp - 2005-11-21 8:53:00 AM - 56985 Bytes
$History: * gpgeesignencrypt.dfm - 2005-09-06 6:00:08 PM - 23656 Bytes
$History: * gpgeesignencrypt.h - 2005-10-07 5:51:54 AM - 5196 Bytes
$History: * gpgeesignencrypt.ddp - 2005-10-07 5:56:20 AM - 1799 Bytes
$History: * Fix translation string mixup - encryption strings 
$History: * were put in the signature options
$History: **** V 1.9 by kfitzner ****
$History: * gpgeesignencrypt.cpp - 2005-12-11 3:30:34 PM - 58045 Bytes
$History: * gpgeesignencrypt.dfm - 2005-12-11 1:54:24 PM - 23796 Bytes
$History: * gpgeesignencrypt.h - 2005-12-11 2:58:38 PM - 5477 Bytes
$History: * gpgeesignencrypt.ddp - 2005-12-11 2:58:56 PM - 1799 Bytes
$History: * Enable symmetrical encryption as a startup default, 
$History: * fix passphrase caching bug (nOpNumber)
$History: **** V 1.10 by kfitzner ****
$History: * gpgeesignencrypt.cpp - 2005-12-11 4:06:37 PM - 58406 Bytes
$History: * gpgeesignencrypt.dfm - 2005-12-11 4:00:06 PM - 23818 Bytes
$History: * gpgeesignencrypt.h - 2005-12-11 2:58:38 PM - 5477 Bytes
$History: * gpgeesignencrypt.ddp - 2005-12-11 4:02:52 PM - 1799 Bytes
$History: * Make Ok button default
$History: **** V 1.11 by kfitzner ****
$History: * gpgeesignencrypt.cpp - 2005-12-13 3:08:45 PM - 59037 Bytes
$History: * gpgeesignencrypt.dfm - 2005-12-11 4:00:06 PM - 23818 Bytes
$History: * gpgeesignencrypt.h - 2005-12-11 2:58:38 PM - 5477 Bytes
$History: * gpgeesignencrypt.ddp - 2005-12-13 2:10:02 PM - 1799 Bytes
$History: * Unset MyGPGME's CAST5 symmetrical encryption default.
$History: **** V 1.12 by kfitzner ****
$History: * gpgeesignencrypt.cpp - 2005-12-14 1:28:43 PM - 59585 Bytes
$History: * gpgeesignencrypt.dfm - 2005-12-14 1:09:38 PM - 24402 Bytes
$History: * gpgeesignencrypt.h - 2005-12-14 1:12:50 PM - 5521 Bytes
$History: * gpgeesignencrypt.ddp - 2005-12-14 1:28:00 PM - 1799 Bytes
$History: * Add textmode support
$History: **** V 1.13 by kfitzner ****
$History: * gpgeesignencrypt.cpp - 2005-12-16 5:07:11 PM - 62012 Bytes
$History: * gpgeesignencrypt.dfm - 2005-12-14 1:09:38 PM - 24402 Bytes
$History: * gpgeesignencrypt.h - 2005-12-16 1:40:48 PM - 5558 Bytes
$History: * gpgeesignencrypt.ddp - 2005-12-16 5:05:32 PM - 1799 Bytes
$History: * Process directories during the operation phase rather 
$History: * than before the context menu appears.  Fixed problems 
$History: * with passphrase caching.
$History: **** V 1.14 by kfitzner ****
$History: * gpgeesignencrypt.cpp - 2005-12-31 4:17:58 AM - 63567 Bytes
$History: * gpgeesignencrypt.dfm - 2005-12-14 1:09:38 PM - 24402 Bytes
$History: * gpgeesignencrypt.h - 2005-12-16 1:40:48 PM - 5558 Bytes
$History: * gpgeesignencrypt.ddp - 2005-12-31 4:17:58 AM - 1799 Bytes
$History: * Added smartcard support
$History: **** V 1.15 by kfitzner ****
$History: * gpgeesignencrypt.cpp - 2006-01-01 8:31:33 AM - 64293 Bytes
$History: * gpgeesignencrypt.dfm - 2005-12-31 3:46:40 PM - 24558 Bytes
$History: * gpgeesignencrypt.h - 2005-12-16 1:40:48 PM - 5558 Bytes
$History: * gpgeesignencrypt.ddp - 2006-01-01 8:30:12 AM - 1799 Bytes
$History: * Make sizeable, don't create passphrase form unless needed
$History: **** Latest ** V 1.16 by kfitzner ** 2006-01-02 3:11:19 PM ****
$History: * Call ProcessMessages() for each file processed, stop 
$History: * processing when cancel pressed, and delete FileNames when form deleted
*/
//----------------------------  $NoKeywords ---------------------------------


//---------------------------------------------------------------------------
// File Notes:
//---------------------------------------------------------------------------
// 21 Mar 2005 - Kurt Fitzner <kfitzner@excelcia.org>
//
// This is the code behind the sign/encrypt form.  It's often not
// encouraged to put user interface code in the same place as functional
// code.  I've never subscribed to this seperation, at least not for smaller
// applications.  I find it decreases code readability when a user interface
// function callback for a button click event does nothing but turn around
// and call something else.  You will find most of the functional code
// right in the event handler callback functions here.
//
// 5 Sept 2005 - Kurt Fitzner <kfitzner@excelcia.org>
//
// Converted this form to support adding multiple signatures to a file in
// a single operation.  You can now select multiple keys from the secret
// key list.
//---------------------------------------------------------------------------
#include <vcl.h>
#include <Registry.hpp>
#pragma hdrstop

#include "TConfiguration.h"
#include "GPGeeExceptions.h"
#include "GPGeePassPhrase.h"
#include "GPGeeUtility.h"
#include "TProgramLog.h"
#include "GPGeeSignEncrypt.h"
//---------------------------------------------------------------------------
#pragma package(smart_init)
#pragma resource "*.dfm"
#pragma link "CheckCombo"
TformGPGeeSignEncrypt *formGPGeeSignEncrypt = NULL;
extern TConfiguration *GPGeeConfig;
extern TStringList *PassphraseCache;
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// Constructor for TformGPGeeSignEncrypt - our sign/encryption form
//
__fastcall TformGPGeeSignEncrypt::TformGPGeeSignEncrypt(TComponent* Owner) : TForm(Owner)
{
  gpgme_error_t err;
  char *buffer;
  AnsiString sConfigString;

  __ENTERFUNCTION__;

  ctxGPG = NULL;
  cachePublic = NULL;
  cacheSecret = NULL;
  FileNames = NULL;

  try {
    // Initialize the GPGME library main context
    ResetGPGMEContext(&ctxGPG);

    // Set GPGME's global settings to match GPGee's configuration
    LOG(LOG_DEBUG, "Setting the GPGME global configuration.");
    sConfigString = GPGeeConfig->Values["GPG Program"];
    if (!sConfigString.IsEmpty())
      gpgme_global_control(GPGME_GLOBAL_GPGPROGRAM, sConfigString.c_str());
    sConfigString = GPGeeConfig->Values["Options File"];
    if (!sConfigString.IsEmpty())
      gpgme_global_control(GPGME_GLOBAL_OPTFILE, sConfigString.c_str());
    sConfigString = GPGeeConfig->Values["Keyring"];
    if (!sConfigString.IsEmpty())
      gpgme_global_control(GPGME_GLOBAL_PUBRING, sConfigString.c_str());
    sConfigString = GPGeeConfig->Values["Secret Keyring"];
    if (!sConfigString.IsEmpty())
      gpgme_global_control(GPGME_GLOBAL_SECRING, sConfigString.c_str());

    pubkeysLastSorted = -1;
    pubkeysAscending = true;
    nEncryptCount = 0;
    __RETURNFUNCTION__;
  }  // try
  catch (Exception &e) {
    LOG(LOG_ERROR, "Exception " + e.ClassName() + " raised with message " + e.Message);
    Cleanup();
    __LEAVEFUNCTION__;
    throw;
  }  // catch
}  // __fastcall TformGPGeeSignEncrypt::TformGPGeeSignEncrypt(TComponent* Owner) : TForm(Owner)
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// Reset (or set if it hasn't been previously) the GPGME context.
//
void __fastcall TformGPGeeSignEncrypt::ResetGPGMEContext(gpgme_ctx_t *pCtx)
{
  gpgme_error_t err;

  __ENTERFUNCTION__;

  // Release the current context, if it exists
  if (*pCtx) {
    LOG(LOG_DEBUG, "Releasing old GPGME main context.");
    gpgme_release(*pCtx);
  }  // if (*pCtx)
  *pCtx = NULL;

  // Create a new one
  LOG(LOG_DEBUG, "Allocating the GPGME main context.");
  err = gpgme_new(pCtx);
  if (err != GPGME_No_Error)
     throw EGPGMEError(FormMessage(MSG_ERROR_GPGME_LIB_INIT, gpgme_strerror(err)));

  // Set GPGME's settings to match GPGee's configuration
  LOG(LOG_DEBUG, "Setting the GPGME context configuration.");
  gpgme_control(*pCtx, GPGME_CTRL_FORCETRUST, GPGeeConfig->Values["Force Trust"]?1:0);

  // GPGME has a default symmetrical encryption (CAST5) - kill this so that GnuPG is free to pick its own
  LOG(LOG_DEBUG, "Clearing the GPGME default symmetrical encryption algorithm.");
  gpgme_control(*pCtx, GPGME_CTRL_CIPHER, -1);

  // Set the passphrase callback function.  This can't be a method because of C++ limitations (bah), so we supply a
  // pointer to this form to the callback so that a class method can be invoked from the callback.
  LOG(LOG_DEBUG, "Setting the passphrase callback function.");
  gpgme_set_passphrase_cb(*pCtx, PassphraseCallback, this);

  __RETURNFUNCTION__;
}  // void __fastcall ResetGPGMEContext(gpgme_ctx_t *pCtx)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Destructor
//
__fastcall TformGPGeeSignEncrypt::~TformGPGeeSignEncrypt()
{
  __ENTERFUNCTION__;
  Cleanup();
  __RETURNFUNCTION__;
}  // __fastcall TformGPGeeSignEncrypt::~TformGPGeeSignEncrypt()
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Cleanup in preparation for the form to be deleted
//
void __fastcall TformGPGeeSignEncrypt::Cleanup(void)
{
  __ENTERFUNCTION__;

  if (ctxGPG) {
    gpgme_release(ctxGPG);
    ctxGPG = NULL;
  }  // if (ctxGPG)
  if (FileNames)
    delete FileNames;

  __RETURNFUNCTION__;
}  // void __fastcall TformGPGeeSignEncrypt::Cleanup(void)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Set the default modes and show the form - this just stores the default
// modes.  The real work of actually changing the controls on the form to
// match these defaults is done in FormShow()
//
int __fastcall TformGPGeeSignEncrypt::ShowModalWithDefaults(bool bSign, bool bPKEncrypt, bool bSEncrypt, TStringList *FileNames)
{
  int retval;

  __ENTERFUNCTION__;
  bStartupSign = bSign;
  bStartupPKEncrypt = bPKEncrypt;
  bStartupSEncrypt = bSEncrypt;
  this->FileNames = FileNames;
  LOG(LOG_DEBUG, "Showing the sign/encryption form.");
  retval = ShowModal();
  __RETURNFUNCTION(retval);
}  // int __fastcall TformGPGeeSignEncrypt::ShowModalWithDefaults(bool bSign, bool bEncrypt)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// OnShow event handler - The OnShow event occurs before the form becomes
// visible, but after all its components have been streamed in.  This is a
// good place to put code that initializes the form's look so that the form
// looks like it should right when it first pops up.
//
void __fastcall TformGPGeeSignEncrypt::FormShow(TObject *Sender)
{
  gpgme_error_t err;
  gpgme_key_t key;
  TListItem *keyvisual;
  const char *caps;
  int nSubkey;
  bool bValidkey;
  AnsiString sValidity, sKeyId, sName, sComment, sAlgo;
  unsigned nKeyId;
  AnsiString sLastKey, sThisKey, sDefaultSigningKeys;
  THashedStringList *DuplicateCheck;

  __ENTERFUNCTION__;

  // Read in the language strings for the form's visual text labels and change them on the form.  Borland's language
  // editor is capable of doing this, but it's much easier for translators if we can give them a single resource
  // string table to translate instead of forcing them to use Borland's translation tool.
  this->Caption                = GetMessage(FORM_SIGN_CAPTION);
  btnOk->Caption               = GetMessage(FORM_BTN_OK);
  btnCancel->Caption           = GetMessage(FORM_BTN_CANCEL);
  btnHelp->Caption             = GetMessage(FORM_BTN_HELP);
  lblKeyGroups->Caption        = GetMessage(FORM_SIGN_LABEL_KEYGROUPS);
  lblSigningKey->Caption       = GetMessage(FORM_SIGN_LABEL_SIGNINGKEYS);
  rgEncryptionOptions->Caption = GetMessage(FORM_SIGN_LABEL_ENCRYPTOPTIONS);
  gbSignatureOptions->Caption  = GetMessage(FORM_SIGN_LABEL_SIGOPTIONS);
  gbMiscOptions->Caption       = GetMessage(FORM_SIGN_LABEL_MISCOPTIONS);
  radioNoSignature->Caption    = GetMessage(FORM_SIGN_RADIO_NONE);
  radioAttached->Caption       = GetMessage(FORM_SIGN_RADIO_ATTACHED);
  radioDetached->Caption       = GetMessage(FORM_SIGN_RADIO_DETACHED);
  chbClearsign->Caption        = GetMessage(FORM_SIGN_CHECK_CLEARSIGN);
  chbASCIIArmor->Caption       = GetMessage(FORM_SIGN_CHECK_ARMOUR);
  chbTextMode->Caption         = GetMessage(FORM_SIGN_CHECK_TEXTMODE);
  chbReadOnly->Caption         = GetMessage(FORM_SIGN_CHECK_READONLY);
  chbSigAppend->Caption        = GetMessage(FORM_SIGN_CHECK_APPEND);
  btnGroupAdd->Hint           = GetMessage(FORM_SIGN_HINT_GROUPADD);
  btnGroupModify->Hint        = GetMessage(FORM_SIGN_HINT_GROUPMODIFY);
  btnGroupDelete->Hint        = GetMessage(FORM_SIGN_HINT_GROUPDELETE);
  rgEncryptionOptions->Items->Strings[0]   = GetMessage(FORM_SIGN_RADIO_NONE);
  rgEncryptionOptions->Items->Strings[1]   = GetMessage(FORM_SIGN_RADIO_PUBLIC);
  rgEncryptionOptions->Items->Strings[2]   = GetMessage(FORM_SIGN_RADIO_SYMMETRIC);
  lvPublicKeys->Columns->Items[0]->Caption = GetMessage(FORM_SIGN_COLUMN_NAME);
  lvPublicKeys->Columns->Items[1]->Caption = GetMessage(FORM_SIGN_COLUMN_KEYID);
  lvPublicKeys->Columns->Items[2]->Caption = GetMessage(FORM_SIGN_COLUMN_SIZE);
  lvPublicKeys->Columns->Items[3]->Caption = GetMessage(FORM_SIGN_COLUMN_TYPE);
  lvPublicKeys->Columns->Items[4]->Caption = GetMessage(FORM_SIGN_COLUMN_VALIDITY);

  DuplicateCheck = new THashedStringList();

  // Create one KEYCACHE each for the public and private keys
  LOG(LOG_DEBUG, "Initializing the public key cache.");
  cachePublic = GetKeyCache(false);
  LOG(LOG_DEBUG, "Initializing the secret key cache.");
  cacheSecret = GetKeyCache(true);

  // Fill in the info on keys that can be encrypted to
  lvPublicKeys->Items->Clear();
  lvPublicKeys->Items->BeginUpdate();
  sLastKey = "";
  gpgme_keycache_rewind(cachePublic);
  while (gpgme_keycache_next_key(cachePublic, false, &key) == 0) {
    // FIXME: This is a workaround for what currently appears to be a bug in the MyGPGME library.  This should
    // be fixed in the library rather than worked around here.
    sThisKey = String(gpgme_key_get_string_attr(key, GPGME_ATTR_KEYID, NULL, 0));
    if (DuplicateCheck->IndexOf(sThisKey) != -1) {
      LOG1(LOG_WARNING, "Skipping key id %s - appears to be duplicated in the public key cache.", sThisKey.c_str());
      continue;
    }  // if (sThisKey == sLastKey)
    DuplicateCheck->Add(sThisKey);
    // FIXME: End bad hack.
    // If the key can be encrypted to, add its info to the listview component
    nSubkey = 0;
    bValidkey = false;
    // Find the first subkey that can be encrypted to
    while ((caps = gpgme_key_get_string_attr(key, GPGME_ATTR_KEY_CAPS, NULL, nSubkey)) != NULL) {
      // If the key can be encrypted to and isn't revoked or expired, we'll call it valid and add it
      if (strchr(caps, 'e') && gpgme_key_get_ulong_attr(key, GPGME_ATTR_KEY_USABLE, NULL, nSubkey)) {
        bValidkey = true;
        break;
      }  // if (strchr(caps, 'e'))
      nSubkey++;
    }  // while ((caps = gpgme_key_get_string_attr(key, GPGME_ATTR_KEY_CAPS, NULL, nSubkey)) != NULL)
    if (!bValidkey)
      continue;
    // Add a new item to the listview component and then add all the information to it for each column.
    sKeyId = "0x" + String(gpgme_key_get_string_attr(key, GPGME_ATTR_KEYID, NULL, nSubkey)).Delete(1,8);
    nKeyId = StrToInt(sKeyId);
    LOG1(LOG_DEBUG, "Adding public key id %s to the encryption key list",sKeyId.c_str());
    if (GPGeeConfig->Values["Parent IDs"])
      sKeyId = "0x" + String(gpgme_key_get_string_attr(key, GPGME_ATTR_KEYID, NULL, 0)).Delete(1,8);
    keyvisual = lvPublicKeys->Items->Add();
    keyvisual->Data = (void *)nKeyId;
    keyvisual->Caption = String(gpgme_key_get_string_attr(key, GPGME_ATTR_NAME, NULL, 0)) + " <" + String(gpgme_key_get_string_attr(key, GPGME_ATTR_EMAIL, NULL, 0)) + ">";
    keyvisual->SubItems->Add(sKeyId);
    keyvisual->SubItems->Add(String(gpgme_key_get_ulong_attr(key, GPGME_ATTR_LEN, NULL, nSubkey)));
    keyvisual->SubItems->Add(String(gpgme_key_get_string_attr(key, GPGME_ATTR_ALGO, NULL, nSubkey)));
    switch (gpgme_key_get_ulong_attr(key, GPGME_ATTR_VALIDITY, NULL, 0)) {
      case GPGME_VALIDITY_UNKNOWN:   sValidity = GetMessage(MSG_GPGME_VALIDITY_UNKNOWN);   break;
      case GPGME_VALIDITY_UNDEFINED: sValidity = GetMessage(MSG_GPGME_VALIDITY_UNDEFINED); break;
      case GPGME_VALIDITY_NEVER:     sValidity = GetMessage(MSG_GPGME_VALIDITY_NEVER);     break;
      case GPGME_VALIDITY_MARGINAL:  sValidity = GetMessage(MSG_GPGME_VALIDITY_MARGINAL);  break;
      case GPGME_VALIDITY_FULL:      sValidity = GetMessage(MSG_GPGME_VALIDITY_FULL);      break;
      case GPGME_VALIDITY_ULTIMATE:  sValidity = GetMessage(MSG_GPGME_VALIDITY_ULTIMATE);  break;
      default:                       sValidity = GetMessage(MSG_GPGME_VALIDITY_ERROR);     break;
    }  // switch (gpgme_key_get_ulong_attr(key, GPGME_ATTR_VALIDITY, NULL, 0))
    keyvisual->SubItems->Add(sValidity);
  }  // while (gpgme_op_keylist_next(ctxGPG, &key) == 0)
  lvPublicKeys->Items->EndUpdate();

  ccbSigningKey->Clear();
  gpgme_keycache_rewind(cacheSecret);
  DuplicateCheck->Clear();
  // Add the secret keys to the signing key selection drop-down combobox.
  while (gpgme_keycache_next_key(cacheSecret, false, &key) == 0) {
    // FIXME: This is a workaround for what currently appears to be a bug in the MyGPGME library.  This should
    // be fixed in the library rather than worked around here.
    sThisKey = gpgme_key_get_string_attr(key, GPGME_ATTR_KEYID, NULL, 0);
    if (DuplicateCheck->IndexOf(sThisKey) != -1) {
      LOG1(LOG_WARNING, "Skipping key id %s - appears to be duplicated in the secret key cache.", sThisKey.c_str());
      continue;
    }  // if (sThisKey == sLastKey)
    DuplicateCheck->Add(sThisKey);
    // FIXME: End bad hack.
    sComment = String(gpgme_key_get_string_attr(key, GPGME_ATTR_COMMENT, NULL, 0));
    sName    = String(gpgme_key_get_string_attr(key, GPGME_ATTR_NAME, NULL, 0)) + (sComment.IsEmpty()?String(""):(" (" + sComment + ")")) + " <" + String(gpgme_key_get_string_attr(key, GPGME_ATTR_EMAIL, NULL, 0)) + ">";
    sAlgo    = String(gpgme_key_get_string_attr(key, GPGME_ATTR_ALGO, NULL, 0));
    sKeyId = "0x" + String(gpgme_key_get_string_attr(key, GPGME_ATTR_KEYID, NULL, 0)).Delete(1,8);
    nKeyId = StrToInt(sKeyId);
    LOG1(LOG_DEBUG, "Adding secret key id 0x%s to the signing key list",sKeyId.c_str());
    ccbSigningKey->Items->AddObject(sName + " (" + sAlgo + "/" + sKeyId + ")", (TObject *)nKeyId);
  }  // while (gpgme_op_keylist_next(ctxGPG, &key) == 0)
  // Make the combobox's dropdown size equal to the number of keys
  if (ccbSigningKey->Items->Count < 8)
    ccbSigningKey->DropDownLines = ccbSigningKey->Items->Count;
  // Find the default signing keys (if any) and select them
  sDefaultSigningKeys = GPGeeConfig->Values["Signing Keys"];
  for (int n = 0; n < ccbSigningKey->Items->Count; n++) {
    AnsiString sSigningKey = "0x" + IntToHex((int)ccbSigningKey->Items->Objects[n],8);
    if (sDefaultSigningKeys.Pos(sSigningKey))
      ccbSigningKey->Checked[n] = true;
  }  // for (int n = 0; n < ccbSigningKey->Items->Count; n++)

  // Load the key groups
  cbKeyGroups->Items->Text = GPGeeConfig->Values["Key Groups"];

  // Set the form up with the defaults we were given for signing/encryption in ShowModalWithDefaults()
  if (bStartupSign && bStartupPKEncrypt) {
    rgEncryptionOptions->ItemIndex = GPGEE_ENCRYPTION_PUBLIC;
    radioAttached->Checked = true;
  } else if (bStartupSign) {
    radioDetached->Checked = true;
    chbASCIIArmor->Checked = true;
  } else if (bStartupPKEncrypt && !bStartupSEncrypt) {
    rgEncryptionOptions->ItemIndex = GPGEE_ENCRYPTION_PUBLIC;
  } else if (bStartupSEncrypt) {
    rgEncryptionOptions->ItemIndex = GPGEE_ENCRYPTION_SYMMETRIC;
    AutoOkTimer->Enabled = true;
  }  // default signing options
  SanityCheck(NULL);
  __RETURNFUNCTION__;
}  // void __fastcall TformGPGeeSignEncrypt::FormActivateFirst(TObject *Sender)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// OnColumnClick event handler for the public keys listview component
// Change the column that the list is sorted by
//
void __fastcall TformGPGeeSignEncrypt::lvPublicKeysColumnClick(TObject *Sender, TListColumn *Column)
{
  __ENTERFUNCTION__;

  // If a column is clicked on once, sort by that column ascending - if it is clicked on again before another
  // column is clicked on, then sort by that column descending.
  if (Column->Index == pubkeysLastSorted)
    pubkeysAscending = !pubkeysAscending;
  else
    pubkeysLastSorted = Column->Index;
  lvPublicKeys->CustomSort(SortByColumn, Column->Index);

  __RETURNFUNCTION__;
}  // void __fastcall TformGPGeeSignEncrypt::lvPublicKeysColumnClick(TObject *Sender, TListColumn *Column)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Custom listview sort comparison callback function
// Sort by the column selected by the user
//
int CALLBACK SortByColumn(long Item1, long Item2, long Data)
{
  int retval;

  __ENTERFUNCTION__;

  if (!Data)
    retval = AnsiCompareText(((TListItem *)Item1)->Caption, ((TListItem *)Item2)->Caption);
  else
    retval = AnsiCompareText(((TListItem *)Item1)->SubItems->Strings[Data-1], ((TListItem *)Item2)->SubItems->Strings[Data-1]);

  if (!pubkeysAscending)
    retval = -retval;

  __RETURNFUNCTION(retval);
}  // int __stdcall SortByColumn(long Item1, long Item2, long Data)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Perform a sanity check on the settings and (attempt to) intelligently
// alter any that don't make sense.  This is where all the logic is that
// determines what controls are active depending on what options are
// selected.
//
void __fastcall TformGPGeeSignEncrypt::SanityCheck(TObject *Sender)
{
  __ENTERFUNCTION__;

  // Form defaults that depend on the state of the Encryption Options radio group box. The state of the Encryption
  // Options affects what sort of signing options are available and whether the public key list view is enabled.  The
  // listview isn't actually disabled since it looks gross that way.  We fake disabling it by greying out the text and
  // not allowing any changes to it (see the lvPublicKeysChanging() method)
  switch (rgEncryptionOptions->ItemIndex) {
    case GPGEE_ENCRYPTION_NONE:
      if (lvPublicKeys->Font->Color == clWindowText) {
        lvPublicKeys->Selected = NULL;
        lvPublicKeys->Font->Color = clInactiveCaptionText;
      }  // if (lvPublicKeys->Font->Color == clWindowText)
      cbKeyGroups->Enabled = false;
      lblKeyGroups->Enabled = false;
      radioNoSignature->Enabled = true;
      radioAttached->Enabled = true;
      radioDetached->Enabled = true;
      break;
    case GPGEE_ENCRYPTION_PUBLIC:
      lvPublicKeys->Font->Color = clWindowText;
      cbKeyGroups->Enabled = true;
      lblKeyGroups->Enabled = true;
      radioNoSignature->Enabled = true;
      radioAttached->Enabled = true;
      if (radioDetached->Checked)
        radioAttached->Checked = true;
      radioDetached->Enabled = false;
      break;
    case GPGEE_ENCRYPTION_SYMMETRIC:
      if (lvPublicKeys->Font->Color == clWindowText) {
        lvPublicKeys->Selected = NULL;
        lvPublicKeys->Font->Color = clInactiveCaptionText;
      }  // if (lvPublicKeys->Font->Color == clWindowText)
      cbKeyGroups->Enabled = false;
      lblKeyGroups->Enabled = false;
      radioNoSignature->Enabled = false;
      radioAttached->Enabled = false;
      radioDetached->Enabled = false;
      break;
  }  // switch (rgEncryptionOptions->ItemIndex)

  if (cbKeyGroups->Enabled) {
    btnGroupAdd->Enabled = nEncryptCount && cbKeyGroups->ItemIndex == -1 && !cbKeyGroups->Text.IsEmpty();
    btnGroupModify->Enabled = nEncryptCount && cbKeyGroups->ItemIndex != -1;
    btnGroupDelete->Enabled = cbKeyGroups->ItemIndex != -1;
  } else {
    btnGroupAdd->Enabled = false;
    btnGroupModify->Enabled = false;
    btnGroupDelete->Enabled = false;
  }  // if (cbKeyGroups->Enabled)
  ccbSigningKey->Enabled = !radioNoSignature->Checked && radioNoSignature->Enabled;
  lblSigningKey->Enabled = ccbSigningKey->Enabled;
  chbClearsign->Enabled = radioAttached->Checked && radioAttached->Enabled  && rgEncryptionOptions->ItemIndex != GPGEE_ENCRYPTION_PUBLIC;
  chbASCIIArmor->Enabled = rgEncryptionOptions->ItemIndex || !radioNoSignature->Checked;
  btnOk->Enabled = (!radioNoSignature->Enabled || radioNoSignature->Checked || !radioNoSignature->Checked && ccbSigningKey->CheckedCount) && (rgEncryptionOptions->ItemIndex==GPGEE_ENCRYPTION_SYMMETRIC ||  rgEncryptionOptions->ItemIndex==GPGEE_ENCRYPTION_PUBLIC && nEncryptCount || !rgEncryptionOptions->ItemIndex && !radioNoSignature->Checked);
  if (radioNoSignature->Enabled && !radioNoSignature->Checked && rgEncryptionOptions->ItemIndex != GPGEE_ENCRYPTION_NONE)
    IconList->GetIcon(0, Icon);
  else if (radioNoSignature->Enabled && !radioNoSignature->Checked && rgEncryptionOptions->ItemIndex == GPGEE_ENCRYPTION_NONE)
    IconList->GetIcon(1, Icon);
  else
    IconList->GetIcon(2, Icon);
  chbSigAppend->Enabled = !rgEncryptionOptions->ItemIndex && radioDetached->Checked;
  __RETURNFUNCTION__;
}  // void __fastcall TformGPGeeSignEncrypt::SanityCheck(TObject *Sender)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// TListView OnChanging event handler
// Forbid any changes if the text color is clInactiveCaptionText
// We use the text color as the indicator of whether the TListView component
// is enabled or not.  This is instead of using the Enabled property because
// setting Enabled to false makes the listview look really really ugly.
// Graying out the text is much more professional looking.
//
void __fastcall TformGPGeeSignEncrypt::lvPublicKeysChanging(TObject *Sender, TListItem *Item, TItemChange Change, bool &AllowChange)
{
  __ENTERFUNCTION__;
  AllowChange = lvPublicKeys->Font->Color != clInactiveCaptionText;
  __RETURNFUNCTION__;
}
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// TListView OnChange event handler
// We use this event as a signal to count how many keys are checkmarked.
// This count is used so we know when to intelligently enable the Ok button
//
void __fastcall TformGPGeeSignEncrypt::lvPublicKeysChange(TObject *Sender, TListItem *Item, TItemChange Change)
{
  int nCount = 0;
  int nOldEncryptCount = nEncryptCount;

  __ENTERFUNCTION__;

  if (Change == ctState) {
    for (int n = 0; n < lvPublicKeys->Items->Count; n++)
      if (lvPublicKeys->Items->Item[n]->Checked)
        nCount++;
    nEncryptCount = nCount;
    LOG2(LOG_DEBUG, "Currently %d/%d keys selected in encrypt-to list.", nCount, lvPublicKeys->Items->Count)
    if (nEncryptCount != nOldEncryptCount)
       SanityCheck(NULL);
  }  // if (Change == ctState)

  __RETURNFUNCTION__;
}  // void __fastcall TformGPGeeSignEncrypt::lvPublicKeysChange(TObject *Sender, TListItem *Item, TItemChange Change)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// cbKeyGroups OnChange event handler
// When a new key group is selected, update the selected public keys to
// reflect it
//
void __fastcall TformGPGeeSignEncrypt::cbKeyGroupsChange(TObject *Sender)
{
  __ENTERFUNCTION__;

  AnsiString KeyGroup;
  TStringList *KeyGroupMembers = NULL;
  TRegistry *Registry = NULL;

  if (cbKeyGroups->ItemIndex == -1)
    __RETURNFUNCTION__;

  try {
    KeyGroup = cbKeyGroups->Text;

    Registry = new TRegistry();
    Registry->RootKey = HKEY_CURRENT_USER;
    if (!Registry->OpenKey("\\Software\\GPGee\\Key Groups", false)) {
      LOG(LOG_ERROR, "Could not open the Key Groups registry key.");
      throw ERegistryException(GetMessage(MSG_ERROR_GPGEE_KEYGROUPOPEN));
    }  // if (!Registry->OpenKey("\\Software\\GPGee\\Key Groups", false))

    KeyGroupMembers = new TStringList();
    KeyGroupMembers->Text = Registry->ReadString(KeyGroup);
    for (int n = 0; n < lvPublicKeys->Items->Count; n++) {
      TListItem *Item = lvPublicKeys->Items->Item[n];
      AnsiString KeyId = "0x" + IntToHex((int)Item->Data,8);
      Item->Checked = KeyGroupMembers->IndexOf(KeyId) != -1;
    }  // for (int n = 0; n < lvPublicKeys->Items->Count; n++)
    SanityCheck(NULL);
  }  // try
  __finally {
    if (KeyGroupMembers)
      delete KeyGroupMembers;
    if (Registry)
      delete Registry;
    __RETURNFUNCTION__;
  }  // __finally
}  // void __fastcall TformGPGeeSignEncrypt::cbKeyGroupsChange(TObject *Sender)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// cbKeyGroups component OnKeyUp event handler
// Perform a SanityCheck() so that the correct buttons are enabled/disabled
// depending on what the user types in the Key Groups combo box
//
void __fastcall TformGPGeeSignEncrypt::cbKeyGroupsKeyUp(TObject *Sender, WORD &Key, TShiftState Shift)
{
  __ENTERFUNCTION__;
  SanityCheck(NULL);
  __RETURNFUNCTION__;
}  // void __fastcall TformGPGeeSignEncrypt::cbKeyGroupsKeyUp(TObject *Sender, WORD &Key, TShiftState Shift)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// OnClick handler for btnGroupAdd
// Add a new key group to the list.
//
void __fastcall TformGPGeeSignEncrypt::btnGroupAddClick(TObject *Sender)
{
  TStringList *KeyIds = NULL, *KeyGroups = NULL;
  TRegistry *Registry = NULL;
  AnsiString NewGroup;

  __ENTERFUNCTION__;

  try {
    KeyIds = new TStringList;
    KeyGroups = new TStringList;
    Registry = new TRegistry;

    NewGroup = cbKeyGroups->Text;

    Registry->RootKey = HKEY_CURRENT_USER;

    KeyGroups->Text = GPGeeConfig->Values["Key Groups"];
    KeyGroups->Sorted = true;
    KeyGroups->Add(NewGroup);

    LOG2(LOG_DEBUG, "Adding new key group \"%s\" with %d members.", NewGroup, lvPublicKeys->Items->Count);
    for (int n=0; n < lvPublicKeys->Items->Count; n++) {
      TListItem *Item = lvPublicKeys->Items->Item[n];
      if (Item->Checked)
        KeyIds->Add("0x" + IntToHex((int)Item->Data,8));
    }  // for (int n=0; n < lvPublicKeys->Items->Count; n++)

    if (!Registry->OpenKey("\\Software\\GPGee\\Key Groups", true)) {
      LOG(LOG_ERROR, "Could not open/create the Key Groups registry key.");
      throw ERegistryException(GetMessage(MSG_ERROR_GPGEE_KEYGROUPCREATE));
    }  // if (!Registry->OpenKey("\\Software\\GPGee\\Key Groups", true))
    Registry->WriteString(NewGroup, KeyIds->Text);

    GPGeeConfig->Values["Key Groups"] = KeyGroups->Text;
    cbKeyGroups->Items = KeyGroups;
    cbKeyGroups->ItemIndex = cbKeyGroups->Items->IndexOf(NewGroup);

    SanityCheck(NULL);
  }  // try
  __finally {
    if (Registry)
      delete Registry;
    if (KeyGroups)
      delete KeyGroups;
    if (KeyIds)
      delete KeyIds;
    __RETURNFUNCTION__;
  }  // __finally
}  // void __fastcall TformGPGeeSignEncrypt::btnGroupAddClick(TObject *Sender)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// btnGroupModify OnClick handler
// Update the currently selected group to have the members as currently
// checked
//
void __fastcall TformGPGeeSignEncrypt::btnGroupModifyClick(TObject *Sender)
{
  TStringList *KeyIds = NULL;
  TRegistry *Registry = NULL;
  AnsiString ModGroup;

  __ENTERFUNCTION__;

  try {
    KeyIds = new TStringList;
    Registry = new TRegistry;

    ModGroup = cbKeyGroups->Text;

    Registry->RootKey = HKEY_CURRENT_USER;

    LOG2(LOG_DEBUG, "Modifying key group \"%s\" to have %d members.", ModGroup, lvPublicKeys->Items->Count);
    for (int n=0; n < lvPublicKeys->Items->Count; n++) {
      TListItem *Item = lvPublicKeys->Items->Item[n];
      if (Item->Checked)
        KeyIds->Add("0x" + IntToHex((int)Item->Data,8));
    }  // for (int n=0; n < lvPublicKeys->Items->Count; n++)

    if (!Registry->OpenKey("\\Software\\GPGee\\Key Groups", true)) {
      LOG(LOG_ERROR, "Could not open the Key Groups registry key.");
      throw ERegistryException(GetMessage(MSG_ERROR_GPGEE_KEYGROUPOPEN));
    }  // if (!Registry->OpenKey("\\Software\\GPGee\\Key Groups", true))
    Registry->WriteString(ModGroup, KeyIds->Text);
  }  // try
  __finally {
    if (Registry)
      delete Registry;
    if (KeyIds)
      delete KeyIds;
    __RETURNFUNCTION__;
  }  // __finally
}  // void __fastcall TformGPGeeSignEncrypt::btnGroupModifyClick(TObject *Sender)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// btnGroupDelete OnClick handler
// Delete the current group
//
void __fastcall TformGPGeeSignEncrypt::btnGroupDeleteClick(TObject *Sender)
{
  AnsiString DeleteGroup;
  TRegistry *Registry = NULL;
  TStringList *KeyGroups = NULL;

  __ENTERFUNCTION__;

  try {
    DeleteGroup = cbKeyGroups->Text;
    if (Application->MessageBox(FormMessage(MSG_CONFIRM_GPGEE_KEYGROUPDELETE, DeleteGroup.c_str()).c_str(),"Confirm Key Group Deletion", MB_ICONQUESTION | MB_YESNO) == IDYES) {
      Registry = new TRegistry();
      if (!Registry->OpenKey("\\Software\\GPGee\\Key Groups", false)) {
        LOG(LOG_ERROR,"Could not open the Key Groups registry key.");
        throw ERegistryException(GetMessage(MSG_ERROR_GPGEE_KEYGROUPOPEN));
      }  // if (!Registry->OpenKey("\\Software\\GPGee\\Key Groups", false)
      Registry->DeleteValue(DeleteGroup);

      KeyGroups = new TStringList();
      KeyGroups->Text = GPGeeConfig->Values["Key Groups"];
      KeyGroups->Delete(KeyGroups->IndexOf(DeleteGroup));
      GPGeeConfig->Values["Key Groups"] = KeyGroups->Text;
      KeyGroups->Sort();
      cbKeyGroups->Items = KeyGroups;
      cbKeyGroups->ItemIndex = -1;
      cbKeyGroups->Text = "";

      LOG1(LOG_DEBUG, "Deleted key group \"%s\".", DeleteGroup.c_str());

      SanityCheck(NULL);
    }  // if (ok to delete)
  }  // try
  __finally {
    if (KeyGroups)
      delete KeyGroups;
    if (Registry)
      delete Registry;
    __RETURNFUNCTION__;
  }  // __finally
}  // void __fastcall TformGPGeeSignEncrypt::btnGroupDeleteClick(TObject *Sender)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Automatic Ok button click - used when symmetric encryption is selected
//
void __fastcall TformGPGeeSignEncrypt::AutoOkTimerTimer(TObject *Sender)
{
  AutoOkTimer->Enabled = false;
  btnOk->Click();
}  // void __fastcall TformGPGeeSignEncrypt::AutoOkTimerTimer(TObject *Sender)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// btnOk OnClick handler
// Time to roll up our forearms and work some magic.  This is where the
// actual work of signing/encryption is done.
//
void __fastcall TformGPGeeSignEncrypt::btnOkClick(TObject *Sender)
{
  AnsiString sSigningKeyId;
  gpgme_key_t keySecret;
  gpgme_recipients_t keyRecipients = NULL;
  gpgme_error_t err;
  AnsiString sSigningKeys;
  AnsiString sPassphrase;
  AnsiString sRecurse;
  gpgme_sigmode_t nSignatureMode;

  __ENTERFUNCTION__;

  formGPGeePassPhrase = NULL;
  try {
    try {
      bPWCancelled = bCancelled = false;
      // These are class variables, not local static variables in the GetPassphrase function (like they are in decrypt form's
      // version of that function) because here the user can start an op, cancel, and start is again.  So we need to be able
      // to clear them from this function, even though they are only ever used in the GetPassphrase() function.
      nPreviousOpNumber = -1;
      sPreviousKeyId = "";

      //---------------------------------------------------------------------------
      // SETUP PHASE
      //---------------------------------------------------------------------------

      // SIGNING - setup
      // If we've been asked to sign anything, then we're going to need to grab the signing key id and then get a passphrase
      // We also set the signature mode here.
      if (!radioNoSignature->Checked && radioNoSignature->Enabled) {
        gpgme_signers_clear(ctxGPG);
        for (int n=0; n < ccbSigningKey->Items->Count; n++) {
          if (ccbSigningKey->Checked[n]) {
            sSigningKeyId = IntToHex((int)ccbSigningKey->Items->Objects[n],8);
            LOG1(LOG_MESSAGE, "Adding secret key id 0x%s to signature key list.", sSigningKeyId.c_str());
            gpgme_keycache_rewind(cacheSecret);
            err = gpgme_keycache_find_key(cacheSecret, sSigningKeyId.c_str(), false, &keySecret);
            if (err) {
              LOG1(LOG_ERROR, "Could not re-find secret key id 0x%s in the keycache.", sSigningKeyId.c_str());
              throw EGPGMEError(FormMessage(MSG_ERROR_GPGEE_FIND_SECRETKEY,sSigningKeyId.c_str(),gpgme_strerror(err)));
            }  // if (err)
            err = gpgme_signers_add(ctxGPG, keySecret);
            if (err) {
              LOG1(LOG_ERROR, "Could not add secret key id 0x%s to the GPGME signers list.", sSigningKeyId.c_str());
              throw EGPGMEError(FormMessage(MSG_ERROR_GPGEE_SET_SIGNINGKEY,sSigningKeyId.c_str(),gpgme_strerror(err)));
            }  // if (err)
            sSigningKeys += "0x" + sSigningKeyId + " ";
          }  // if (ccbSigningKey->Checked[n])
        }  // for (int n=0, n < ccbSigningKey->CheckedCount, n++)
        if (radioAttached->Checked)
          if (chbClearsign->Checked && chbClearsign->Enabled)
            nSignatureMode = GPGME_SIG_MODE_CLEAR;
          else
            nSignatureMode = GPGME_SIG_MODE_NORMAL;
        else
          nSignatureMode = GPGME_SIG_MODE_DETACH;
      }  // if (!radioNoSignature->Checked)

      // PK ENCRYPTION - setup
      // Check if we are encrypting and set things up if we are...
      // For pk encryption, go through the keys listed in the lvPublicKeys listview component and for each one that
      // is checked, add it as a recipient.
      if (rgEncryptionOptions->ItemIndex == GPGEE_ENCRYPTION_PUBLIC) {
        LOG(LOG_DEBUG, "PK encryption requested, parsing encrypt-to list.");
        gpgme_recipients_new(&keyRecipients);
        for (int n = 0; n < lvPublicKeys->Items->Count; n++) {
          TListItem *item = lvPublicKeys->Items->Item[n];
          if (item->Checked) {
            AnsiString sEncryptKeyId;
            sEncryptKeyId = "0x" + IntToHex((int)item->Data,8);
            LOG1(LOG_DEBUG, "Adding public key id %s as a recipient.", sEncryptKeyId.c_str());
            err = gpgme_recipients_add_name(keyRecipients, sEncryptKeyId.c_str());
            if (err) {
              LOG2(LOG_ERROR, "Error adding public key id %s as a recipient: %s", sEncryptKeyId.c_str(), gpgme_strerror(err));
              throw EGPGMEError(FormMessage(MSG_ERROR_GPGEE_ADD_RECIPIENT,sEncryptKeyId.c_str(),gpgme_strerror(err)));
            }  // if (err)
          }  // if (item->Checked)
        }  // for (int n = 0; n < lvPublicKeys->Items->Count; n++)
      }  // if (rgEncryptionOptions->ItemIndex == GPGEE_ENCRYPTION_PUBLIC)

      // SYMMETRIC ENCRYPTION - setup
      // For symmetric encryption all we need is a passphrase - we have to get it up front because MyGPGME's passphrase
      // callback doesn't seem to work for symmetric encryption.
      else if (rgEncryptionOptions->ItemIndex == GPGEE_ENCRYPTION_SYMMETRIC) {
        int nPassphraseResult;
        LOG(LOG_MESSAGE, "Symmetric encryption requested.");
        // Create the passphrase requester form
        formGPGeePassPhrase = new TformGPGeePassPhrase(NULL);
        nPassphraseResult = formGPGeePassPhrase->GetPassphrase("");
        if (nPassphraseResult == mrOk) {
          sPassphrase = formGPGeePassPhrase->edtPassphrase->Text;
          VirtualLock((void *)(sPassphrase.data()),sPassphrase.Length());
        }  // if (nPassphraseResult == mrOk)
        else {  // else if (nPassphraseResult != mrOk)
          LOG(LOG_DEBUG, "User cancelled passphrase dialog.");
          __RETURNFUNCTION__;
        }  // else if (nPassphraseResult != mrOk)
      }  // else if (rgEncryptionOptions->ItemIndex == GPGEE_ENCRYPTION_SYMMETRIC)

      // Configure ASCII armor
      LOG1(LOG_DEBUG, "Setting ASCII armor to %s.", chbASCIIArmor->Checked?"true":"false");
      gpgme_control(ctxGPG, GPGME_CTRL_ARMOR, chbASCIIArmor->Checked);

      // Configure textmode
      LOG1(LOG_DEBUG, "Setting textmode to %s.", chbTextMode->Checked?"true":"false");
      gpgme_control(ctxGPG, GPGME_CTRL_TEXTMODE, chbTextMode->Checked);

      //---------------------------------------------------------------------------
      // OPERATION PHASE
      //---------------------------------------------------------------------------

      // Find out if we will be processing directories
      sRecurse = String(GPGeeConfig->Values["Recurse Directories"]).LowerCase();

      // Change the mouse cursor to an hourglass...
      for (int n = ComponentCount - 1; n >= 0; n--)
        if (Components[n]->InheritsFrom(__classid(TControl)))
          ((TControl *)Components[n])->Cursor = crHourGlass;

      // Whatever we've just been told to do, lets do it to each file highlighted by the user.  This file list
      // was passed to this form from GPGeeContextMenuExtensionImpl.cpp via the ShowModalWithDefaults() method
      for (nOpNumber=0; FileNames->Count != 0 && !bCancelled; nOpNumber++) {
        Application->ProcessMessages();

        AnsiString sUnquotedFileName = FileNames->Strings[0];
        // The file name fed to GPGME operations has to have quotes around it for passing to the gpg.exe command line.
        sCurrentOpFileName = "\"" + sUnquotedFileName + "\"";

        // If this is a directory then process it in the way specified by the preferences
        if (DirectoryExists(sUnquotedFileName)) {
          if (sRecurse == "ask")
            switch (Application->MessageBox(GetMessage(MSG_QUERY_DIRECTORIES_MSG).c_str(), GetMessage(MSG_QUERY_DIRECTORIES_CAPTION).c_str(), MB_YESNOCANCEL | MB_ICONQUESTION)) {
              case IDYES:    sRecurse = "always"; break;
              case IDNO:     sRecurse = "never";  break;
              case IDCANCEL: FileNames->Clear();  continue;
            }  // switch (messagebox result)
          if (sRecurse == "always")
            DirectoryList(sUnquotedFileName, FileNames);
          FileNames->Delete(0);
          continue;
        }  // if (DirectoryExists(sCurrentOpFileName))

        switch (rgEncryptionOptions->ItemIndex) {

          //
          // This handles sign-only operations
          //
          case GPGEE_ENCRYPTION_NONE: {
            AnsiString sSignedFilename;
            AnsiString sOutputFilename;
            bool bAppend = false;
            LOG1(LOG_MESSAGE, "Performing sign operation on %s.", sCurrentOpFileName.c_str());
            if (nSignatureMode == GPGME_SIG_MODE_DETACH) {
              // Check for if a signature exists already - if it does, offer to append to it rather than overwrite
              if (chbASCIIArmor->Checked)
                sSignedFilename = sUnquotedFileName + ".asc";
              else
                sSignedFilename = sUnquotedFileName + ".gpg";
              if (FileExists(sSignedFilename)) {
                if (chbSigAppend->State == cbChecked ||
                     (chbSigAppend->State == cbGrayed &&
                     (Application->MessageBoxA(FormMessage(MSG_QUERY_SIGAPPEND_MSG, sSignedFilename.c_str()).c_str(),
                                            GetMessage(MSG_QUERY_SIGAPPEND_CAPTION).c_str(), MB_YESNO) == IDYES))) {
                  bAppend = true;
                  sOutputFilename = GetTempFileName("GPGee");
                }  // if (should append)
              }  // if (FileExists(sOutputFilename))
            }  // if (nSignatureMode == GPGME_SIG_MODE_DETACH)
            // Sign the file
            err = gpgme_op_file_sign(ctxGPG, nSignatureMode, sCurrentOpFileName.c_str(), bAppend?("\"" + sOutputFilename + "\"").c_str():NULL);
            // Append the signature onto the outputfile, if we were requested to do so, then delete the temp file
            if (bAppend) {
              if (!AppendFile(sOutputFilename, sSignedFilename))
                throw EGPGeeFileError(FormMessage(MSG_ERROR_GPGEE_APPENDSIGNATURE, sSignedFilename.c_str()));
              else
                DeleteFile(sOutputFilename);
            }  // if (bAppend)
            break;
          }  // case GPGEE_ENCRYPTION_NONE:

          //
          // This handles both encrypt-only and sign+encrypt.  Most of this code is to handle the case where
          // encrypt-to-self has been selected.
          //
          case GPGEE_ENCRYPTION_PUBLIC:   // PK encryption
            // Check if we are signing and encrypting, or encrypting only
            if (radioNoSignature->Enabled && !radioNoSignature->Checked) {
              // if encrypt-to-self is set then add the key used to sign to the list of recipients
              if (GPGeeConfig->Values["Encrypt to Self"]) {
                gpgme_key_t EncryptSelfKey;
                AnsiString sEncryptSelfKeyId;
                int nSubkey;
                const char *caps;
                bool bValidkey = false;

                // Find the first encryption subkey of the keys that are used to sign with...
                for (int l=0; l < ccbSigningKey->Items->Count; l++) {
                  if (!ccbSigningKey->Checked[l])
                    continue;
                  gpgme_keycache_rewind(cachePublic);
                  sSigningKeyId = IntToHex((int)ccbSigningKey->Items->Objects[l],8);
                  err = gpgme_keycache_find_key(cachePublic, sSigningKeyId.c_str(), false, &EncryptSelfKey);
                  if (err == GPGME_No_Error) {
                    while ((caps = gpgme_key_get_string_attr(EncryptSelfKey, GPGME_ATTR_KEY_CAPS, NULL, nSubkey)) != NULL) {
                      // If the key can be encrypted to and isn't revoked or expired, we'll call it valid and add it
                      if (strchr(caps, 'e') && gpgme_key_get_ulong_attr(EncryptSelfKey, GPGME_ATTR_KEY_USABLE, NULL, nSubkey)) {
                        bValidkey = true;
                        break;
                      }  // if (strchr(caps, 'e'))
                      nSubkey++;
                    }  // while ((caps = gpgme_key_get_string_attr(key, GPGME_ATTR_KEY_CAPS, NULL, nSubkey)) != NULL)
                    if (bValidkey) {
                      sEncryptSelfKeyId = String(gpgme_key_get_string_attr(EncryptSelfKey, GPGME_ATTR_KEYID, NULL, nSubkey)).Delete(1,8);
                      LOG1(LOG_DEBUG, "Adding public encryption key id %s to recipient list for encrypt-to-self.", sEncryptSelfKeyId.c_str());
                      gpgme_recipients_add_name(keyRecipients, sEncryptSelfKeyId.c_str());
                    }  // if (bValidKey)
                  }  // if (err != GPGME_No_Error)
                  else
                    LOG1(LOG_WARNING, "Could not find public encryption key matching secret key id %s for encrypt-to-self.",sSigningKeyId.c_str());
                }  // for (int l=0; l < ccbSigningKey->Items->Count; l++)
              }  // if (GPGeeConfig->Values["Encrypt to Self"])
              LOG1(LOG_MESSAGE, "Performing sign+encrypt operation on %s.", sCurrentOpFileName.c_str());
              // Sign and encrypt the file
              err = gpgme_op_file_sign_encrypt(ctxGPG, keyRecipients, sCurrentOpFileName.c_str(), NULL);
            }  // if (radioNoSignature->Enabled && !radioNoSignature->Checked)
            else {
              LOG1(LOG_MESSAGE, "Performing PK-encryption operation on %s.", sCurrentOpFileName.c_str());
              // PK-encrypt the file
              err = gpgme_op_file_encrypt(ctxGPG, keyRecipients, sCurrentOpFileName.c_str(), NULL);
            }  // if (radioNoSignature->Enabled && !radioNoSignature->Checked) else
            break;

          //
          // This handles symmetric encryption
          //
          case GPGEE_ENCRYPTION_SYMMETRIC:
            LOG1(LOG_MESSAGE, "Performing symmetric encryption operation on %s.", sCurrentOpFileName.c_str());
            // We are performing symmetrical encryption so we need to set the passphrase
            gpgme_set_passphrase(ctxGPG, sPassphrase.c_str());
            err = gpgme_op_file_encrypt(ctxGPG, NULL, sCurrentOpFileName.c_str(), NULL);
          }  // switch (rgEncryptionOptions->ItemIndex)

        if (err) {
          LOG1(LOG_ERROR, "Operation failed: %s.", gpgme_strerror(err));
          // If the operation fails, tell the user about it and bail out
          throw EGPGMEError(FormMessage(MSG_ERROR_GPGEE_FAILED_OPERATION,gpgme_strerror(err)));
        }  // if (err)
        if (chbReadOnly->Checked) {
          int nFileAttributes;
          nFileAttributes = FileGetAttr(sUnquotedFileName);
          FileSetAttr(sUnquotedFileName, faReadOnly | nFileAttributes);
        }  // if (chbReadOnly->Checked)
        FileNames->Delete(0);
      }  // for (int n=0; n < FileNames->Count && !bCancelled; n++)
      // At this point we have successfully performed the operation - if we have signed a file, then set the key we used
      // to sign with as the default signing key for next time.
      if (radioNoSignature->Enabled && !radioNoSignature->Checked)
        GPGeeConfig->Values["Signing Keys"] = sSigningKeys;
      ModalResult = mrOk;
      // Close();
    }  // inner try

    catch (Exception &e) {
      CleanupPassphraseCache(PassphraseCache);
      LOG(LOG_ERROR, "Exception " + e.ClassName() + " raised with message " + e.Message);
      if (!bPWCancelled)
        Application->MessageBox(e.Message.c_str(),"GPGee Exception",MB_OK);
      // MyGPGME has problems ever accepting a passphrase again once a wrong or null one was given - so we need to
      // reset the MyGPGME main context at this point.
      ResetGPGMEContext(&ctxGPG);
    }  // catch
  }  // outer try
  // Cleanup after ourselves
  __finally {
    // Set the cursor back to default - in case there was an error and the form is still up.
    for (int n = ComponentCount - 1; n >= 0; n--)
      if (Components[n]->InheritsFrom(__classid(TControl)))
        ((TControl *)Components[n])->Cursor = crDefault;
    // Free allocated memory for our recipient key list (if any)
    if (keyRecipients)
      gpgme_recipients_release(keyRecipients);
    // Delete the passphrase form, if it was used
    if (formGPGeePassPhrase) {
      delete formGPGeePassPhrase;
      formGPGeePassPhrase = NULL;
    }  // if (formGPGeePassPhrase)
    // Wipe the passphrase cache
    CleanupPassphraseCache(PassphraseCache);
    // Wipe the local copy of the passphrase we used for symmetric encryption and also the copy we sent to gpgme.
    if (!sPassphrase.IsEmpty()) {
      VirtualUnlock((void *)(sPassphrase.data()), sPassphrase.Length());
      sPassphrase = sPassphrase.StringOfChar('*',128);
      gpgme_set_passphrase(ctxGPG, sPassphrase.c_str());
      gpgme_set_passphrase(ctxGPG, NULL);
    }  // if (!sPassphrase.IsEmpty())
    __RETURNFUNCTION__;
  }  // __finally
}  // void __fastcall TformGPGeeSignEncrypt::btnOkClick(TObject *Sender)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// btnCancel OnClick event handler - user clicked Cancel
//
void __fastcall TformGPGeeSignEncrypt::btnCancelClick(TObject *Sender)
{
  __ENTERFUNCTION__
  bCancelled = true;
  ModalResult = mrCancel;
  __RETURNFUNCTION__;
}  // void __fastcall TformGPGeeSignEncrypt::btnCancelClick(TObject *Sender)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Help button OnClick event handler
// Bring up the help file on the title page
//
void __fastcall TformGPGeeSignEncrypt::btnHelpClick(TObject *Sender)
{
  __ENTERFUNCTION__;
  Application->HelpContext(2);
  __RETURNFUNCTION__;
}
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// This is heavily (read a copy of that was slightly edited) based on the
// method of the same name in the TformGPGeeVerifyDecrypt class.
// One of these days, the two should be consolidated.
//
const char * __fastcall TformGPGeeSignEncrypt::GetPassphrase(const char *desc)
{
  static AnsiString Passphrase = "";
  gpgme_key_t Key;
  AnsiString KeyId, KeyName, KeyType, KeySize, KeyCreationDate;
  int nCachePos;

  __ENTERFUNCTION__;

  if (bPWCancelled)
    __RETURNFUNCTION(NULL);

  if (desc) {
    TStringList *desclines = new TStringList;

    // Drop the passphrase "description" text into a TStringList to parse it out into the individual lines
    desclines->Text = desc;
    LOG1(LOG_DEBUG, "Passphrase callback activated: User ID hint=\"%s\".", desclines->Count==1?desclines->Strings[0].c_str():desclines->Strings[1].c_str());
    // If there is only one line in the hint string, then it is the serial number of a smartcard
    if (desclines->Count == 1) {
      KeyId = desclines->Strings[0];
      nCachePos = PassphraseCache->IndexOfName(KeyId);
      if (nCachePos != -1 && (nOpNumber != nPreviousOpNumber || KeyId != sPreviousKeyId)) {
        nPreviousOpNumber = nOpNumber;
        sPreviousKeyId = KeyId;
        Passphrase = PassphraseCache->Values[KeyId];
        __RETURNFUNCTION(Passphrase.c_str());
      }  // if (nCachePos != -1 && nOpNumber != nPreviousOpNumber)
      formGPGeePassPhrase = new TformGPGeePassPhrase(NULL);
      if (formGPGeePassPhrase->GetPassphrase("", KeyId, "", "", "") == mrOk)
        Passphrase = formGPGeePassPhrase->edtPassphrase->Text;
      else { // if (cancelled)
        Passphrase = "";
        bPWCancelled = true;
      }  // else if (cancelled)
    }  // if (desclines->Count == 1)
    // If there is no User ID hint then we assume that we are being invoked to get a symmetrical-encryption passphrase
    // That shouldn't happen here because passphrase callbacks aren't used for symetrical encryption, but we handle it anyway
    // for future compatibility
    else if (desclines->Strings[1] == "[User ID hint missing]") {
      formGPGeePassPhrase = new TformGPGeePassPhrase(NULL);
      if (formGPGeePassPhrase->GetPassphrase(FormMessage(MSG_QUERY_GPGEE_SYMPASSPHRASE,sCurrentOpFileName.c_str())) == mrOk)
        Passphrase = formGPGeePassPhrase->edtPassphrase->Text;
      else {  // else if (formGPGeePassPhrase->GetPassphrase(FormMessage(MSG_QUERY_GPGEE_SYMPASSPHRASE,sCurrentOpFileName.c_str())) != mrOk)
        Passphrase = "";
        bPWCancelled = true;
      }  // else if (formGPGeePassPhrase->GetPassphrase(FormMessage(MSG_QUERY_GPGEE_SYMPASSPHRASE,sCurrentOpFileName.c_str())) != mrOk)
    }  // if (desclines->Strings[1] == "[User ID hint missing]")
    // Otherwise, check line 2 of the description for the key id of the secret key we are getting the passphrase for and
    // present the user with the info about that key
    else if (!desclines->Strings[1].IsEmpty()) {
      KeyId = desclines->Strings[1].SubString(9,8);
      nCachePos = PassphraseCache->IndexOfName(KeyId);
      if (nCachePos != -1 && (nOpNumber != nPreviousOpNumber || KeyId != sPreviousKeyId)) {
        nPreviousOpNumber = nOpNumber;
        sPreviousKeyId = KeyId;
        Passphrase = PassphraseCache->Values[KeyId];
        __RETURNFUNCTION(Passphrase.c_str());
      }  // if (nCachePos != -1 && nOpNumber != nPreviousOpNumber)
      gpgme_keycache_rewind(cacheSecret);
      if (gpgme_keycache_find_key(cacheSecret,KeyId.c_str(), false, &Key) == GPGME_No_Error) {
        KeyName = String(gpgme_key_get_string_attr(Key, GPGME_ATTR_NAME, NULL, 0)) + " <" + String(gpgme_key_get_string_attr(Key, GPGME_ATTR_EMAIL, NULL, 0)) + ">";
        KeyType = String(gpgme_key_get_string_attr(Key, GPGME_ATTR_ALGO, NULL, 0));
        KeySize = String(gpgme_key_get_ulong_attr(Key, GPGME_ATTR_LEN, NULL, 0));
        KeyCreationDate = DateToStr(GPGToDateTime(gpgme_key_get_ulong_attr(Key, GPGME_ATTR_CREATED, NULL, 0)));
        formGPGeePassPhrase = new TformGPGeePassPhrase(NULL);
        if (formGPGeePassPhrase->GetPassphrase(KeyName, KeyId, KeyType, KeySize, KeyCreationDate) == mrOk)
          Passphrase = formGPGeePassPhrase->edtPassphrase->Text;
        else {  // (formGPGeePassPhrase->GetPassphrase(KeyName, KeyId, KeyType, KeySize, KeyCreationDate) != mrOk)
          Passphrase = "";
          bPWCancelled = true;;
        }  // else (formGPGeePassPhrase->GetPassphrase(KeyName, KeyId, KeyType, KeySize, KeyCreationDate) != mrOk)
      }  // if (gpgme_keycache_find_key(cacheSecret,KeyId.c_str(), false, &Key) == GPGME_No_Error)
      else
        LOG(LOG_WARNING, "Could not find secret key referenced by passphrase callback in keycache.");
    }  // else if (desclines->Strings[1].IsEmpty())
    // If we used the passphrase form, delete it
    if (formGPGeePassPhrase) {
      delete formGPGeePassPhrase;
      formGPGeePassPhrase = NULL;
    }  // if (formGPGeePassPhrase)
    nPreviousOpNumber = nOpNumber;
    sPreviousKeyId = KeyId;
    // If there are any previous entries in the cache for this key id, then they were typos, otherwise we wouldn't
    // be asked for this passphrase again.  Delete any previous entries.  There should only ever be one, but loop just in case.
    for (int n = PassphraseCache->IndexOfName(KeyId); n != -1; n = PassphraseCache->IndexOfName(KeyId))
      PassphraseCache->Delete(n);
    PassphraseCache->Add(KeyId + "=" + Passphrase);
    VirtualLock((void *)(PassphraseCache->Strings[PassphraseCache->Count-1].data()),Passphrase.Length());
    __RETURNFUNCTION(Passphrase.c_str());
  }  // if (desc)
  // If we're called with an empty desc then we are being asked to clean up after a previous call (the passphrase is
  // no longer needed).  If our passphrase variable isn't empty then overwrite it before setting it to null length.
  else if (!Passphrase.IsEmpty()) {
    // Make sure that when the passphrase is no longer needed that it is erased
    Passphrase.StringOfChar('*', Passphrase.Length());
    Passphrase.StringOfChar('*', 128);
    Passphrase = "";
  }  // else if (!Passphrase.IsEmpty())
  __RETURNFUNCTION(NULL);
}  // const char * __fastcall TformGPGeeSignEncrypt::GetPassphrase(const char *desc)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// The actual passphrase callback - just dumps control to the Passphrase
// method of the form class.
//
static const char * PassphraseCallback(void *hook, const char *desc, void *dummy)
{
  const char *retval;

  __ENTERFUNCTION__;
  TformGPGeeSignEncrypt *formGPGeeSignEncrypt = (TformGPGeeSignEncrypt *)hook;
  retval = formGPGeeSignEncrypt->GetPassphrase(desc);
  __RETURNFUNCTION(retval);
}
//---------------------------------------------------------------------------

