unit Undo;

{
  Inno Setup
  Copyright (C) 1998-2000 Jordan Russell
  For conditions of distribution and use, see LICENSE.TXT.

  Uninstallation Procedures
}

{ Note: This unit is shared by both the 'Setup' and 'Uninst' projects }

interface

{$I VERSION.INC}

uses
  WinProcs, WinTypes, SysUtils;

const
  HighestSupportedVersion = 4;
  { Each time the format of the uninstall log changes (usually a new entry type
    is added), I increment HighestSupportedVersion and the version info in
    Uninst.res to match (51.x). Do NOT do this yourself; doing so could cause
    incompatibilies with future Inno Setup releases. It's recommended that you
    use the "utUserDefined" log entry type if you wish to implement your own
    custom uninstall log entries; see below for more information. }

type
  TUninstallRecTyp = type Word;
const
  { Values for TUninstallRecTyp.
    If you wish to define your own custom uninstall entry type, you should use
    "utUserDefined". (Do NOT define your own ut* constants; this could cause
    incompatibilies with future Inno Setup releases.) The first field in a
    utUserDefined record must be a string which specifies a unique name for
    the record type. Example:
    UninstLog.Add (utUserDefined, ['MyRecordType', ... ], 0);
 }
  utUserDefined          = $01;
  utStartInstall         = $10;
  utEndInstall           = $11;
  utRun                  = $80;
  utDeleteDirOrFiles     = $81;
  utDeleteFile           = $82;
  utDeleteGroupOrItem    = $83;
  utIniDeleteEntry       = $84;
  utIniDeleteSection     = $85;
  utRegDeleteEntireKey   = $86;
  utRegClearValue        = $87;
  utRegDeleteKeyIfEmpty  = $88;
  utRegDeleteValue       = $89;
  utDecrementSharedCount = $8A;
  utRefreshFileAssoc     = $8B;
  utMutexCheck           = $8C;
  utRunCompleted         = $FFF0;  { dummy }

  ValidUninstallRecTypes: array[0..13] of TUninstallRecTyp = (
    utUserDefined, utStartInstall, utEndInstall, utRun, utDeleteDirOrFiles,
    utDeleteFile, utDeleteGroupOrItem, utIniDeleteEntry, utIniDeleteSection,
    utRegDeleteEntireKey, utRegClearValue, utRegDeleteKeyIfEmpty,
    utRegDeleteValue, utDecrementSharedCount);

  { Flags on ExtraData }
  utRun_NoWait = 1;
  utRun_WaitUntilIdle = 2;
  utRun_ShellExec = 4;
  utRun_RunMinimized = 8;
  utRun_RunMaximized = 16;
  utDeleteFile_ExistedBeforeInstall = 1;
  utDeleteFile_Extra = 2;
  utDeleteFile_IsFont = 4;
  utDeleteFile_SharedFile = 8;
  utDeleteFile_RegisteredServer = 16;
  utDeleteFile_CallChangeNotify = 32;
  utDeleteFile_RegisteredTypeLib = 64;
  utDeleteDirOrFiles_Extra = 1;
  utDeleteDirOrFiles_IsDir = 2;
  utDeleteDirOrFiles_DeleteFiles = 4;
  utDeleteDirOrFiles_DeleteSubdirsAlso = 8;
  utDeleteDirOrFiles_CallChangeNotify = 16;
  utIniDeleteSection_OnlyIfEmpty = 1;

const
  UninstallRecNonDataSize =
    (SizeOf(Pointer) * 2) + SizeOf(Longint) + SizeOf(Cardinal) + SizeOf(TUninstallRecTyp);
type
  PUninstallRec = ^TUninstallRec;
  TUninstallRec = packed record
    {if this is modified, you must to update UninstallRecNonDataSize above}
    Prev, Next: PUninstallRec;
    ExtraData: Longint;
    DataSize: Cardinal;
    Typ: TUninstallRecTyp;
    Data: array[0..$6FFFFFFF] of Char;
  end;

  TDeleteUninstallDataFilesProc = procedure;

  TUninstallLogFlags = set of (ufAdminInstalled, ufDontCheckRecCRCs, ufModernStyle);

  TUninstallLog = class
  private
    FList, FLastList: PUninstallRec;
    FCount: Integer;
    procedure InternalAdd (const Typ: TUninstallRecTyp; const Data: Pointer;
      const Size: Cardinal; const ExtraData: Longint);
  protected
    function ShouldRemoveSharedFile (const Filename: String): Boolean; virtual;
    procedure ShowException (E: Exception); virtual;
    procedure StatusUpdate (StartingCount, CurCount: Integer); virtual;
  public
    AppId, AppName: String;
    Flags: TUninstallLogFlags;
    Version: Integer;
    constructor Create;
    destructor Destroy; override;
    procedure Add (const Typ: TUninstallRecTyp; const Data: array of String;
      const ExtraData: Longint);
    function CheckMutexes: Boolean;
    procedure Clear;
    function ExtractRecData (const Rec: PUninstallRec;
      var Data: array of String): Integer;
    procedure Load (const Filename: String);
    function PerformUninstall (const CallFromUninstaller: Boolean;
      const DeleteUninstallDataFilesProc: TDeleteUninstallDataFilesProc): Boolean;
    procedure Save (const Filename: String;
      const Append, UpdateUninstallLogAppName: Boolean);
    function Test (const Filename, AAppId: String): Boolean;
    property List: PUninstallRec read FList;
    property LastList: PUninstallRec read FLastList;
  end;

implementation

uses
  Messages, ShellAPI, ShlObj,
  {$IFNDEF Delphi3orHigher} OLE2, {$ELSE} ActiveX, ComObj, {$ENDIF}
  CmnFunc2, Struct, Msgs, MsgIDs, DDEInt, InstFunc, zlib;

type
  TUninstallLogHeader = packed record
    ID: TUninstallLogID;
    AppId: array[0..127] of Char;
    AppName: array[0..127] of Char;
    Version, NumRecs: Integer;
    EndOffset: Cardinal;
    Flags: Longint;
    Reserved: array[0..26] of Longint;  { reserved for future use }
    CRC: Longint;
  end;
  TUninstallCrcHeader = packed record
    Size, NotSize: Cardinal;
    CRC: Longint;
  end;
  TUninstallFileRec = packed record
    Typ: TUninstallRecTyp;
    ExtraData: Longint;
    DataSize: Cardinal;
  end;


{ Misc. uninstallation functions }

function DecrementSharedCount (const Filename: String): Boolean;
{ Returns True if OK to delete }
const
  SharedDLLsKey = NEWREGSTR_PATH_SETUP + '\SharedDLLs';  {don't localize}
var
  K: HKEY;
  Count, CurType, NewType, Size, Disp: DWORD;
  CountStr: String;
begin
  Result := False;

  if RegCreateKeyEx(HKEY_LOCAL_MACHINE, SharedDLLsKey, 0, nil, REG_OPTION_NON_VOLATILE,
     KEY_QUERY_VALUE or KEY_SET_VALUE, nil, K, @Disp) <> ERROR_SUCCESS then
    raise Exception.Create(FmtSetupMessage(msgErrorRegOpenKey,
      [RegRootKeyNames[HKEY_LOCAL_MACHINE], SharedDLLsKey]));
  if RegQueryValueEx(K, PChar(Filename), nil, @CurType, nil, @Size) <> ERROR_SUCCESS then begin
    RegCloseKey (K);
    Exit;
  end;

  Count := 2;
  NewType := REG_DWORD;
  try
    case CurType of
      REG_SZ:
        if RegQueryStringValue(K, PChar(Filename), CountStr) then begin
          Count := StrToInt(CountStr);
          NewType := REG_SZ;
        end;
      REG_BINARY: begin
          if (Size >= 1) and (Size <= 4) then begin
            if RegQueryValueEx(K, PChar(Filename), nil, nil, @Count, @Size) <> ERROR_SUCCESS then
              { ^ relies on the high 3 bytes of Count being initialized to 0 }
              Abort;
            NewType := REG_BINARY;
          end;
        end;
      REG_DWORD: begin
          Size := SizeOf(DWORD);
          if RegQueryValueEx(K, PChar(Filename), nil, nil, @Count, @Size) <> ERROR_SUCCESS then
            Abort;
        end;
    end;
  except
    Count := 2;  { default to 2 if an error occurred }
  end;
  Dec (Count);
  if Count <= 0 then begin
    Result := True;
    RegDeleteValue (K, PChar(Filename));
  end
  else begin
    case NewType of
      REG_SZ: begin
          CountStr := IntToStr(Count);
          RegSetValueEx (K, PChar(Filename), 0, NewType, PChar(CountStr), Length(CountStr)+1);
        end;
      REG_BINARY, REG_DWORD:
        RegSetValueEx (K, PChar(Filename), 0, NewType, @Count, SizeOf(Count));
    end;
  end;
  RegCloseKey (K);
end;

function UnregisterServer (const Filename: String): Boolean;
var
  SaveErrorMode: UINT;
  LibHandle: THandle;
  UnregisterServerProc: function: HRESULT; stdcall;
begin
  Result := True;
  try
    { Add SEM_FAILCRITICALERRORS to the error mode so that no critical error
      boxes are displayed on NT if the DLL references another DLL which doesn't
      exist. }
    SaveErrorMode := SetErrorMode(SEM_NOOPENFILEERRORBOX or SEM_FAILCRITICALERRORS);
    try
      LibHandle := LoadLibrary(PChar(Filename));
      if LibHandle <> 0 then begin
        try
          @UnregisterServerProc := GetProcAddress(LibHandle, 'DllUnregisterServer');
          if Assigned(@UnregisterServerProc) and SUCCEEDED(UnregisterServerProc) then
            Exit;
        finally
          FreeLibrary (LibHandle);
        end;
      end;
    finally
      SetErrorMode (SaveErrorMode);
    end;
  except
  end;
  Result := False;
end;

function UnregisterTypeLib (const Filename: String): Boolean;
type
  TUnRegTlbProc = function (const libID: TGUID; wVerMajor, wVerMinor: Word;
    lcid: TLCID; syskind: TSysKind): HResult; stdcall;
{$IFNDEF Delphi3OrHigher}
var
  WideFilename: PWideChar;
  TypeLib: ITypeLib;
  OleAutLib: THandle;
  UnRegTlbProc: TUnRegTlbProc;
  LibAttr: PTLibAttr;
begin
  Result := False;
  WideFilename := nil;
  TypeLib := nil;
  LibAttr := nil;
  try
    { Dynamically import UnRegisterTypeLib since older OLEAUT32.DLL versions
      don't have this function }
    OleAutLib := GetModuleHandle('OLEAUT32.DLL');
    if OleAutLib = 0 then
      Exit;
    @UnRegTlbProc := GetProcAddress(OleAutLib, 'UnRegisterTypeLib');;
    if @UnRegTlbProc = nil then
      Exit;
    WideFilename := StringToOleStr(ExpandFilename(Filename));
    if FAILED(LoadTypeLib(WideFilename, TypeLib)) then
      Exit;
    if FAILED(TypeLib.GetLibAttr(LibAttr)) then
      Exit;
    with LibAttr^ do
      if FAILED(UnRegTlbProc(Guid, wMajorVerNum, wMinorVerNum, LCID, SysKind)) then
        Exit;
  finally
    if Assigned(TypeLib) then begin
      if Assigned(LibAttr) then
        TypeLib.ReleaseTLibAttr (LibAttr);
      TypeLib.Release;
    end;
    if Assigned(WideFilename) then
      SysFreeString (WideFilename);
  end;
  Result := True;
end;
{$ELSE}
var
  WideFilename: WideString;
  TypeLib: ITypeLib;
  OleAutLib: THandle;
  UnRegTlbProc: TUnRegTlbProc;
  LibAttr: PTLibAttr;
begin
  Result := False;
  { Dynamically import UnRegisterTypeLib since older OLEAUT32.DLL versions
    don't have this function }
  OleAutLib := GetModuleHandle('OLEAUT32.DLL');
  if OleAutLib = 0 then
    Exit;
  @UnRegTlbProc := GetProcAddress(OleAutLib, 'UnRegisterTypeLib');;
  if @UnRegTlbProc = nil then
    Exit;
  WideFilename := ExpandFilename(Filename);
  if FAILED(LoadTypeLib(PWideChar(WideFilename), TypeLib)) then
    Exit;
  if FAILED(TypeLib.GetLibAttr(LibAttr)) then
    Exit;
  try
    with LibAttr^ do
      if FAILED(UnRegTlbProc(Guid, wMajorVerNum, wMinorVerNum, LCID, SysKind)) then
        Exit;
  finally
    TypeLib.ReleaseTLibAttr (LibAttr);
  end;
  Result := True;
end;
{$ENDIF}

function DeleteDirWait (const DirName: String;
  const DirsRemoved, DirsNotRemoved: TSimpleStringList): Boolean;
var
  I, J: Integer;
  FindHandle: THandle;
  FindData: TWin32FindData;
  DidntGetRemoved: Boolean;
begin
  { Sometimes the IE4 shell has shell folders locked, so wait
    up to 5 seconds for previously removed subdirectories to
    *actually* be deleted. }
  for I := 0 to 19 do begin
    DidntGetRemoved := False;
    for J := 0 to DirsRemoved.Count-1 do
      if Copy(DirsRemoved[J], 1, Length(DirName)+1) =
         DirName + '\' then begin
        FindHandle := FindFirstFile(PChar(DirsRemoved[J]), FindData);
        if FindHandle <> INVALID_HANDLE_VALUE then begin
          WinProcs.FindClose (FindHandle);
          if FindData.dwFileAttributes and FILE_ATTRIBUTE_DIRECTORY <> 0 then begin
            DidntGetRemoved := True;
            Break;
          end;
        end;
      end;
    if not DidntGetRemoved then Break;
    Sleep (250);
  end;
  Result := RemoveDirectory(PChar(DirName));
  if Result then
    DirsRemoved.AddIfDoesntExist (DirName)
  else
    if Assigned(DirsNotRemoved) then
      DirsNotRemoved.AddIfDoesntExist (DirName);
end;

function IsSectionEmpty (const Section, Filename: PChar): Boolean;
var
  Test: array[0..255] of Char;
begin
  Test[0] := #0;
  if Filename^ <> #0 then
    GetPrivateProfileString (Section, nil, '', Test, SizeOf(Test), Filename)
  else
    GetProfileString (Section, nil, '', Test, SizeOf(Test));
  Result := Test[0] = #0;
end;

{ TUninstallLog }

constructor TUninstallLog.Create;
begin
  inherited Create;
  Clear;
end;

destructor TUninstallLog.Destroy;
begin
  Clear;
  inherited Destroy;
end;

procedure TUninstallLog.InternalAdd (const Typ: TUninstallRecTyp;
  const Data: Pointer; const Size: Cardinal; const ExtraData: Longint);
{ Adds a new entry to the uninstall list }
var
  NewRec: PUninstallRec;
begin
  NewRec := AllocMem(UninstallRecNonDataSize + Size);
  NewRec^.ExtraData := ExtraData;
  NewRec^.Typ := Typ;
  NewRec^.DataSize := Size;
  Move (Data^, NewRec^.Data, Size);
  if List = nil then begin
    FList := NewRec;
    FLastList := List;
  end
  else begin
    LastList^.Next := NewRec;
    NewRec^.Prev := LastList;
    FLastList := NewRec;
  end;
  Inc (FCount);
end;

procedure TUninstallLog.Add (const Typ: TUninstallRecTyp; const Data: array of String;
  const ExtraData: Longint);
{ Same as AddP, but accepts string type }
var
  I, L: Integer;
  S, X: String;
begin
  for I := 0 to High(Data) do begin
    L := Length(Data[I]);
    if L < $FD then
      S := S + Char(L)
    else if L <= $FFFF then begin
      SetLength (X, SizeOf(Byte) + SizeOf(Word));
      X[1] := #$FD;
      Word((@X[2])^) := Word(L);
      S := S + X;
    end
    else begin
      SetLength (X, SizeOf(Byte) + SizeOf(Integer));
      X[1] := #$FE;
      Integer((@X[2])^) := Integer(L);
      S := S + X;
    end;
    S := S + Data[I];
  end;
  S := S + #$FF;
  InternalAdd (Typ, PChar(S), Length(S), ExtraData);

  if Version < HighestSupportedVersion then
    Version := HighestSupportedVersion;
end;

procedure TUninstallLog.Clear;
{ Frees all entries in the uninstall list and clears AppName/AppDir }
var
  CurRec, NextRec: PUninstallRec;
begin
  CurRec := List;
  while CurRec <> nil do begin
    NextRec := CurRec^.Next;
    FreeMem (CurRec);
    CurRec := NextRec;
  end;
  FList := nil;
  FLastList := nil;
  FCount := 0;
  AppId := '';
  AppName := '';
  Flags := [];
end;

type
  PDeleteDirData = ^TDeleteDirData;
  TDeleteDirData = record
    DirsRemoved, DirsNotRemoved: TSimpleStringList;
  end;

function DeleteDirProc (const DirName: String; Param: Pointer): Boolean; far;
begin
  Result := DeleteDirWait(DirName, PDeleteDirData(Param)^.DirsRemoved,
    PDeleteDirData(Param)^.DirsNotRemoved);
end;

procedure ProcessMessagesProc; far;
var
  Msg: TMsg;
begin
  while PeekMessage(Msg, 0, 0, 0, PM_REMOVE) do begin
    TranslateMessage (Msg);
    DispatchMessage (Msg);
  end;
end;

function TUninstallLog.ExtractRecData (const Rec: PUninstallRec;
  var Data: array of String): Integer;
var
  I, L: Integer;
  X: ^Byte;
begin
  for I := 0 to High(Data) do
    Data[I] := '';
  I := 0;
  L := 0;  { prevent warning }
  X := @Rec^.Data;
  while I <= High(Data) do begin
    case X^ of
      $00..$FC: begin
           L := X^;
           Inc (X);
         end;
      $FD: begin
           Inc (X);
           L := Word(X^);
           Inc (X, SizeOf(Word));
         end;
      $FE: begin
           Inc (X);
           L := Integer(X^);
           Inc (X, SizeOf(Integer));
         end;
      $FF: Break;
    end;
    SetString (Data[I], PChar(X), L);
    Inc (X, L);
    Inc (I);
  end;
  Result := I;
end;

function TUninstallLog.CheckMutexes: Boolean;
var
  CurRec: PUninstallRec;
  Data: String;
begin
  Result := False;
  CurRec := LastList;
  while CurRec <> nil do begin
    ExtractRecData (CurRec, Data);
    if CurRec^.Typ = utMutexCheck then
      if CheckForMutexes(Data) then begin
        Result := True;
        Exit;
      end;
    CurRec := CurRec^.Prev;
  end;
end;

function TUninstallLog.PerformUninstall (const CallFromUninstaller: Boolean;
  const DeleteUninstallDataFilesProc: TDeleteUninstallDataFilesProc): Boolean;
{ Undoes all the changes in the uninstall list, in reverse order they were
  added. Deletes entries that were successfully undone.
  Returns True if all elements were successfully removed; False if some
  could not be removed. }

var
  ChangeNotifyList, RunOnceList: TSimpleStringList;
  DeleteDirData: TDeleteDirData;

  function FileDelete (const Filename: String; const NotifyChange: Boolean): Boolean;
  begin
    Result := True;
    if CompareText(ExtractFileExt(Filename), '.HLP') = 0 then begin
      DeleteFile (ChangeFileExt(Filename, '.GID'));
      DeleteFile (ChangeFileExt(Filename, '.FTS'));
    end;
    if FileExists(Filename) then begin
      if not DeleteFile(Filename) then
        Result := False
      else
        if NotifyChange then begin
          SHChangeNotify (SHCNE_DELETE, SHCNF_PATH, PChar(Filename), nil);
          ChangeNotifyList.AddIfDoesntExist (RemoveBackslashUnlessRoot(ExtractFilePath(Filename)));
        end;
    end;
  end;

const
  GroupInfoChars: array[0..3] of Char = ('"', '"', ',', ',');
  NullChar: Char = #0;
var
  StartCount: Integer;
  CurRec, NextRec, CurRunRec: PUninstallRec;
  CurRecDataPChar: array[0..9] of PChar;
  CurRecData: array[0..9] of String;
  FN: String;
  P: Integer;
  K: HKEY;
  NumSubkeys, NumValues: DWORD;
  ShowCmd: Integer;

  PROGMAN, Item: TDDEString;
  Conv: TDDEConversation;
  ItemName: String;
  DataHandle: TDDEData;
  DataPointer: PChar;
  DataSize: Longint;
  NoItemsLeft: Boolean;

  procedure SplitData (const Rec: PUninstallRec);
  var
    C, I: Integer;
  begin
    C := ExtractRecData(Rec, CurRecData);
    for I := 0 to 9 do begin
      if I < C then
        CurRecDataPChar[I] := PChar(CurRecData[I])
      else
        CurRecDataPChar[I] := nil;
    end;
  end;

begin
  Result := True;

  PROGMAN := 0;
  Conv := 0;
  RunOnceList := nil;
  DeleteDirData.DirsRemoved := nil;
  DeleteDirData.DirsNotRemoved := nil;
  ChangeNotifyList := TSimpleStringList.Create;
  try
    RunOnceList := TSimpleStringList.Create;
    DeleteDirData.DirsRemoved := TSimpleStringList.Create;
    if Assigned(DeleteUninstallDataFilesProc) then
      DeleteDirData.DirsNotRemoved := TSimpleStringList.Create;

    StartCount := FCount;
    CurRec := LastList;
    while CurRec <> nil do begin
      StatusUpdate (StartCount, FCount);
      SplitData (CurRec);

      try
        case CurRec^.Typ of
          utUserDefined: begin
              {if CurRecData[0] = 'MyRecordType' then begin
                 ... your code here ...
              end
              else}
                raise Exception.Create(FmtSetupMessage1(msgUninstallUnknownEntry,
                  'utUserDefined:' + CurRecData[0]));
            end;
          utStartInstall,
          utEndInstall: { do nothing on these };
          utRun:
            { Process *all* utRun entries in one swipe when it encounters the
              first utRun entry. This is new in 1.3.9; previously it executed
              them in the order they appeared in the log, but this didn't work
              on apps that were installed multiple times, since it was possible
              for the program to be deleted before it was executed. }
            if CallFromUninstaller then begin
              { Scan through all log entries, starting with the current one }
              CurRunRec := CurRec;
              repeat
                if CurRunRec^.Typ = utRun then begin
                  CurRunRec^.Typ := utRunCompleted;  { mark it as 'done' }
                  SplitData (CurRunRec);
                   { Verify that a utRun entry with the same RunOnceId has not
                     already been executed }
                  if ((CurRecData[3] = '') or (RunOnceList.IndexOf(CurRecData[3]) = -1)) then begin
                    ShowCmd := SW_SHOWNORMAL;
                    if CurRunRec^.ExtraData and utRun_RunMinimized <> 0 then
                      ShowCmd := SW_SHOWMINNOACTIVE
                    else if CurRunRec^.ExtraData and utRun_RunMaximized <> 0 then
                      ShowCmd := SW_SHOWMAXIMIZED;
                    if CurRunRec^.ExtraData and utRun_ShellExec = 0 then begin
                      if not InstExec(CurRecData[0], CurRecData[1], CurRecData[2],
                         CurRunRec^.ExtraData and utRun_NoWait = 0,
                         CurRunRec^.ExtraData and utRun_WaitUntilIdle <> 0,
                         ShowCmd, ProcessMessagesProc, P) then
                        Result := False;
                    end
                    else begin
                      if not InstShellExec(CurRecData[0], CurRecData[1], CurRecData[2],
                         ShowCmd, P) then
                        Result := False;
                    end;
                    if CurRecData[3] <> '' then
                      RunOnceList.Add (CurRecData[3]);
                  end;
                end;
                CurRunRec := CurRunRec^.Prev;
              until CurRunRec = nil;
            end;
          utDeleteDirOrFiles:
            if (CallFromUninstaller or (CurRec^.ExtraData and utDeleteDirOrFiles_Extra = 0)) then begin
              { Since NT cannot remove the current directory of a drive,
                and there is a chance that the application directory
                could be the current directory, change to the Windows
                directory to prevent failure. }
              SetCurrentDirectory (PChar(GetWinDir));
              if DelTree(CurRecData[0], CurRec^.ExtraData and utDeleteDirOrFiles_IsDir <> 0,
                 CurRec^.ExtraData and utDeleteDirOrFiles_DeleteFiles <> 0,
                 CurRec^.ExtraData and utDeleteDirOrFiles_DeleteSubdirsAlso <> 0,
                 DeleteDirProc, @DeleteDirData) then begin
                if (CurRec^.ExtraData and utDeleteDirOrFiles_IsDir <> 0) and
                   (CurRec^.ExtraData and utDeleteDirOrFiles_CallChangeNotify <> 0) then begin
                  SHChangeNotify (SHCNE_RMDIR, SHCNF_PATH, CurRecDataPChar[0], nil);
                  ChangeNotifyList.AddIfDoesntExist (RemoveBackslashUnlessRoot(ExtractFilePath(CurRecData[0])));
                end;
              end;
            end;
          utDeleteFile: begin
              FN := CurRecData[1];
              if CallFromUninstaller or (FN = '') then
                FN := CurRecData[0];
              if CallFromUninstaller or (CurRec^.ExtraData and utDeleteFile_ExistedBeforeInstall = 0) then begin
                if (CurRec^.ExtraData and utDeleteFile_SharedFile = 0) or
                   (DecrementSharedCount(CurRecData[0]) and FileExists(FN) and
                    ((FN <> CurRecData[0]) or ShouldRemoveSharedFile(CurRecData[0]))) then begin
                  if (CurRec^.ExtraData and utDeleteFile_RegisteredServer <> 0) and
                     (FN = CurRecData[0]) then
                    UnregisterServer (CurRecData[0]);
                  if (CurRec^.ExtraData and utDeleteFile_RegisteredTypeLib <> 0) and
                     (FN = CurRecData[0]) then
                    UnregisterTypeLib (CurRecData[0]);
                  if CurRec^.ExtraData and utDeleteFile_IsFont <> 0 then begin
                    if Win32Platform <> VER_PLATFORM_WIN32_WINDOWS then begin
                      WriteProfileString ('Fonts', CurRecDataPChar[2], nil);
                    end
                    else begin
                      if RegOpenKeyEx(HKEY_LOCAL_MACHINE, NEWREGSTR_PATH_SETUP + '\Fonts',
                         0, KEY_SET_VALUE, K) <> ERROR_SUCCESS then
                        Result := False
                      else begin
                        RegDeleteValue (K, CurRecDataPChar[2]);
                        RegCloseKey (K);
                      end;
                    end;
                    if RemoveFontResource(CurRecDataPChar[3]) then
                      SendNotifyMessage (HWND_BROADCAST, WM_FONTCHANGE, 0, 0);
                  end;
                  if CallFromUninstaller or (CurRec^.ExtraData and utDeleteFile_Extra = 0) then
                    if not FileDelete(FN, CurRec^.ExtraData and utDeleteFile_CallChangeNotify <> 0) then
                      Result := False;
                end;
              end
              else begin
                if CurRec^.ExtraData and utDeleteFile_SharedFile <> 0 then
                  DecrementSharedCount (CurRecData[0]);
                if FN <> CurRecData[0] then
                  if not FileDelete(FN, CurRec^.ExtraData and utDeleteFile_CallChangeNotify <> 0) then
                    Result := False;
              end;
            end;
          utDeleteGroupOrItem: begin
              try
                if PROGMAN = 0 then PROGMAN := DDE.CreateString('PROGMAN'); {don't localize}
                if Conv = 0 then Conv := DDE.BeginConnection(PROGMAN, PROGMAN);
                ItemName := CurRecData[1];
                if ItemName <> '' then
                  { Delete just the item }
                  {don't localize}
                  DDE.Execute (Conv, '[ShowGroup("' + CurRecData[0] + '",5)]' +
                    '[DeleteItem("' + ItemName + '")]', True)
                else begin
                  { Delete the group if it's empty }
                  Item := DDE.CreateString(CurRecData[0]);
                  try
                    DDE.RequestBegin (Conv, Item, CF_TEXT, DataHandle, Pointer(DataPointer), DataSize);
                    try
                      for P := 0 to 3 do begin
                        DataPointer := StrScan(DataPointer, GroupInfoChars[P]);
                        if DataPointer = nil then Abort; {exception is trapped below}
                        Inc (DataPointer);
                      end;
                      NoItemsLeft := DataPointer[0] = '0';
                    finally
                      DDE.RequestEnd (DataHandle);
                    end;
                  finally
                    DDE.FreeString (Item);
                  end;
                  if NoItemsLeft then
                    {don't localize}
                    DDE.Execute (Conv, '[DeleteGroup("' + CurRecData[0] + '")]', True);
                end;
              except
              end;
            end;
          utIniDeleteEntry: begin
              if CurRecDataPChar[0] <> #0 then
                WritePrivateProfileString (CurRecDataPChar[1], CurRecDataPChar[2],
                  nil, CurRecDataPChar[0])
              else
                WriteProfileString (CurRecDataPChar[1], CurRecDataPChar[2], nil);
            end;
          utIniDeleteSection: begin
              if (CurRec^.ExtraData and utIniDeleteSection_OnlyIfEmpty = 0) or
                 IsSectionEmpty(CurRecDataPChar[1], CurRecDataPChar[0]) then begin
                if CurRecDataPChar[0] <> #0 then
                  WritePrivateProfileString (CurRecDataPChar[1], nil, nil,
                    CurRecDataPChar[0])
                else
                  WriteProfileString (CurRecDataPChar[1], nil, nil);
              end;
            end;
          utRegDeleteEntireKey:
            if RegOpenKeyEx(CurRec^.ExtraData, CurRecDataPChar[0], 0, KEY_QUERY_VALUE, K) = ERROR_SUCCESS then begin
              RegCloseKey (K);
              if not RegDeleteKeyIncludingSubkeys(CurRec^.ExtraData, CurRecDataPChar[0]) then
                Result := False;
            end;
          utRegClearValue:
            if RegOpenKeyEx(CurRec^.ExtraData, CurRecDataPChar[0], 0, KEY_SET_VALUE, K) = ERROR_SUCCESS then begin
              if RegSetValueEx(K, CurRecDataPChar[1], 0, REG_SZ, @NullChar,
                 SizeOf(NullChar)) <> ERROR_SUCCESS then
                Result := False;
              RegCloseKey (K);
            end;
          utRegDeleteKeyIfEmpty: begin
              if RegOpenKeyEx(CurRec^.ExtraData, CurRecDataPChar[0], 0,
                 KEY_QUERY_VALUE, K) = ERROR_SUCCESS then begin
                if RegQueryInfoKey(K, nil, nil, nil, @NumSubkeys, nil, nil,
                   @NumValues, nil, nil, nil, nil) <> ERROR_SUCCESS then begin
                  RegCloseKey (K);
                  Result := False;
                end
                else begin
                  RegCloseKey (K);
                  if (NumSubkeys = 0) and (NumValues = 0) then
                    if not RegDeleteKeyIncludingSubkeys(CurRec^.ExtraData, CurRecDataPChar[0]) then
                      Result := False;
                end;
              end;
            end;
          utRegDeleteValue: begin
              if RegOpenKeyEx(CurRec^.ExtraData, CurRecDataPChar[0], 0,
                 KEY_QUERY_VALUE or KEY_ENUMERATE_SUB_KEYS or KEY_SET_VALUE,
                 K) = ERROR_SUCCESS then begin
                if RegValueExists(K, CurRecDataPChar[1]) and
                   (RegDeleteValue(K, CurRecDataPChar[1]) <> ERROR_SUCCESS) then
                  Result := False;
              end;
            end;
          utDecrementSharedCount:
            DecrementSharedCount (CurRecData[0]);
          utRefreshFileAssoc:
            SHChangeNotify (SHCNE_ASSOCCHANGED, SHCNF_IDLIST, nil, nil);
          utMutexCheck: ;    { do nothing; utMutexChecks aren't processed here }
          utRunCompleted: ;  { do nothing; it's a dummy type }
        else
          raise Exception.Create(FmtSetupMessage1(msgUninstallUnknownEntry,
            Format('$%x', [CurRec^.Typ])));
        end;
      except
        on E: Exception do begin
          Result := False;
          if not(E is EAbort) then
            ShowException (E);
        end;
      end;

      Dec (FCount);
      NextRec := CurRec^.Prev;
      FreeMem (CurRec);
      CurRec := NextRec;
      FLastList := CurRec;
    end;
    FList := nil;
    StatusUpdate (StartCount, FCount);

    if Assigned(DeleteUninstallDataFilesProc) then begin
      DeleteUninstallDataFilesProc;
      { Now that uninstall data is deleted, try removing the directories it
        was in that couldn't be deleted before. }
      for P := 0 to DeleteDirData.DirsNotRemoved.Count-1 do
        DeleteDirWait (DeleteDirData.DirsNotRemoved[P],
          DeleteDirData.DirsRemoved, nil);
    end;
  finally
    DeleteDirData.DirsNotRemoved.Free;
    DeleteDirData.DirsRemoved.Free;
    for P := 0 to ChangeNotifyList.Count-1 do
      if DirExists(ChangeNotifyList[P]) then
        SHChangeNotify (SHCNE_UPDATEDIR, SHCNF_PATH or SHCNF_FLUSH,
          PChar(ChangeNotifyList[P]), nil);
    RunOnceList.Free;
    ChangeNotifyList.Free;
    if Conv <> 0 then DDE.EndConnection (Conv);
    if PROGMAN <> 0 then DDE.FreeString (PROGMAN);
  end;
end;

function TUninstallLog.ShouldRemoveSharedFile (const Filename: String): Boolean;
begin
  Result := True;
end;

procedure TUninstallLog.ShowException (E: Exception);
begin
  MessageBox (0, PChar(E.Message + '.'), PChar(SetupMessages[msgUninstallAppTitle]),
    MB_OK or MB_ICONSTOP);
end;

procedure TUninstallLog.StatusUpdate (StartingCount, CurCount: Integer);
begin
end;

procedure TUninstallLog.Save (const Filename: String;
  const Append, UpdateUninstallLogAppName: Boolean);
{ Saves all undo data to Filename. If Append is True, it appends the current
  undo data to the end of the existing file. When Append is True, it assumes
  compatibility has already been verified with the Test method. }
var
  F: File;
  Buffer: array[0..4095] of Byte;
  BufCount: Cardinal;

  procedure Flush;
  var
    CrcHeader: TUninstallCrcHeader;
  begin
    if BufCount <> 0 then begin
      CrcHeader.Size := BufCount;
      CrcHeader.NotSize := not CrcHeader.Size;
      CrcHeader.CRC := GetCRC32(Buffer, BufCount);
      BlockWrite (F, CrcHeader, SizeOf(CrcHeader));
      BlockWrite (F, Buffer, BufCount);
      BufCount := 0;
    end;
  end;

  procedure WriteBuf (const Buf; Size: Cardinal);
  var
    P: Pointer;
    S: Cardinal;
  begin
    P := @Buf;
    while Size <> 0 do begin
      S := Size;
      if S > SizeOf(Buffer) - BufCount then
        S := SizeOf(Buffer) - BufCount;
      Move (P^, Buffer[BufCount], S);
      Inc (BufCount, S);
      if BufCount = SizeOf(Buffer) then
        Flush;
      Inc (Cardinal(P), S);
      Dec (Size, S);
    end;
  end;

var
  Header: TUninstallLogHeader;
  FileRec: TUninstallFileRec;
  CurRec: PUninstallRec;
begin
  BufCount := 0;
  AssignFile (F, Filename);
  FileMode := fmOpenReadWrite or fmShareExclusive;
  if not Append then
    Rewrite (F, 1)
  else
    Reset (F, 1);
  try
    if not Append then begin
      FillChar (Header, SizeOf(Header), 0);
      BlockWrite (F, Header, SizeOf(Header));
      {goes back and fills in correct values later}
    end
    else begin
      BlockRead (F, Header, SizeOf(Header));
      Seek (F, Header.EndOffset);
    end;

    CurRec := List;
    while CurRec <> nil do begin
      FileRec.Typ := Ord(CurRec^.Typ);
      FileRec.ExtraData := CurRec^.ExtraData;
      FileRec.DataSize := CurRec^.DataSize;
      WriteBuf (FileRec, SizeOf(FileRec));
      WriteBuf (CurRec^.Data, CurRec^.DataSize);
      Inc (Header.NumRecs);

      CurRec := CurRec^.Next;
    end;
    Flush;

    Header.EndOffset := FilePos(F);
    Seek (F, 0);
    Header.ID := UninstallLogID;
    StrPLCopy (Header.AppId, AppId, SizeOf(Header.AppId)-1);
    if not Append or UpdateUninstallLogAppName then
      StrPLCopy (Header.AppName, AppName, SizeOf(Header.AppName)-1);
    if Version > Header.Version then
      Header.Version := Version;
    TUninstallLogFlags((@Header.Flags)^) := TUninstallLogFlags((@Header.Flags)^) + Flags;
    Header.CRC := GetCRC32(Header, SizeOf(Header)-SizeOf(Longint));
    BlockWrite (F, Header, SizeOf(Header));
  finally
    CloseFile (F);
  end;
end;

procedure TUninstallLog.Load (const Filename: String);
{ Loads all undo data from Filename }
var
  F: File;
  Buffer: array[0..4095] of Byte;
  BufPos, BufLeft: Cardinal;
  Header: TUninstallLogHeader;

  procedure Corrupt;
  begin
    raise Exception.Create(FmtSetupMessage1(msgUninstallDataCorrupted, Filename));
  end;

  procedure FillBuffer;
  var
    CrcHeader: TUninstallCrcHeader;
  begin
    while BufLeft = 0 do begin
      if Cardinal(FilePos(F)) + SizeOf(CrcHeader) > Header.EndOffset then
        Corrupt;
      BlockRead (F, CrcHeader, SizeOf(CrcHeader));
      if (CrcHeader.Size <> not CrcHeader.NotSize) or
         (CrcHeader.Size > SizeOf(Buffer)) or
         (Cardinal(FilePos(F)) + CrcHeader.Size > Header.EndOffset) then
        Corrupt;
      BlockRead (F, Buffer, CrcHeader.Size);
      if not(ufDontCheckRecCRCs in Flags) and
        (CrcHeader.CRC <> GetCRC32(Buffer, CrcHeader.Size)) then
        Corrupt;
      BufPos := 0;
      BufLeft := CrcHeader.Size;
    end;
  end;

  procedure ReadBuf (var Buf; Size: Cardinal);
  var
    P: Pointer;
    S: Cardinal;
  begin
    P := @Buf;
    while Size <> 0 do begin
      if BufLeft = 0 then
        FillBuffer;
      S := Size;
      if S > BufLeft then
        S := BufLeft;
      Move (Buffer[BufPos], P^, S);
      Inc (BufPos, S);
      Dec (BufLeft, S);
      Inc (Cardinal(P), S);
      Dec (Size, S);
    end;
  end;

var
  FileRec: TUninstallFileRec;
  I: Integer;
  P, P2: Pointer;
begin
  BufPos := 0;
  BufLeft := 0;
  P := nil;

  AssignFile (F, Filename);
  FileMode := fmOpenRead or fmShareDenyWrite;
  Reset (F, 1);
  try
    if FileSize(F) < SizeOf(Header) then
      Corrupt;
    BlockRead (F, Header, SizeOf(Header));
    if ((Header.CRC <> $11111111) and
        { ^ for debugging purposes, you can change the CRC field in the file to
          $11111111 to disable CRC checking on the header}
        (Header.CRC <> GetCRC32(Header, SizeOf(Header)-SizeOf(Longint)))) or
       (Header.ID <> UninstallLogID) then
      Corrupt;
    if Header.Version > HighestSupportedVersion then
      raise Exception.Create(FmtSetupMessage1(msgUninstallUnsupportedVer, Filename));
    AppId := StrPas(Header.AppId);
    AppName := StrPas(Header.AppName);
    Flags := TUninstallLogFlags((@Header.Flags)^);

    for I := 1 to Header.NumRecs do begin
      ReadBuf (FileRec, SizeOf(FileRec));
      GetMem (P, FileRec.DataSize);
      { if ReadBuf raises an exception, P will be freed in the 'finally' section }
      ReadBuf (P^, FileRec.DataSize);
      P2 := P;
      P := nil;  { clear P so the 'finally' section won't free it }
      InternalAdd (TUninstallRecTyp(FileRec.Typ), P2, FileRec.DataSize, FileRec.ExtraData);
    end;
  finally
    CloseFile (F);
    FreeMem (P);
  end;
end;

function TUninstallLog.Test (const Filename, AAppId: String): Boolean;
{ Returns True if Filename is a recognized uninstall log format, and its
  AppId header field matches the AppId parameter }
var
  F: File;
  Header: TUninstallLogHeader;
begin
  Result := False;
  try
    AssignFile (F, Filename);
    FileMode := fmOpenRead or fmShareDenyWrite;
    Reset (F, 1);
    try
      if FileSize(F) < SizeOf(Header) then
        Exit;
      BlockRead (F, Header, SizeOf(Header));
      if ((Header.CRC <> $11111111) and
          { ^ for debugging purposes, you can change the CRC field in the file to
            $11111111 to disable CRC checking on the header}
          (Header.CRC <> GetCRC32(Header, SizeOf(Header)-SizeOf(Longint)))) or
         (Header.ID <> UninstallLogID) or
         (StrComp(Header.AppId, PChar(AAppId)) <> 0) then
        Exit;
      Result := True;
    finally
      CloseFile (F);
    end;
  except
  end;
end;

procedure InitOle;
var
  OleResult: HRESULT;
begin
  OleResult := CoInitialize(nil);
  if FAILED(OleResult) then
    raise Exception.CreateFmt('CoInitialize failed (0x%.8x)', [OleResult]);
    { ^ doesn't use a SetupMessage since messages probably aren't loaded
      during 'initialization' section below, which calls this procedure }
end;

initialization
  InitOle;
finalization
  CoUninitialize;
end.
