program SetupLdr;

{
  Inno Setup
  Copyright (C) 1998-2000 Jordan Russell
  For conditions of distribution and use, see LICENSE.TXT.

  Setup Loader
}

uses
  WinProcs,
  WinTypes,
  Messages,
  SysUtils,
  zlib,
  SetupEnt in 'SETUPENT.PAS',
  CmnFunc2 in 'CMNFUNC2.PAS',
  Msgs in 'MSGS.PAS',
  MsgIDs in 'MSGIDS.PAS',
  Struct in 'STRUCT.PAS',
  InstFunc in 'InstFunc.pas';

{$R *.RES}

procedure RaiseLastError (const Msg: TSetupMessageID);
var
  ErrorCode: DWORD;
begin
  ErrorCode := GetLastError;
  raise Exception.Create(FmtSetupMessage(msgLastErrorMessage,
    [SetupMessages[Msg], IntToStr(ErrorCode), SysErrorMessage(ErrorCode)]));
end;

procedure ShowExceptionMsg (const Msg: String);
var
  M: String;
begin
  M := Msg;
  if (M <> '') and (M[Length(M)] > '.') then M := M + '.';
  MessageBox (0, PChar(M), Pointer(SetupMessages[msgErrorTitle]),
    MB_OK or MB_ICONSTOP);
    { ^ use a Pointer cast instead of a PChar cast so that it will use "nil"
      if SetupMessages[msgErrorTitle] is empty due to the messages not being
      loaded yet. MessageBox displays 'Error' as the caption if the lpCaption
      parameter is nil. }
end;

var
  SetupLdrWnd: HWND = 0;
  OrigWndProc: Pointer;
  RestartSystem: Boolean = False;

function SetupLdrWndProc (Wnd: HWND; Msg: UINT; WParam: WPARAM; LParam: LPARAM): LRESULT;
stdcall;
begin
  Result := 0;
  case Msg of
    WM_QUERYENDSESSION: begin
        { Return zero so that a shutdown attempt can't kill SetupLdr }
      end;
    WM_USER + 150: begin
        if WParam = 10000 then begin
          RestartSystem := True;
          Result := 1;
        end;
      end;
  else
    Result := CallWindowProc(OrigWndProc, Wnd, Msg, WParam, LParam);
  end;
end;

procedure ExecAndWait (const Filename, Parms: String);
var
  CmdLine: String;
  StartupInfo: TStartupInfo;
  ProcessInfo: TProcessInformation;
  Msg: TMsg;
begin
  CmdLine := AddQuotes(Filename) + ' ' + Parms;

  FillChar (StartupInfo, SizeOf(StartupInfo), 0);
  StartupInfo.cb := SizeOf(StartupInfo);
  if not CreateProcess(nil, PChar(CmdLine), nil, nil, False, 0, nil, nil,
     StartupInfo, ProcessInfo) then
    RaiseLastError (msgLdrCannotExecTemp);
  with ProcessInfo do begin
    { Don't need the thread handle, so close it now }
    CloseHandle (hThread);
    { Wait until the process returns. Uses MsgWaitForMultipleObjects
      because it has to check the message queue so the "feedback"
      cursor doesn't stay on. }
    repeat
      while PeekMessage(Msg, 0, 0, 0, PM_REMOVE) do begin
        TranslateMessage (Msg);
        DispatchMessage (Msg);
      end;
    until MsgWaitForMultipleObjects(1, hProcess, False, INFINITE,
      QS_ALLINPUT) <> WAIT_OBJECT_0+1;
    { Then close the process handle }
    CloseHandle (hProcess);
  end;
end;

function SwitchPassed (const Name: String): Boolean;
var
  I: Integer;
begin
  for I := 1 to NewParamCount do
    if CompareText(NewParamStr(I), '/' + Name) = 0 then begin
      Result := True;
      Exit;
    end;
  Result := False;
end;

procedure SetupCorruptError;
begin
  if SetupMessages[msgSetupFileCorrupt] <> '' then
    raise Exception.Create(SetupMessages[msgSetupFileCorrupt])
  else
    { In case the messages haven't been loaded yet, use the constant }
    raise Exception.Create(SSetupFileCorrupt);
end;

const
  SelfReadMode = fmOpenRead or fmShareDenyWrite;
var
  SelfFilename: String;
  SourceF, DestF: File;
  SourceFOpened: Boolean = False;
  SizeOfFile: Longint;
  ExeHeader: TSetupLdrExeHeader;
  OffsetTable: TSetupLdrOffsetTable;
  TempFileP: array[0..MAX_PATH-1] of Char;
  TempFile: String;
  TestID: TSetupID;
  SetupHeader: TSetupHeader;
  Adler: Longint;
  Data: PDeflateBlockReadData;
  BeganInflating: Boolean;
begin
  try
    SelfFilename := NewParamStr(0);
    AssignFile (SourceF, SelfFilename);
    FileMode := SelfReadMode;  Reset (SourceF, 1);
    try
      SourceFOpened := True;

      SizeOfFile := FileSize(SourceF);
      Seek (SourceF, SetupLdrExeHeaderOffset);
      BlockRead (SourceF, ExeHeader, SizeOf(ExeHeader));
      if (ExeHeader.ID <> SetupLdrExeHeaderID) or
         (ExeHeader.OffsetTableOffset <> not ExeHeader.NotOffsetTableOffset) or
         (ExeHeader.OffsetTableOffset + SizeOf(OffsetTable) > SizeOfFile) then
        SetupCorruptError;
      Seek (SourceF, ExeHeader.OffsetTableOffset);
      BlockRead (SourceF, OffsetTable, SizeOf(OffsetTable));
      if (OffsetTable.ID <> SetupLdrOffsetTableID) or
         (SizeOfFile < OffsetTable.TotalSize) then
        SetupCorruptError;

      LoadCompressedSetupMessages (SelfFilename, OffsetTable.OffsetMsg,
        SelfReadMode);

      if Win32Platform = VER_PLATFORM_WIN32s then begin
        MessageBox (0, PChar(FmtSetupMessage1(msgNotOnThisPlatform, 'Win32s')),
          PChar(SetupMessages[msgErrorTitle]), MB_OK or MB_ICONSTOP);
        Exit;
      end;

      Seek (SourceF, OffsetTable.Offset0);
      BlockRead (SourceF, TestID, SizeOf(TestID));
      if TestID <> SetupID then
        SetupCorruptError;
      try
        BeganInflating := False;
        New (Data);
        try
          InflateBlockReadBegin (SourceF, Data^);
          BeganInflating := True;
          SEInflateBlockRead (Data^, SetupHeader, SizeOf(SetupHeader),
            SetupHeaderStrings);
        finally
          if BeganInflating then
            InflateBlockReadEnd (Data^);
          Dispose (Data);
        end;
      except
        on EZlibDataError do
          SetupCorruptError;
      end;
      if not(shDisableStartupPrompt in SetupHeader.Options) and
         not SwitchPassed('SP-') and
         (MessageBox(0, PChar(FmtSetupMessage1(msgSetupLdrStartupMessage, SetupHeader.AppName)),
           PChar(SetupMessages[msgSetupAppTitle]), MB_YESNO or MB_ICONQUESTION) <> IDYES) then
        Exit;

      { Extract the embedded setup program to the temp directory }
      if GetTempFileName(PChar(GetTempDir), 'INS', 0, TempFileP) = 0 then
        RaiseLastError (msgLdrCannotCreateTemp);
      TempFile := StrPas(TempFileP);

      Seek (SourceF, OffsetTable.OffsetEXE);

      AssignFile (DestF, TempFile);
      FileMode := fmOpenWrite or fmShareExclusive;  Rewrite (DestF, 1);
      try
        try
          if not InflateData(SourceF, OffsetTable.CompressedSizeEXE, True,
             DestF, OffsetTable.UncompressedSizeEXE, True, Adler) then
            SetupCorruptError;
        except
          on EZlibDataError do
            SetupCorruptError;
        end;
        if Adler <> OffsetTable.AdlerEXE then
          SetupCorruptError;
      finally
        CloseFile (DestF);
      end;

      CloseFile (SourceF);
      SourceFOpened := False;

      { Create SetupLdrWnd, which is used by Setup to communicate with
        SetupLdr }
      SetupLdrWnd := CreateWindowEx(0, 'STATIC', 'InnoSetupLdrWindow', 0,
        0, 0, 0, 0, HWND_DESKTOP, 0, HInstance, nil);
      Longint(OrigWndProc) := SetWindowLong(SetupLdrWnd, GWL_WNDPROC,
        Longint(@SetupLdrWndProc));

      { Now we execute it }
      ExecAndWait (TempFile, Format('/SL3 $%x %s %d %d %d %s',
        [SetupLdrWnd, AddQuotes(SelfFilename), OffsetTable.OffsetMsg,
         OffsetTable.Offset0, OffsetTable.Offset1, GetCmdTail]));
    finally
      if SourceFOpened then
        CloseFile (SourceF);
      if TempFile <> '' then
        { Even though Setup has terminated by now, the system may still have
          the file locked for a short period of time (esp. on multiprocessor
          systems), so use DelayDeleteFile to delete it. }
        DelayDeleteFile (TempFile, 12);
      if SetupLdrWnd <> 0 then
        { SetupLdrWnd must be destroyed before RestartComputer is called,
          otherwise SetupLdrWndProc would deny the shutdown request }
        DestroyWindow (SetupLdrWnd);
    end;
    if RestartSystem then
      RestartComputer;
  except
    on E: Exception do
      ShowExceptionMsg (E.Message);
  end;
end.
