unit DDEInt;

{
  Inno Setup
  Copyright (C) 1998-99 Jordan Russell
  For conditions of distribution and use, see LICENSE.TXT.

  Code for TDDE class, which is a simple encapsulation of a few of the DDEML
  (Dynamic Data Exchange Management Library) functions.

  Note: This unit is shared by both the 'Setup' and 'Uninst' projects.
}

interface

uses
  WinProcs, WinTypes, SysUtils, DDEML;

type
  TDDEString = HSZ;
  TDDEConversation = HConv;
  TDDEData = HDdeData;

  TDDE = class(TObject)
  private
    Instance: Longint;
  public
    constructor Create;
    destructor Destroy; override;
    function CreateString (const S: String): TDDEString;
    procedure FreeString (var S: TDDEString);
    function BeginConnection (const Service, Topic: TDDEString): TDDEConversation;
    procedure EndConnection (var Conv: TDDEConversation);
    function Execute (const Conv: TDDEConversation; const Command: String;
      const AllowNotProcessed: Boolean): Boolean;
    procedure RequestBegin (const Conv: TDDEConversation; const Item: TDDEString;
      const Format: Word; var DataHandle: TDDEData; var DataPointer: Pointer;
      var DataSize: Longint);
    procedure RequestEnd (const DataHandle: TDDEData);
  end;

function DDE: TDDE;

implementation

uses
  CmnFunc2, Msgs, MsgIDs;

var
  ADDE: TDDE;

function DDE: TDDE;
begin
  if ADDE = nil then
    ADDE := TDDE.Create;
  Result := ADDE;
end;

function DDECallbackProc (uType, uFmt: UINT; hConv: HConv; hsz1, hsz2: HSZ;
  hData: HDDEData; dwData1, dwData2: DWORD): HDDEData; stdcall; 
begin
  Result := 0;
end;

procedure DDEError (const Func: String; const Code: UINT);
begin
  raise Exception.Create(FmtSetupMessage(msgErrorFunctionFailed, [Func,
    IntToStr(Code)]));
end;

constructor TDDE.Create;
var
  Code: UINT;
begin
  inherited Create;
  Code := DdeInitialize(Instance, DDECallbackProc, APPCMD_CLIENTONLY, 0);
  if Code <> DMLERR_NO_ERROR then begin
    Instance := 0;
    DDEError ('DdeInitialize', Code);
  end;
end;

destructor TDDE.Destroy;
begin
  if Instance <> 0 then begin
    DdeUninitialize (Instance);
    Instance := 0;
  end;
  inherited Destroy;
end;

function TDDE.CreateString (const S: String): TDDEString;
begin
  Result := DdeCreateStringHandle(Instance, PChar(S), 0);
  if Result = 0 then
    DDEError ('DdeCreateStringHandle', DdeGetLastError(Instance));
end;

procedure TDDE.FreeString (var S: TDDEString);
begin
  DdeFreeStringHandle (Instance, S);
  S := 0;
end;

function TDDE.BeginConnection (const Service, Topic: TDDEString): TDDEConversation;
begin
  Result := DdeConnect(Instance, Service, Topic, nil);
  if Result = 0 then
    DDEError ('DdeConnect', DdeGetLastError(Instance));
end;

procedure TDDE.EndConnection (var Conv: TDDEConversation);
begin
  DdeDisconnect (Conv);
  Conv := 0;
end;

function TDDE.Execute (const Conv: TDDEConversation; const Command: String;
  const AllowNotProcessed: Boolean): Boolean;
{ Returns True if command was successful, False if failed. An exception is
  also raised if AllowNotProcessed is False }
var
  LastError: UINT;
begin
  DdeClientTransaction (PChar(Command), Length(Command)+1, Conv, 0, 0,
    XTYP_EXECUTE, 7500, nil);
  LastError := DdeGetLastError(Instance);
  case LastError of
    DMLERR_NO_ERROR: ;
    DMLERR_NOTPROCESSED:
      if not AllowNotProcessed then
        raise Exception.Create(SetupMessages[msgErrorDDECommandFailed])
      else begin
        Result := False;
        Exit;
      end;
  else
    raise Exception.Create(FmtSetupMessage1(msgErrorDDEExecute,
      IntToStr(LastError)));
  end;
  Result := True;
end;

procedure TDDE.RequestBegin (const Conv: TDDEConversation; const Item: TDDEString;
  const Format: Word; var DataHandle: TDDEData; var DataPointer: Pointer;
  var DataSize: Longint);
begin
  DataHandle := DdeClientTransaction(nil, 0, Conv, Item, Format, XTYP_REQUEST, 7500, nil);
  if DataHandle = 0 then
    raise Exception.Create(FmtSetupMessage1(msgErrorDDERequest,
      IntToStr(DdeGetLastError(Instance))));
  DataPointer := DdeAccessData(DataHandle, @DataSize);
  if DataPointer = nil then
    DDEError ('DdeAccessData', DdeGetLastError(Instance));
end;

procedure TDDE.RequestEnd (const DataHandle: TDDEData);
begin
  DdeUnaccessData (DataHandle);
end;


initialization
finalization
  ADDE.Free;
  ADDE := nil;
end.
