unit Compile;

{
  Inno Setup
  Copyright (C) 1998-2000 Jordan Russell
  For conditions of distribution and use, see LICENSE.TXT.

  Compiler

  $Id: Compile.pas,v 1.8 2000/12/28 21:05:16 jr Exp $
}

{$I ISVER.INC}

interface

uses
  Windows, SysUtils, CompInt;

function ISCompileScript (const Params: TCompileScriptParams;
  const PropagateExceptions: Boolean): Integer;
function ISGetVersion: PCompilerVersionInfo;

type
  EISCompileError = class(Exception)
  public
    Filename: String;
    LineNumber: Integer;
  end;

implementation

uses
  Classes, IniFiles, TypInfo,
  CmnFunc2, Struct, CompMsgs, SetupEnt, zlib, MsgIDs;

type
  TBreakStringRec = record
    ParamName: String;
    ParamData: String;
  end;
  PBreakStringArray = ^TBreakStringArray;
  TBreakStringArray = array[0..15] of TBreakStringRec;

  TParamInfo = record
    Name: PChar;
    Flags: set of (piNoEmpty, piNoQuotes);
  end;

  TEnumIniSectionProc = procedure(const Line: PChar;
    const Ext, LineNumber: Integer) of object;

  TSetupSectionDirectives = (
    ssAdminPrivilegesRequired,
    ssAllowNoIcons,
    ssAllowRootDirectory,
    ssAlwaysCreateUninstallIcon,
    ssAlwaysRestart,
    ssAlwaysUsePersonalGroup,
    ssAppCopyright,
    ssAppId,
    ssAppMutex,
    ssAppName,
    ssAppPublisher,
    ssAppPublisherURL,
    ssAppSupportURL,
    ssAppUpdatesURL,
    ssAppVerName,
    ssAppVersion,
    ssBackColor,
    ssBackColor2,
    ssBackColorDirection,
    ssBackSolid,
    ssBits,
    ssChangesAssociations,
    ssCompressLevel,
    ssCreateAppDir,
    ssCreateUninstallRegKey,
    ssDefaultDirName,
    ssDefaultGroupName,
    ssDirExistsWarning,
    ssDisableAppendDir,
    ssDisableDirExistsWarning,
    ssDisableDirPage,
    ssDisableFinishedPage,
    ssDisableProgramGroupPage,
    ssDisableStartupPrompt,
    ssDiskClusterSize,
    ssDiskSize,
    ssDiskSpanning,
    ssDontMergeDuplicateFiles,
    ssEnableDirDoesntExistWarning,
    ssExtraDiskSpaceRequired,
    ssInfoAfterFile,
    ssInfoBeforeFile,
    ssLicenseFile,
    ssMessagesFile,
    ssMinVersion,
    ssOnlyBelowVersion,
    ssOutputBaseFilename,
    ssOutputDir,
    ssOverwriteUninstRegEntries,
    ssPassword,
    ssReserveBytes,
    ssSourceDir,
    ssUpdateUninstallLogAppName,
    ssUninstallable,
    ssUninstallDisplayIcon,
    ssUninstallDisplayName,
    ssUninstallFilesDir,
    ssUninstallIconName,
    ssUninstallLogMode,
    ssUsePreviousAppDir,
    ssUsePreviousGroup,
    ssUseSetupLdr,
    ssWindowResizable,
    ssWindowShowCaption,
    ssWindowStartMaximized,
    ssWindowVisible,
    ssWizardImageBackColor,
    ssWizardImageFile);

  TAllowedConst = (acOldData, acBreak);
  TAllowedConsts = set of TAllowedConst;

  TSetupCompiler = class
  private
    DirEntries,
    FileEntries,
    FileLocationEntries,
    IconEntries,
    IniEntries,
    RegistryEntries,
    InstallDeleteEntries,
    UninstallDeleteEntries,
    RunEntries,
    UninstallRunEntries: TList;

    FileLocationEntryFilenames, WarningsList: TStringList;

    Messages: array[TSetupMessageID] of PChar;

    OutputDir, OutputBaseFilename, ExeFilename: String;
    TempFile: TFileStream;
    TempFilename: String;
    TempFileOffset: Longint;
    CompressLevel: Integer;
    DontMergeDuplicateFiles: Boolean;

    SetupHeader: ^TSetupHeader;

    SetupDirectiveLines: array[TSetupSectionDirectives] of Integer;
    UseSetupLdr, DiskSpanning, HasRegSvr, HasRestart, BackSolid: Boolean;
    DiskTotalBytes, DiskClusterSize, ReserveBytes: Longint;
    MessagesFile, LicenseFile, InfoBeforeFile, InfoAfterFile, WizardImageFile: String;

    AbortLineNumber: Integer;
    AbortFilename: String;

    procedure AddStatus (const S: String);
    procedure AbortCompile (const Msg: String);
    procedure AbortCompileFmt (const Msg: String; const Args: array of const);
    procedure AbortCompileOnLine (const LineNumber: Integer; const Msg: String);
    procedure AbortCompileOnLineFmt (const LineNumber: Integer; const Msg: String;
      const Args: array of const);
    procedure AbortCompileParamError (const LineNumber: Integer;
      const Msg, ParamName: String);
    function PrependSourceDirName (const Filename: String): String;
    procedure BreakString (S: PChar; const Output: PBreakStringArray;
      const LineNumber: Integer);
    procedure CallIdleProc;
    procedure DoCallback (const Code: Integer; var Data: TCompilerCallbackData);
    procedure EnumIniSection (const EnumProc: TEnumIniSectionProc;
      const SectionName: String; const Ext: Integer; const Verbose: Boolean;
      const ReadFromFile: TextFile);
    function MinVersionLessThan4 (const MinVersion: TSetupVersionData): Boolean;
    procedure CheckDirConst (const S: String; const LineNumber: Integer;
      const MinVersion: TSetupVersionData; const AllowedConsts: TAllowedConsts);
    function CompareParamName (const LineNumber: Integer;
      const S: TBreakStringRec; const ParamInfo: array of TParamInfo;
      var ParamNamesFound: array of Boolean): Integer;
    procedure EnumDirs (const Line: PChar; const Ext, LineNumber: Integer);
    procedure EnumIcons (const Line: PChar; const Ext, LineNumber: Integer);
    procedure EnumINI (const Line: PChar; const Ext, LineNumber: Integer);
    procedure EnumRegistry (const Line: PChar; const Ext, LineNumber: Integer);
    procedure EnumDelete (const Line: PChar; const Ext, LineNumber: Integer);
    procedure EnumFiles (const Line: PChar; const Ext, LineNumber: Integer);
    procedure EnumRun (const Line: PChar; const Ext, LineNumber: Integer);
    procedure EnumSetup (const Line: PChar; const Ext, LineNumber: Integer);
    procedure EnumMessages (const Line: PChar; const Ext, LineNumber: Integer);
    procedure ReadMessages;
  public
    AppData: Longint;
    CallbackProc: TCompilerCallbackProc;
    CompilerDir, SourceDir: String;
    constructor Create (AOwner: TComponent);
    destructor Destroy; override;
    procedure Compile;
  end;

const
  ParamCommonFlags = 'Flags';
  ParamCommonMinVersion = 'MinVersion';
  ParamCommonOnlyBelowVersion = 'OnlyBelowVersion';

type
  TColor = $7FFFFFFF-1..$7FFFFFFF;

const
  clScrollBar = TColor(COLOR_SCROLLBAR or $80000000);
  clBackground = TColor(COLOR_BACKGROUND or $80000000);
  clActiveCaption = TColor(COLOR_ACTIVECAPTION or $80000000);
  clInactiveCaption = TColor(COLOR_INACTIVECAPTION or $80000000);
  clMenu = TColor(COLOR_MENU or $80000000);
  clWindow = TColor(COLOR_WINDOW or $80000000);
  clWindowFrame = TColor(COLOR_WINDOWFRAME or $80000000);
  clMenuText = TColor(COLOR_MENUTEXT or $80000000);
  clWindowText = TColor(COLOR_WINDOWTEXT or $80000000);
  clCaptionText = TColor(COLOR_CAPTIONTEXT or $80000000);
  clActiveBorder = TColor(COLOR_ACTIVEBORDER or $80000000);
  clInactiveBorder = TColor(COLOR_INACTIVEBORDER or $80000000);
  clAppWorkSpace = TColor(COLOR_APPWORKSPACE or $80000000);
  clHighlight = TColor(COLOR_HIGHLIGHT or $80000000);
  clHighlightText = TColor(COLOR_HIGHLIGHTTEXT or $80000000);
  clBtnFace = TColor(COLOR_BTNFACE or $80000000);
  clBtnShadow = TColor(COLOR_BTNSHADOW or $80000000);
  clGrayText = TColor(COLOR_GRAYTEXT or $80000000);
  clBtnText = TColor(COLOR_BTNTEXT or $80000000);
  clInactiveCaptionText = TColor(COLOR_INACTIVECAPTIONTEXT or $80000000);
  clBtnHighlight = TColor(COLOR_BTNHIGHLIGHT or $80000000);
  cl3DDkShadow = TColor(COLOR_3DDKSHADOW or $80000000);
  cl3DLight = TColor(COLOR_3DLIGHT or $80000000);
  clInfoText = TColor(COLOR_INFOTEXT or $80000000);
  clInfoBk = TColor(COLOR_INFOBK or $80000000);

  clBlack = TColor($000000);
  clMaroon = TColor($000080);
  clGreen = TColor($008000);
  clOlive = TColor($008080);
  clNavy = TColor($800000);
  clPurple = TColor($800080);
  clTeal = TColor($808000);
  clGray = TColor($808080);
  clSilver = TColor($C0C0C0);
  clRed = TColor($0000FF);
  clLime = TColor($00FF00);
  clYellow = TColor($00FFFF);
  clBlue = TColor($FF0000);
  clFuchsia = TColor($FF00FF);
  clAqua = TColor($FFFF00);
  clLtGray = TColor($C0C0C0);
  clDkGray = TColor($808080);
  clWhite = TColor($FFFFFF);
  clNone = TColor($1FFFFFFF);
  clDefault = TColor($20000000);

type
  TColorEntry = record
    Value: TColor;
    Name: string;
  end;

const
  Colors: array[0..41] of TColorEntry = (
    (Value: clBlack; Name: 'clBlack'),
    (Value: clMaroon; Name: 'clMaroon'),
    (Value: clGreen; Name: 'clGreen'),
    (Value: clOlive; Name: 'clOlive'),
    (Value: clNavy; Name: 'clNavy'),
    (Value: clPurple; Name: 'clPurple'),
    (Value: clTeal; Name: 'clTeal'),
    (Value: clGray; Name: 'clGray'),
    (Value: clSilver; Name: 'clSilver'),
    (Value: clRed; Name: 'clRed'),
    (Value: clLime; Name: 'clLime'),
    (Value: clYellow; Name: 'clYellow'),
    (Value: clBlue; Name: 'clBlue'),
    (Value: clFuchsia; Name: 'clFuchsia'),
    (Value: clAqua; Name: 'clAqua'),
    (Value: clWhite; Name: 'clWhite'),
    (Value: clScrollBar; Name: 'clScrollBar'),
    (Value: clBackground; Name: 'clBackground'),
    (Value: clActiveCaption; Name: 'clActiveCaption'),
    (Value: clInactiveCaption; Name: 'clInactiveCaption'),
    (Value: clMenu; Name: 'clMenu'),
    (Value: clWindow; Name: 'clWindow'),
    (Value: clWindowFrame; Name: 'clWindowFrame'),
    (Value: clMenuText; Name: 'clMenuText'),
    (Value: clWindowText; Name: 'clWindowText'),
    (Value: clCaptionText; Name: 'clCaptionText'),
    (Value: clActiveBorder; Name: 'clActiveBorder'),
    (Value: clInactiveBorder; Name: 'clInactiveBorder'),
    (Value: clAppWorkSpace; Name: 'clAppWorkSpace'),
    (Value: clHighlight; Name: 'clHighlight'),
    (Value: clHighlightText; Name: 'clHighlightText'),
    (Value: clBtnFace; Name: 'clBtnFace'),
    (Value: clBtnShadow; Name: 'clBtnShadow'),
    (Value: clGrayText; Name: 'clGrayText'),
    (Value: clBtnText; Name: 'clBtnText'),
    (Value: clInactiveCaptionText; Name: 'clInactiveCaptionText'),
    (Value: clBtnHighlight; Name: 'clBtnHighlight'),
    (Value: cl3DDkShadow; Name: 'cl3DDkShadow'),
    (Value: cl3DLight; Name: 'cl3DLight'),
    (Value: clInfoText; Name: 'clInfoText'),
    (Value: clInfoBk; Name: 'clInfoBk'),
    (Value: clNone; Name: 'clNone'));

function IdentToColor(const Ident: string; var Color: Longint): Boolean;
var
  I: Integer;
begin
  for I := Low(Colors) to High(Colors) do
    if CompareText(Colors[I].Name, Ident) = 0 then
    begin
      Result := True;
      Color := Longint(Colors[I].Value);
      Exit;
    end;
  Result := False;
end;

function StringToColor(const S: string): TColor;
begin
  if not IdentToColor(S, Longint(Result)) then
    Result := TColor(StrToInt(S));
end;

function AddPeriod (const S: String): String;
begin
  Result := S;
  if (Result <> '') and (Result[Length(Result)] > '.') then
    Result := Result + '.';
end;

function IsRelativePath (const Filename: String): Boolean;
var
  L: Integer;
begin
  Result := True;
  L := Length(Filename);
  if ((L >= 1) and (Filename[1] = '\')) or
     ((L >= 2) and (Filename[1] in ['A'..'Z', 'a'..'z']) and (Filename[2] = ':')) then
    Result := False;
end;

function GetSelfFilename: String;
{ Returns Filename of the calling DLL or application. (ParamStr(0) can only
  return the filename of the calling application.) }
var
  Buf: array[0..MAX_PATH-1] of Char;
begin
  SetString (Result, Buf, GetModuleFileName(HInstance, Buf, SizeOf(Buf)))
end;


{ TSetupCompiler }

constructor TSetupCompiler.Create (AOwner: TComponent);
begin
  inherited Create;
  DirEntries := TList.Create;
  FileEntries := TList.Create;
  FileLocationEntries := TList.Create;
  IconEntries := TList.Create;
  IniEntries := TList.Create;
  RegistryEntries := TList.Create;
  InstallDeleteEntries := TList.Create;
  UninstallDeleteEntries := TList.Create;
  RunEntries := TList.Create;
  UninstallRunEntries := TList.Create;
  FileLocationEntryFilenames := TStringList.Create;
  WarningsList := TStringList.Create;
end;

destructor TSetupCompiler.Destroy;
begin
  WarningsList.Free;
  FileLocationEntryFilenames.Free;
  UninstallRunEntries.Free;
  RunEntries.Free;
  UninstallDeleteEntries.Free;
  InstallDeleteEntries.Free;
  RegistryEntries.Free;
  IniEntries.Free;
  IconEntries.Free;
  FileLocationEntries.Free;
  FileEntries.Free;
  DirEntries.Free;
  inherited Destroy;
end;

procedure TSetupCompiler.DoCallback (const Code: Integer;
  var Data: TCompilerCallbackData);
begin
  if CallbackProc(Code, Data, AppData) = iscrRequestAbort then
    Abort;
end;

procedure TSetupCompiler.CallIdleProc;
begin
  DoCallback (iscbNotifyIdle, TCompilerCallbackData(nil^));
end;

procedure TSetupCompiler.EnumIniSection (const EnumProc: TEnumIniSectionProc;
  const SectionName: String; const Ext: Integer; const Verbose: Boolean;
  const ReadFromFile: TextFile);
var
  Data: TCompilerCallbackData;
  Reset: Boolean;
  FoundSection: Boolean;
  B, L, LastSection: String;
  LineNumber, I: Integer;
begin
  FoundSection := False;
  LineNumber := 0;
  Reset := True;
  LastSection := '';
  while True do begin
    Inc (LineNumber);
    if @ReadFromFile = nil then begin
      Data.Reset := Reset;
      Data.LineRead := nil;
      DoCallback (iscbReadScript, Data);
      if Data.LineRead = nil then
        Break;
      B := Data.LineRead;
    end
    else begin
      if Reset then
        System.Reset (ReadFromFile);
      if Eof(ReadFromFile) then
        Break;
      Readln (ReadFromFile, B);
    end;
    Reset := False;
    L := Trim(B);
    { Check for blank lines or comments }
    if (L = '') or (L[1] = ';') then Continue;
    if L[1] = '[' then begin
      I := Pos(']', L);
      if I < 3 then
        AbortCompileOnLine (LineNumber, SCompilerSectionTagInvalid);
      L := Copy(L, 2, I-2);
      if L[1] = '/' then begin
        L := Copy(L, 2, Maxint);
        if (LastSection = '') or (CompareText(L, LastSection) <> 0) then
          AbortCompileOnLineFmt (LineNumber, SCompilerSectionBadEndTag, [L]);
        LastSection := '';
        if FoundSection then
          { end tag was for the section we were enumerating }
          Break;
      end
      else begin
        if FoundSection then
          { no end tag, but a new section has been encountered }
          Break;
        if CompareText(L, SectionName) = 0 then
          FoundSection := True;
        LastSection := L;
      end;
    end
    else begin
      if not FoundSection then Continue;  { not on the right section }
      if Verbose then
        AddStatus (Format(SCompilerStatusReadSectionLine, [SectionName, LineNumber]));
      EnumProc (PChar(B), Ext, LineNumber);
    end;
  end;
end;

procedure TSetupCompiler.BreakString (S: PChar; const Output: PBreakStringArray;
  const LineNumber: Integer);
var
  ColonPos, SemicolonPos, QuotePos, P, P2: PChar;
  ParamName, Data: String;
  QuoteFound, FirstQuoteFound, LastQuoteFound, AddChar, FirstNonspaceFound: Boolean;
  CurParm, Len, I: Integer;
begin
  CurParm := 0;
  while (S <> nil) and (CurParm <= High(TBreakStringArray)) do begin
    ColonPos := StrScan(S, ':');
    if ColonPos = nil then
      ParamName := StrPas(S)
    else
      SetString (ParamName, S, ColonPos-S);
    ParamName := Trim(ParamName);
    if ParamName = '' then Break;
    if ColonPos = nil then
      AbortCompileOnLineFmt (LineNumber, SCompilerParamHasNoValue, [ParamName]);
    S := ColonPos + 1;
    SemicolonPos := StrScan(S, ';');
    QuotePos := StrScan(S, '"');
    QuoteFound := QuotePos <> nil;
    if QuoteFound and (SemicolonPos <> nil) and (QuotePos > SemicolonPos) then
      QuoteFound := False;
    if not QuoteFound then begin
      Data := '';
      P := S;
      if SemicolonPos <> nil then
        P2 := SemicolonPos
      else
        P2 := StrEnd(S);
      FirstNonspaceFound := False;
      Len := 0;
      I := 0;
      while P < P2 do begin
        if (P^ <> ' ') or FirstNonspaceFound then begin
          FirstNonspaceFound := True;
          Data := Data + P^;
          Inc (I);
          if P^ <> ' ' then Len := I;
        end;
        Inc (P);
      end;
      SetLength (Data, Len);
    end
    else begin
      Data := '';
      SemicolonPos := nil;
      P := S;
      FirstQuoteFound := False;
      LastQuoteFound := False;
      while P^ <> #0 do begin
        AddChar := False;
        case P^ of
          ' ': AddChar := FirstQuoteFound;
          '"': if not FirstQuoteFound then
                 FirstQuoteFound := True
               else begin
                 Inc (P);
                 if P^ = '"' then
                   AddChar := True
                 else begin
                   LastQuoteFound := True;
                   while P^ <> #0 do begin
                     case P^ of
                       ' ': ;
                       ';': begin
                              SemicolonPos := P;
                              Break;
                            end;
                     else
                       AbortCompileOnLineFmt (LineNumber, SCompilerParamQuoteError, [ParamName]);
                     end;
                     Inc (P);
                   end;
                   Break;
                 end;
               end;
        else
          if not FirstQuoteFound then
            AbortCompileOnLineFmt (LineNumber, SCompilerParamQuoteError, [ParamName]);
          AddChar := True;
        end;
        if AddChar then
          Data := Data + P^;
        Inc (P);
      end;
      if not LastQuoteFound then
        AbortCompileOnLineFmt (LineNumber, SCompilerParamMissingClosingQuote, [ParamName]);
    end;
    S := SemicolonPos;
    if S <> nil then Inc (S);
    Output^[CurParm].ParamName := ParamName;
    Output^[CurParm].ParamData := Data;
    Inc (CurParm);
  end;
end;

procedure TSetupCompiler.AddStatus (const S: String);
var
  Data: TCompilerCallbackData;
begin
  Data.StatusMsg := PChar(S);
  CallbackProc (iscbNotifyStatus, Data, AppData);
end;

procedure TSetupCompiler.AbortCompile (const Msg: String);
var
  E: EISCompileError;
begin
  E := EISCompileError.Create(Msg);
  E.Filename := AbortFilename;
  E.LineNumber := AbortLineNumber;
  raise E;
end;

procedure TSetupCompiler.AbortCompileFmt (const Msg: String; const Args: array of const);
begin
  AbortCompile (Format(Msg, Args));
end;

procedure TSetupCompiler.AbortCompileOnLine (const LineNumber: Integer; const Msg: String);
begin
  AbortLineNumber := LineNumber;
  AbortCompile (Msg);
end;

procedure TSetupCompiler.AbortCompileOnLineFmt (const LineNumber: Integer;
  const Msg: String; const Args: array of const);
begin
  AbortCompileOnLine (LineNumber, Format(Msg, Args));
end;

procedure TSetupCompiler.AbortCompileParamError (const LineNumber: Integer;
  const Msg, ParamName: String);
begin
  AbortCompileOnLineFmt (LineNumber, Msg, [ParamName]);
end;

function TSetupCompiler.PrependSourceDirName (const Filename: String): String;
begin
  if CompareText(Copy(Filename, 1, 9), 'compiler:') = 0 then
    Result := CompilerDir + Copy(Filename, 10, Maxint)
  else begin
    if (Filename = '') or not IsRelativePath(Filename) then
      Result := Filename
    else
      Result := SourceDir + Filename;
  end;
end;

function TSetupCompiler.MinVersionLessThan4 (const MinVersion: TSetupVersionData): Boolean;
begin
  with MinVersion do
    Result := ((WinVersion <> 0) and (WinVersion < $04000000)) or
      ((NTVersion <> 0) and (NTVersion < $04000000));
end;

function MinVersionErrorMessage (const Param: Boolean;
  const WinVersion, NTVersion: Word): String;

  function VerToStr (Ver: Cardinal; ServicePack: Word): String;
  var
    Digits: Integer;
  begin
    with TSetupVersionDataVersion(Ver) do begin
      Digits := 2;
      if Minor mod 10 = 0 then begin
        Dec (Digits);
        Minor := Minor div 10;
      end;
      FmtStr (Result, '%d.%.*d', [Major, Digits, Minor]);
      if Build <> 0 then
        Result := Result + Format('.%d', [Build]);
      if ServicePack <> 0 then begin
        Result := Result + ' Service Pack ' + IntToStr(Hi(ServicePack));
        if Lo(ServicePack) <> 0 then
          Result := Result + Format('.%d', [Lo(ServicePack)]);
      end;
    end;
  end;

var
  WinVer, NTVer: String;
begin
  WinVer := VerToStr(WinVersion shl 16, 0);
  NTVer := VerToStr(NTVersion shl 16, 0);
  if not Param then
    FmtStr (Result, SCompilerMinVersionError, [WinVer, NTVer, WinVer, NTVer])
  else
    FmtStr (Result, SCompilerMinVersionErrorParam, [WinVer, NTVer,
      WinVer, NTVer, WinVer, NTVer]);
end;

procedure TSetupCompiler.CheckDirConst (const S: String; const LineNumber: Integer;
  const MinVersion: TSetupVersionData; const AllowedConsts: TAllowedConsts);

  function CheckRegConst (C: String): Boolean;
  { based on ExpandRegConst in Main.pas }
  type
    TKeyNameConst = packed record
      KeyName: String;
      KeyConst: HKEY;
    end;
  const
    KeyNameConsts: array[0..4] of TKeyNameConst = (
      (KeyName: 'HKCR'; KeyConst: HKEY_CLASSES_ROOT),
      (KeyName: 'HKCU'; KeyConst: HKEY_CURRENT_USER),
      (KeyName: 'HKLM'; KeyConst: HKEY_LOCAL_MACHINE),
      (KeyName: 'HKU';  KeyConst: HKEY_USERS),
      (KeyName: 'HKCC'; KeyConst: HKEY_CURRENT_CONFIG));
  var
    Z, Subkey, Value, Default: String;
    I, J: Integer;
    RootKey: HKEY;
  begin
    Delete (C, 1, 4);  { skip past 'reg:' }
    I := ConstPos('\', C);
    if I <> 0 then begin
      Z := Copy(C, 1, I-1);
      if Z <> '' then begin
        RootKey := 0;
        for J := Low(KeyNameConsts) to High(KeyNameConsts) do
          if CompareText(KeyNameConsts[J].KeyName, Z) = 0 then begin
            RootKey := KeyNameConsts[J].KeyConst;
            Break;
          end;
        if RootKey <> 0 then begin
          Z := Copy(C, I+1, Maxint);
          I := ConstPos('|', Z);  { check for a 'default' data }
          if I = 0 then
            I := Length(Z)+1;
          Default := Copy(Z, I+1, Maxint);
          SetLength (Z, I-1);
          I := ConstPos(',', Z);  { comma separates subkey and value }
          if I <> 0 then begin
            Subkey := Copy(Z, 1, I-1);
            Value := Copy(Z, I+1, Maxint);
            if ConvertConstPercentStr(Subkey) and ConvertConstPercentStr(Value) and
               ConvertConstPercentStr(Default) then begin
              CheckDirConst (Subkey, LineNumber, MinVersion, AllowedConsts);
              CheckDirConst (Value, LineNumber, MinVersion, AllowedConsts);
              CheckDirConst (Default, LineNumber, MinVersion, AllowedConsts);
              Result := True;
              Exit;
            end;
          end;
        end;
      end;
    end;
    { it will only reach here if there was a parsing error }
    Result := False;
  end;

  function CheckIniConst (C: String): Boolean;
  { based on ExpandIniConst in Main.pas }
  var
    Z, Filename, Section, Key, Default: String;
    I: Integer;
  begin
    Delete (C, 1, 4);  { skip past 'ini:' }
    I := ConstPos(',', C);
    if I <> 0 then begin
      Z := Copy(C, 1, I-1);
      if Z <> '' then begin
        Filename := Z;
        Z := Copy(C, I+1, Maxint);
        I := ConstPos('|', Z);  { check for a 'default' data }
        if I = 0 then
          I := Length(Z)+1;
        Default := Copy(Z, I+1, Maxint);
        SetLength (Z, I-1);
        I := ConstPos(',', Z);  { comma separates section and key }
        if I <> 0 then begin
          Section := Copy(Z, 1, I-1);
          Key := Copy(Z, I+1, Maxint);
          if ConvertConstPercentStr(Filename) and ConvertConstPercentStr(Section) and
             ConvertConstPercentStr(Key) and ConvertConstPercentStr(Default) then begin
            CheckDirConst (Filename, LineNumber, MinVersion, AllowedConsts);
            CheckDirConst (Section, LineNumber, MinVersion, AllowedConsts);
            CheckDirConst (Key, LineNumber, MinVersion, AllowedConsts);
            CheckDirConst (Default, LineNumber, MinVersion, AllowedConsts);
            Result := True;
            Exit;
          end;
        end;
      end;
    end;
    { it will only reach here if there was a parsing error }
    Result := False;
  end;

const
  Consts: array[0..14] of PChar = (
    'src', 'srcexe', 'tmp', 'app', 'win', 'sys', 'sd', 'groupname', 'fonts',
    'hwnd', 'pf', 'cf', 'computername', 'dao', 'username');
  ShellFolderConsts: array[0..16] of PChar = (
    'group', 'userdesktop', 'userstartmenu', 'userprograms', 'userstartup',
    'commondesktop', 'commonstartmenu', 'commonprograms', 'commonstartup',
    'sendto', 'userappdata', 'userdocs', 'commonappdata', 'commondocs',
    'usertemplates', 'commontemplates', 'localappdata');
  ShellFolderConsts98: array[0..1] of PChar = ('userfavorites', 'commonfavorites');
  AllowedConstsNames: array[TAllowedConst] of PChar = (
    'olddata', 'break');
var
  NoShellConsts, NotWin98orNT4: Boolean;
  I, Start, K: Integer;
  C: TAllowedConst;
  Cnst: String;
label 1;
begin
  NoShellConsts := MinVersionLessThan4(MinVersion);
  with MinVersion do
    NotWin98orNT4 := ((WinVersion <> 0) and (WinVersion < $040A0000)) or
      ((NTVersion <> 0) and (NTVersion < $04000000));
  I := 1;
  while I <= Length(S) do begin
    if S[I] = '{' then begin
      if (I < Length(S)) and (S[I+1] = '{') then
        Inc (I)
      else begin
        Start := I;
        { Find the closing brace, skipping over any embedded constants }
        I := SkipPastConst(S, I);
        if I = 0 then  { unclosed constant? }
          AbortCompileOnLineFmt (LineNumber, SCompilerUnterminatedConst, [Copy(S, Start+1, Maxint)]);
        Dec (I);  { 'I' now points to the closing brace }

        { Now check the constant }
        Cnst := Copy(S, Start+1, I-(Start+1));
        if Cnst <> '' then begin
          if (Cnst[1] = '%') or (Cnst = '\') then
            goto 1;
          if Copy(Cnst, 1, 4) = 'reg:' then begin
            if not CheckRegConst(Cnst) then
              AbortCompileOnLineFmt (LineNumber, SCompilerBadRegConst, [Cnst]);
            goto 1;
          end;
          if Copy(Cnst, 1, 4) = 'ini:' then begin
            if not CheckIniConst(Cnst) then
              AbortCompileOnLineFmt (LineNumber, SCompilerBadIniConst, [Cnst]);
            goto 1;
          end;
          for K := Low(Consts) to High(Consts) do
            if Cnst = StrPas(Consts[K]) then
              goto 1;
          for K := Low(ShellFolderConsts) to High(ShellFolderConsts) do
            if Cnst = StrPas(ShellFolderConsts[K]) then begin
              if NoShellConsts then
                AbortCompileOnLineFmt (LineNumber, SCompilerConstUsed + SNewLine2 +
                  MinVersionErrorMessage(True, $400, $400), [Cnst]);
              goto 1;
            end;
          for K := Low(ShellFolderConsts98) to High(ShellFolderConsts98) do
            if Cnst = StrPas(ShellFolderConsts98[K]) then begin
              if NotWin98orNT4 then
                AbortCompileOnLineFmt (LineNumber, SCompilerConstUsed + SNewLine2 +
                  MinVersionErrorMessage(True, $40A, $400), [Cnst]);
              goto 1;
            end;
          for C := Low(C) to High(C) do
            if Cnst = StrPas(AllowedConstsNames[C]) then begin
              if not(C in AllowedConsts) then
                AbortCompileOnLineFmt (LineNumber, SCompilerConstCannotUse, [Cnst]);
              goto 1;
            end;
        end;
        AbortCompileOnLineFmt (LineNumber, SCompilerUnknownConst, [Cnst]);

      1:{ Constant is OK }
      end;
    end;
    Inc (I);
  end;
end;

function TSetupCompiler.CompareParamName (const LineNumber: Integer;
  const S: TBreakStringRec; const ParamInfo: array of TParamInfo;
  var ParamNamesFound: array of Boolean): Integer;
var
  I: Integer;
begin
  Result := -1;
  for I := 0 to High(ParamInfo) do
    if StrIComp(ParamInfo[I].Name, PChar(S.ParamName)) = 0 then begin
      Result := I;
      if ParamNamesFound[I] then
        AbortCompileParamError (LineNumber, SCompilerParamDuplicated, StrPas(ParamInfo[I].Name));
      ParamNamesFound[I] := True;
      if (piNoEmpty in ParamInfo[I].Flags) and (S.ParamData = '') then
        AbortCompileParamError (LineNumber, SCompilerParamEmpty2, StrPas(ParamInfo[I].Name));
      if (piNoQuotes in ParamInfo[I].Flags) and (Pos('"', S.ParamData) <> 0) then
        AbortCompileParamError (LineNumber, SCompilerParamNoQuotes2, StrPas(ParamInfo[I].Name));
      Break;
    end;
end;

function ExtractFlag (var S: String; const FlagStrs: array of PChar): Integer;
var
  I: Integer;
  F: array[0..255] of Char;
begin
  I := Pos(' ', S);
  if I = 0 then I := Length(S)+1;
  StrPLCopy (F, Trim(Copy(S, 1, I-1)), SizeOf(F)-1);
  if F[0] = #0 then begin
    Result := -2;
    Exit;
  end;
  S := Trim(Copy(S, I+1, Maxint));
  Result := -1;
  for I := 0 to High(FlagStrs) do
    if StrIComp(FlagStrs[I], F) = 0 then begin
      Result := I;
      Break;
    end;
end;

function StrToVersionNumbers (const S: String; var VerData: TSetupVersionData): Boolean;
  procedure Split (const Str: String; var Ver: TSetupVersionDataVersion;
    var ServicePack: Word);
  var
    I, J: Integer;
    Z, B: String;
    HasBuild: Boolean;
  begin
    Cardinal(Ver) := 0;
    ServicePack := 0;
    Z := Lowercase(Str);
    I := Pos('sp', Z);
    if I <> 0 then begin
      J := StrToInt(Copy(Z, I+2, Maxint));
      if (J < Low(Byte)) or (J > High(Byte)) then
        Abort;
      ServicePack := J shl 8;
      { ^ Shift left 8 bits because we're setting the "major" service pack
        version number. This parser doesn't currently accept "minor" service
        pack version numbers. }
      SetLength (Z, I-1);
    end;
    I := Pos('.', Z);
    if I = Length(Z) then Abort;
    if I <> 0 then begin
      J := StrToInt(Copy(Z, 1, I-1));
      if (J < Low(Ver.Major)) or (J > High(Ver.Major)) then
        Abort;
      Ver.Major := J;
      Z := Copy(Z, I+1, Maxint);
      I := Pos('.', Z);
      HasBuild := I <> 0;
      if not HasBuild then
        I := Length(Z)+1;
      B := Copy(Z, I+1, Maxint);
      Z := Copy(Z, 1, I-1);
      J := StrToInt(Z);
      if (J < 0) or (J > 99) then Abort;
      if (J < 10) and (Z[1] <> '0') then J := J * 10;
      Ver.Minor := J;
      if HasBuild then begin
        J := StrToInt(B);
        if (J < Low(Ver.Build)) or (J > High(Ver.Build)) then
          Abort;
        Ver.Build := J;
      end;
    end
    else begin  { no minor version specified }
      J := StrToInt(Str);
      if (J < Low(Ver.Major)) or (J > High(Ver.Major)) then
        Abort;
      Ver.Major := J;
    end;
  end;
var
  I: Integer;
  SP: Word;
begin
  try
    I := Pos(',', S);
    if I = 0 then Abort;
    Split (Trim(Copy(S, 1, I-1)),
      TSetupVersionDataVersion(VerData.WinVersion), SP);
    if SP <> 0 then Abort;  { only NT has service packs }
    Split (Trim(Copy(S, I+1, Maxint)),
      TSetupVersionDataVersion(VerData.NTVersion), VerData.NTServicePack);
    Result := True;
  except
    Result := False;
  end;
end;

procedure TSetupCompiler.EnumSetup (const Line: PChar; const Ext, LineNumber: Integer);

  procedure Separate (const Line: PChar; var Key, Value: String);
  var
    P, P2: PChar;
    L: Cardinal;
  begin
    Key := '';
    Value := '';
    P := Line;
    while (P^ <> #0) and (P^ <= ' ') do
      Inc (P);
    if P^ = #0 then
      Exit;
    P2 := P;
    while (P2^ <> #0) and (P2^ <> '=') do
      Inc (P2);
    L := P2 - P;
    SetLength (Key, L);
    Move (P^, Key[1], Length(Key));
    Key := TrimRight(Key);
    if P2^ = #0 then
      Exit;
    P := P2 + 1;
    while (P^ <> #0) and (P^ <= ' ') do
      Inc (P);
    if P^ = #0 then
      Exit;
    Value := TrimRight(StrPas(P));
  end;

var
  KeyName, Value: String;
  I: Integer;
  Directive: TSetupSectionDirectives;

  procedure Invalid;
  begin
    AbortCompileOnLineFmt (LineNumber, SCompilerEntryInvalid2, ['Setup',
      KeyName]);
  end;

  function StrToBool (S: String): Boolean;
  begin
    Result := False;
    S := Lowercase(S);
    if (S = '0') or (S = 'no') or (S = 'false') then
      { Result already False }
    else if (S = '1') or (S = 'yes') or (S = 'true') then
      Result := True
    else
      Invalid;
  end;

  procedure SetSetupHeaderOption (const Option: TSetupHeaderOption);
  begin
    if not StrToBool(Value) then
      Exclude (SetupHeader^.Options, Option)
    else
      Include (SetupHeader^.Options, Option);
  end;

begin
  Separate (Line, KeyName, Value);

  if KeyName = '' then
    Exit;
  I := GetEnumValue(TypeInfo(TSetupSectionDirectives), 'ss' + KeyName);
  if I = -1 then
    AbortCompileOnLineFmt (LineNumber, SCompilerUnknownDirective,
      ['Setup', KeyName]);
  Directive := TSetupSectionDirectives(I);
  if SetupDirectiveLines[Directive] <> 0 then
    AbortCompileOnLineFmt (LineNumber, SCompilerEntryAlreadySpecified,
      ['Setup', KeyName]);
  SetupDirectiveLines[Directive] := LineNumber;
  case Directive of
    ssAdminPrivilegesRequired: begin
        SetSetupHeaderOption (shAdminPrivilegesRequired);
      end;
    ssAllowNoIcons: begin
        SetSetupHeaderOption (shAllowNoIcons);
      end;
    ssAllowRootDirectory: begin
        SetSetupHeaderOption (shAllowRootDirectory);
      end;
    ssAlwaysCreateUninstallIcon: begin
        SetSetupHeaderOption (shAlwaysCreateUninstallIcon);
      end;
    ssAlwaysRestart: begin
        SetSetupHeaderOption (shAlwaysRestart);
      end;
    ssAlwaysUsePersonalGroup: begin
        SetSetupHeaderOption (shAlwaysUsePersonalGroup);
      end;
    ssAppCopyright: begin
        SetupHeader^.AppCopyright := Value;
      end;
    ssAppId: begin
        if Length(Value) > 127 then
          AbortCompileOnLineFmt (LineNumber, SCompilerEntryTooLong, ['Setup',
            KeyName, 127]);
        SetupHeader^.AppId := Value;
      end;
    ssAppMutex: begin
        SetupHeader^.AppMutex := Trim(Value);
      end;
    ssAppName: begin
        if Value = '' then
          Invalid;
        if Length(Value) > 127 then
          AbortCompileOnLineFmt (LineNumber, SCompilerEntryTooLong, ['Setup',
            KeyName, 127]);
        SetupHeader^.AppName := Value;
      end;
    ssAppPublisher: begin
        if Value = '' then
          Invalid;
        SetupHeader^.AppPublisher := Value;
      end;
    ssAppPublisherURL: begin
        if Value = '' then
          Invalid;
        SetupHeader^.AppPublisherURL := Value;
      end;
    ssAppSupportURL: begin
        if Value = '' then
          Invalid;
        SetupHeader^.AppSupportURL := Value;
      end;
    ssAppUpdatesURL: begin
        if Value = '' then
          Invalid;
        SetupHeader^.AppUpdatesURL := Value;
      end;
    ssAppVerName: begin
        if Value = '' then
          Invalid;
        SetupHeader^.AppVerName := Value;
      end;
    ssAppVersion: begin
        if Value = '' then
          Invalid;
        SetupHeader^.AppVersion := Value;
      end;
    ssBackColor: begin
        try
          SetupHeader^.BackColor := StringToColor(Value);
        except
          Invalid;
        end;
      end;
    ssBackColor2: begin
        try
          SetupHeader^.BackColor2 := StringToColor(Value);
        except
          Invalid;
        end;
      end;
    ssBackColorDirection: begin
        if CompareText(Value, 'toptobottom') = 0 then
          Exclude (SetupHeader^.Options, shBackColorHorizontal)
        else if CompareText(Value, 'lefttoright') = 0 then
          Include (SetupHeader^.Options, shBackColorHorizontal)
        else
          Invalid;
      end;
    ssBackSolid: begin
        BackSolid := StrToBool(Value);
      end;
    ssBits: begin  { obsolete }
        I := StrToIntDef(Value, -1);
        if (I <> 16) and (I <> 32) then
          AbortCompileOnLine (LineNumber, SCompilerBitsNot16or32_2);
        if I = 16 then
          AbortCompileOnLine (LineNumber, SCompilerNeedToUse16);
        WarningsList.Add (Format(SCompilerEntryObsolete, ['Setup', KeyName]));
      end;
    ssChangesAssociations: begin
        SetSetupHeaderOption (shChangesAssociations)
      end;
    ssCompressLevel: begin
        I := StrToIntDef(Value, -1);
        if (I < 0) or (I > 9) then
          Invalid;
        CompressLevel := I;
      end;
    ssCreateAppDir: begin
        SetSetupHeaderOption (shCreateAppDir);
      end;
    ssCreateUninstallRegKey: begin
        SetSetupHeaderOption (shCreateUninstallRegKey);
      end;
    ssDefaultDirName: begin
        SetupHeader^.DefaultDirName := Value;
      end;
    ssDefaultGroupName: begin
        SetupHeader^.DefaultGroupName := Value;
      end;
    ssDirExistsWarning: begin
        if CompareText(Value, 'auto') = 0 then
          SetupHeader^.DirExistsWarning := ddAuto
        else if StrToBool(Value) then
          { ^ exception will be raised if Value is invalid }
          SetupHeader^.DirExistsWarning := ddYes
        else
          SetupHeader^.DirExistsWarning := ddNo;
      end;
    ssDisableAppendDir: begin
        SetSetupHeaderOption (shDisableAppendDir);
      end;
    ssDisableDirExistsWarning: begin  { obsolete; superceded by "DirExistsWarning" }
        if SetupDirectiveLines[ssDirExistsWarning] = 0 then begin
          if StrToBool(Value) then
            SetupHeader^.DirExistsWarning := ddNo
          else
            SetupHeader^.DirExistsWarning := ddAuto;
        end;
        WarningsList.Add (Format(SCompilerEntrySuperseded2, ['Setup', KeyName,
          'DirExistsWarning']));
      end;
    ssDisableDirPage: begin
        SetSetupHeaderOption (shDisableDirPage);
      end;
    ssDisableFinishedPage: begin
        SetSetupHeaderOption (shDisableFinishedPage);
      end;
    ssDisableProgramGroupPage: begin
        SetSetupHeaderOption (shDisableProgramGroupPage);
      end;
    ssDisableStartupPrompt: begin
        SetSetupHeaderOption (shDisableStartupPrompt);
      end;
    ssDiskClusterSize: begin
        Val (Value, DiskClusterSize, I);
        if I <> 0 then
          Invalid;
        if (DiskClusterSize < 1) or (DiskClusterSize > 32768) then
          AbortCompileOnLine (LineNumber, SCompilerDiskClusterSizeInvalid);
      end;
    ssDiskSize: begin
        Val (Value, DiskTotalBytes, I);
        if I <> 0 then
          Invalid;
        if DiskTotalBytes < 262144 then
          AbortCompileFmt (SCompilerDiskSpaceTooSmall, [262144]);
      end;
    ssDiskSpanning: begin
        DiskSpanning := StrToBool(Value);
      end;
    ssDontMergeDuplicateFiles: begin
        DontMergeDuplicateFiles := StrToBool(Value);
      end;
    ssEnableDirDoesntExistWarning: begin
        SetSetupHeaderOption (shEnableDirDoesntExistWarning);
      end;
    ssExtraDiskSpaceRequired: begin
        Val (Value, SetupHeader^.ExtraDiskSpaceRequired, I);
        if (I <> 0) or (SetupHeader^.ExtraDiskSpaceRequired < 0) then
          Invalid;
      end;
    ssInfoBeforeFile: begin
        InfoBeforeFile := RemoveQuotes(Value);
      end;
    ssInfoAfterFile: begin
        InfoAfterFile := RemoveQuotes(Value);
      end;
    ssLicenseFile: begin
        LicenseFile := RemoveQuotes(Value);
      end;
    ssMessagesFile: begin
        Value := RemoveQuotes(Value);
        if Value = '' then
          Invalid;
        MessagesFile := Value;
      end;
    ssMinVersion: begin
        if not StrToVersionNumbers(Value, SetupHeader^.MinVersion) then
          Invalid;
        if (SetupHeader^.MinVersion.WinVersion <> 0) and
           (SetupHeader^.MinVersion.WinVersion < $04000000{4.0}) then
          AbortCompileOnLineFmt (LineNumber, SCompilerMinVersionWinTooLow, ['4.0']);
        if (SetupHeader^.MinVersion.NTVersion <> 0) and
           (SetupHeader^.MinVersion.NTVersion < $03330000{3.51}) then
          AbortCompileOnLineFmt (LineNumber, SCompilerMinVersionNTTooLow, ['3.51']);
      end;
    ssOnlyBelowVersion: begin
        if not StrToVersionNumbers(Value, SetupHeader^.OnlyBelowVersion) then
          Invalid;
      end;
    ssOutputBaseFilename: begin
        Value := RemoveQuotes(Value);
        if Value = '' then
          Invalid;
        OutputBaseFilename := Value;
      end;
    ssOutputDir: begin
        Value := RemoveQuotes(Value);
        if Value = '' then
          Invalid;
        OutputDir := Value;
      end;
    ssOverwriteUninstRegEntries: begin  { obsolete; ignored }
        { was: SetSetupHeaderOption (shOverwriteUninstRegEntries); }
        WarningsList.Add (Format(SCompilerEntryObsolete, ['Setup', KeyName]));
      end;
    ssPassword: begin
        if Value <> '' then begin
          SetupHeader^.Password := GetCRC32(Value[1], Length(Value));
          Include (SetupHeader^.Options, shPassword);
        end;
      end;
    ssReserveBytes: begin
        Val (Value, ReserveBytes, I);
        if (I <> 0) or (ReserveBytes < 0) then
          Invalid;
      end;
    ssSourceDir: begin
        if Value = '' then
          Invalid;
        if IsRelativePath(Value) then
          SourceDir := AddBackslash(SourceDir) + Value
        else
          SourceDir := Value;
      end;
    ssUpdateUninstallLogAppName: begin
        SetSetupHeaderOption (shUpdateUninstallLogAppName);
      end;
    ssUninstallable: begin
        SetSetupHeaderOption (shUninstallable);
      end;
    ssUninstallDisplayIcon: begin
        SetupHeader^.UninstallDisplayIcon := RemoveQuotes(Value);
      end;
    ssUninstallDisplayName: begin
        SetupHeader^.UninstallDisplayName := Value;
      end;
    ssUninstallFilesDir: begin
        Value := RemoveQuotes(Value);
        if Value = '' then
          Invalid;
        SetupHeader^.UninstallFilesDir := Value;
      end;
    ssUninstallIconName: begin
        SetupHeader^.UninstallIconName := RemoveQuotes(Value);
      end;
    ssUninstallLogMode: begin
        if CompareText(Value, 'append') = 0 then
          SetupHeader^.UninstallLogMode := lmAppend
        else if CompareText(Value, 'new') = 0 then
          SetupHeader^.UninstallLogMode := lmNew
        else if CompareText(Value, 'overwrite') = 0 then
          SetupHeader^.UninstallLogMode := lmOverwrite
        else
          Invalid;
      end;
    ssUsePreviousAppDir: begin
        SetSetupHeaderOption (shUsePreviousAppDir);
      end;
    ssUsePreviousGroup: begin
        SetSetupHeaderOption (shUsePreviousGroup);
      end;
    ssUseSetupLdr: begin
        UseSetupLdr := StrToBool(Value);
      end;
    ssWindowResizable: begin
        SetSetupHeaderOption (shWindowResizable);
      end;
    ssWindowShowCaption: begin
        SetSetupHeaderOption (shWindowShowCaption);
      end;
    ssWindowStartMaximized: begin
        SetSetupHeaderOption (shWindowStartMaximized);
      end;
    ssWindowVisible: begin
        SetSetupHeaderOption (shWindowVisible);
      end;
    ssWizardImageBackColor: begin
        try
          SetupHeader^.WizardImageBackColor := StringToColor(Value);
        except
          Invalid;
        end;
      end;
    ssWizardImageFile: begin
        Value := RemoveQuotes(Value);
        if Value = '' then
          Invalid;
        WizardImageFile := Value;
      end;
  end;
end;

procedure TSetupCompiler.EnumDirs (const Line: PChar; const Ext, LineNumber: Integer);
const
  ParamDirsName = 'Name';
  ParamNames: array[0..3] of TParamInfo = (
    (Name: ParamCommonFlags; Flags: []),
    (Name: ParamDirsName; Flags: [piNoEmpty, piNoQuotes]),
    (Name: ParamCommonMinVersion; Flags: []),
    (Name: ParamCommonOnlyBelowVersion; Flags: []));
  Flags: array[0..2] of PChar = (
    'uninsneveruninstall', 'deleteafterinstall', 'uninsalwaysuninstall');
var
  Params: PBreakStringArray;
  ParamNameFound: array[Low(ParamNames)..High(ParamNames)] of Boolean;
  NewDirEntry: PSetupDirEntry;
  P: Integer;
begin
  New (Params);
  try
    FillChar (Params^, SizeOf(Params^), 0);
    FillChar (ParamNameFound, SizeOf(ParamNameFound), 0);
    BreakString (PChar(Line), Params, LineNumber);

    NewDirEntry := AllocMem(SizeOf(TSetupDirEntry));
    try
      with NewDirEntry^ do begin
        MinVersion := SetupHeader^.MinVersion;

        for P := Low(Params^) to High(Params^) do
          with Params^[P] do begin
            if ParamName = '' then Break;
            case CompareParamName(LineNumber, Params^[P], ParamNames, ParamNameFound) of
              -1: AbortCompileOnLineFmt (LineNumber, SCompilerParamUnknownParam, [Params^[P].ParamName]);
              0: while True do
                   case ExtractFlag(ParamData, Flags) of
                     -2: Break;
                     -1: AbortCompileParamError (LineNumber, SCompilerParamUnknownFlag2, ParamCommonFlags);
                     0: Include (Options, doUninsNeverUninstall);
                     1: Include (Options, doDeleteAfterInstall);
                     2: Include (Options, doUninsAlwaysUninstall);
                   end;
              1: DirName := ParamData;
              2: if not StrToVersionNumbers(ParamData, MinVersion) then
                   AbortCompileParamError (LineNumber, SCompilerParamInvalid2, ParamCommonMinVersion);
              3: if not StrToVersionNumbers(ParamData, OnlyBelowVersion) then
                   AbortCompileParamError (LineNumber, SCompilerParamInvalid2, ParamCommonOnlyBelowVersion);
            end;
          end;

        if not ParamNameFound[1] then
          AbortCompileParamError (LineNumber, SCompilerParamNotSpecified, ParamDirsName);

        if (doUninsNeverUninstall in Options) and
           (doUninsAlwaysUninstall in Options) then
          AbortCompileOnLineFmt (LineNumber, SCompilerParamErrorBadCombo2,
            [ParamCommonFlags, 'uninsneveruninstall', 'uninsalwaysuninstall']);

        CheckDirConst (DirName, LineNumber, MinVersion, []);
      end;
    except
      SEFreeRec (NewDirEntry, SetupDirEntryStrings);
      raise;
    end;
    DirEntries.Add (NewDirEntry);
  finally
    Dispose (Params);
  end;
end;

const
  BadPMChars = ',()[]';

function ContainsBadPMChars (const S: String): Boolean;
var
  I: Integer;
begin
  for I := 1 to Length(S) do
    if Pos(S[I], BadPMChars) <> 0 then begin
      Result := True;
      Exit;
    end;
  Result := False;
end;

function SpaceString (const S: String): String;
var
  I: Integer;
begin
  Result := '';
  for I := 1 to Length(S) do begin
    if S[I] = ' ' then Continue;
    if Result <> '' then Result := Result + ' ';
    Result := Result + S[I];
  end;
end;

procedure TSetupCompiler.EnumIcons (const Line: PChar; const Ext, LineNumber: Integer);
const
  ParamIconsName = 'Name';
  ParamIconsFilename = 'Filename';
  ParamIconsParameters = 'Parameters';
  ParamIconsWorkingDir = 'WorkingDir';
  ParamIconsIconFilename = 'IconFilename';
  ParamIconsIconIndex = 'IconIndex';
  ParamIconsComment = 'Comment';
  ParamNames: array[0..9] of TParamInfo = (
    (Name: ParamCommonFlags; Flags: []),
    (Name: ParamIconsName; Flags: [piNoEmpty, piNoQuotes]),
    (Name: ParamIconsFilename; Flags: [piNoEmpty, piNoQuotes]),
    (Name: ParamIconsParameters; Flags: []),
    (Name: ParamIconsWorkingDir; Flags: [piNoQuotes]),
    (Name: ParamIconsIconFilename; Flags: [piNoQuotes]),
    (Name: ParamIconsIconIndex; Flags: []),
    (Name: ParamCommonMinVersion; Flags: []),
    (Name: ParamCommonOnlyBelowVersion; Flags: []),
    (Name: ParamIconsComment; Flags: []));
  Flags: array[0..6] of PChar = (
    'uninsneveruninstall', 'runminimized', 'createonlyiffileexists',
    'useapppaths', 'closeonexit', 'dontcloseonexit', 'runmaximized');
var
  Params: PBreakStringArray;
  ParamNameFound: array[Low(ParamNames)..High(ParamNames)] of Boolean;
  NewIconEntry: PSetupIconEntry;
  P: Integer;
  S: String;
begin
  New (Params);
  try
    FillChar (Params^, SizeOf(Params^), 0);
    FillChar (ParamNameFound, SizeOf(ParamNameFound), 0);
    BreakString (PChar(Line), Params, LineNumber);

    NewIconEntry := AllocMem(SizeOf(TSetupIconEntry));
    try
      with NewIconEntry^ do begin
        MinVersion := SetupHeader^.MinVersion;
        ShowCmd := SW_SHOWNORMAL;

        for P := Low(Params^) to High(Params^) do
          with Params^[P] do begin
            if ParamName = '' then Break;
            case CompareParamName(LineNumber, Params^[P], ParamNames, ParamNameFound) of
              -1: AbortCompileOnLineFmt (LineNumber, SCompilerParamUnknownParam, [Params^[P].ParamName]);
              0: while True do
                   case ExtractFlag(ParamData, Flags) of
                     -2: Break;
                     -1: AbortCompileParamError (LineNumber, SCompilerParamUnknownFlag2, ParamCommonFlags);
                     0: Include (Options, ioUninsNeverUninstall);
                     1: ShowCmd := SW_SHOWMINNOACTIVE;
                     2: Include (Options, ioCreateOnlyIfFileExists);
                     3: Include (Options, ioUseAppPaths);
                     4: CloseOnExit := icYes;
                     5: CloseOnExit := icNo;
                     6: ShowCmd := SW_SHOWMAXIMIZED;
                   end;
              1: IconName := ParamData;
              2: Filename := ParamData;
              3: Parameters := ParamData;
              4: WorkingDir := ParamData;
              5: IconFilename := ParamData;
              6: begin
                   try
                     IconIndex := StrToInt(ParamData);
                   except
                     AbortCompileOnLine (LineNumber, SCompilerIconsIconIndexInvalid);
                   end;
                 end;
              7: if not StrToVersionNumbers(ParamData, MinVersion) then
                   AbortCompileParamError (LineNumber, SCompilerParamInvalid2, ParamCommonMinVersion);
              8: if not StrToVersionNumbers(ParamData, OnlyBelowVersion) then
                   AbortCompileParamError (LineNumber, SCompilerParamInvalid2, ParamCommonOnlyBelowVersion);
              9: Comment := ParamData;
            end;
          end;

        if not ParamNameFound[1] then
          AbortCompileParamError (LineNumber, SCompilerParamNotSpecified, ParamIconsName);
        if not ParamNameFound[2] then
          AbortCompileParamError (LineNumber, SCompilerParamNotSpecified, ParamIconsFilename);

        if MinVersionLessThan4(MinVersion) then begin
          S := IconName;
          if Copy(S, 1, 8) <> '{group}\' then
            AbortCompileOnLine (LineNumber, SCompilerIconsIconNameNoFolderConsts);
          Delete (S, 1, 8);
          if (Pos('\', S) <> 0) or (Pos('{', S) <> 0) or (Pos('}', S) <> 0) then
            AbortCompileOnLine (LineNumber, SCompilerIconsIconNameNoFolderConsts);
          if Pos('"', Parameters) <> 0 then
            { Win 3.x's Program Manager has problems with quotes }
            AbortCompileParamError (LineNumber, SCompilerParamIncludesQuotes +
              SNewLine2 + MinVersionErrorMessage(True, $400, $400),
              ParamIconsParameters);
        end;

        if Pos('"', IconName) <> 0 then
          AbortCompileParamError (LineNumber, SCompilerParamNoQuotes2, ParamIconsName);
        if Pos('\', IconName) = 0 then
          AbortCompileOnLine (LineNumber, SCompilerIconsIconNamePathNotSpecified);

        if MinVersionLessThan4(MinVersion) and ContainsBadPMChars(Parameters) then
          AbortCompileOnLine (LineNumber, SCompilerIconsParametersBadChar);
        S := IconName;
        if Copy(S, 1, 8) = '{group}\' then
          Delete (S, 1, 8);
        CheckDirConst (S, LineNumber, MinVersion, []);
        CheckDirConst (Filename, LineNumber, MinVersion, []);
        CheckDirConst (Parameters, LineNumber, MinVersion, []);
        CheckDirConst (WorkingDir, LineNumber, MinVersion, []);
        CheckDirConst (IconFilename, LineNumber, MinVersion, []);
        CheckDirConst (Comment, LineNumber, MinVersion, []);
      end;
    except
      SEFreeRec (NewIconEntry, SetupIconEntryStrings);
      raise;
    end;
    IconEntries.Add (NewIconEntry);
  finally
    Dispose (Params);
  end;
end;

procedure TSetupCompiler.EnumINI (const Line: PChar; const Ext, LineNumber: Integer);
const
  ParamIniFilename = 'Filename';
  ParamIniSection = 'Section';
  ParamIniKey = 'Key';
  ParamIniString = 'String';
  ParamNames: array[0..6] of TParamInfo = (
    (Name: ParamCommonFlags; Flags: []),
    (Name: ParamIniFilename; Flags: [piNoQuotes]),
    (Name: ParamIniSection; Flags: [piNoEmpty]),
    (Name: ParamIniKey; Flags: [piNoEmpty]),
    (Name: ParamIniString; Flags: []),
    (Name: ParamCommonMinVersion; Flags: []),
    (Name: ParamCommonOnlyBelowVersion; Flags: []));
  Flags: array[0..3] of PChar = (
    'uninsdeleteentry', 'uninsdeletesection', 'createkeyifdoesntexist',
    'uninsdeletesectionifempty');
var
  Params: PBreakStringArray;
  ParamNameFound: array[Low(ParamNames)..High(ParamNames)] of Boolean;
  NewIniEntry: PSetupIniEntry;
  P: Integer;
begin
  New (Params);
  try
    FillChar (Params^, SizeOf(Params^), 0);
    FillChar (ParamNameFound, SizeOf(ParamNameFound), 0);
    BreakString (PChar(Line), Params, LineNumber);

    NewIniEntry := AllocMem(SizeOf(TSetupIniEntry));
    try
      with NewIniEntry^ do begin
        MinVersion := SetupHeader^.MinVersion;

        for P := Low(Params^) to High(Params^) do
          with Params^[P] do begin
            if ParamName = '' then Break;
            case CompareParamName(LineNumber, Params^[P], ParamNames, ParamNameFound) of
              -1: AbortCompileOnLineFmt (LineNumber, SCompilerParamUnknownParam, [Params^[P].ParamName]);
              0: while True do
                   case ExtractFlag(ParamData, Flags) of
                     -2: Break;
                     -1: AbortCompileParamError (LineNumber, SCompilerParamUnknownFlag2, ParamCommonFlags);
                     0: Include (Options, ioUninsDeleteEntry);
                     1: Include (Options, ioUninsDeleteEntireSection);
                     2: Include (Options, ioCreateKeyIfDoesntExist);
                     3: Include (Options, ioUninsDeleteSectionIfEmpty);
                   end;
              1: Filename := ParamData;
              2: Section := ParamData;
              3: Entry := ParamData;
              4: begin
                   Value := ParamData;
                   Include (Options, ioHasValue);
                 end;
              5: if not StrToVersionNumbers(ParamData, MinVersion) then
                   AbortCompileParamError (LineNumber, SCompilerParamInvalid2, ParamCommonMinVersion);
              6: if not StrToVersionNumbers(ParamData, OnlyBelowVersion) then
                   AbortCompileParamError (LineNumber, SCompilerParamInvalid2, ParamCommonOnlyBelowVersion);
            end;
          end;

        if not ParamNameFound[1] then
          AbortCompileParamError (LineNumber, SCompilerParamNotSpecified, ParamIniFilename);
        if not ParamNameFound[2] then
          AbortCompileParamError (LineNumber, SCompilerParamNotSpecified, ParamIniSection);

        if (ioUninsDeleteEntry in Options) and
           (ioUninsDeleteEntireSection in Options) then
          AbortCompileOnLineFmt (LineNumber, SCompilerParamErrorBadCombo2,
            [ParamCommonFlags, 'uninsdeleteentry', 'uninsdeletesection']);
        if (ioUninsDeleteEntireSection in Options) and
           (ioUninsDeleteSectionIfEmpty in Options) then
          AbortCompileOnLineFmt (LineNumber, SCompilerParamErrorBadCombo2,
            [ParamCommonFlags, 'uninsdeletesection', 'uninsdeletesectionifempty']);

        CheckDirConst (Filename, LineNumber, MinVersion, []);
        CheckDirConst (Section, LineNumber, MinVersion, []);
        CheckDirConst (Entry, LineNumber, MinVersion, []);
        CheckDirConst (Value, LineNumber, MinVersion, []);
      end;
    except
      SEFreeRec (NewIniEntry, SetupIniEntryStrings);
      raise;
    end;
    IniEntries.Add (NewIniEntry);
  finally
    Dispose (Params);
  end;
end;

procedure TSetupCompiler.EnumRegistry (const Line: PChar; const Ext, LineNumber: Integer);
const
  ParamRegistryRoot = 'Root';
  ParamRegistrySubkey = 'Subkey';
  ParamRegistryValueType = 'ValueType';
  ParamRegistryValueName = 'ValueName';
  ParamRegistryValueData = 'ValueData';
  ParamNames: array[0..7] of TParamInfo = (
    (Name: ParamCommonFlags; Flags: []),
    (Name: ParamRegistryRoot; Flags: []),
    (Name: ParamRegistrySubkey; Flags: []),
    (Name: ParamRegistryValueType; Flags: []),
    (Name: ParamRegistryValueName; Flags: []),
    (Name: ParamRegistryValueData; Flags: []),
    (Name: ParamCommonMinVersion; Flags: []),
    (Name: ParamCommonOnlyBelowVersion; Flags: []));
  Flags: array[0..9] of PChar = (
    'createvalueifdoesntexist', 'uninsdeletevalue', 'uninsdeletekey',
    'uninsdeletekeyifempty', 'uninsclearvalue', 'preservestringtype',
    'deletekey', 'deletevalue', 'noerror', 'dontcreatekey');

  function ConvertBinaryString (const S: String): String;
    procedure Invalid;
    begin
      AbortCompileParamError (LineNumber, SCompilerParamInvalid2, ParamRegistryValueData);
    end;
  var
    I: Integer;
    C: Char;
    B: Byte;
    N: Integer;
    procedure EndByte;
    begin
      case N of
        0: ;
        2: begin
             Result := Result + Chr(B);
             N := 0;
             B := 0;
           end;
      else
        Invalid;
      end;
    end;
  begin
    Result := '';
    N := 0;
    B := 0;
    for I := 1 to Length(S) do begin
      C := UpCase(S[I]);
      case C of
        ' ': EndByte;
        '0'..'9': begin
               Inc (N);
               if N > 2 then
                 Invalid;
               B := (B shl 4) or (Ord(C) - Ord('0'));
             end;
        'A'..'F': begin
               Inc (N);
               if N > 2 then
                 Invalid;
               B := (B shl 4) or (10 + Ord(C) - Ord('A'));
             end;
      else
        Invalid;
      end;
    end;
    EndByte;
  end;

  function ConvertDWordString (const S: String): String;
  var
    DW: Longint;
    E: Integer;
  begin
    Val (S, DW, E);
    if E <> 0 then
      AbortCompileParamError (LineNumber, SCompilerParamInvalid2, ParamRegistryValueData);
    SetLength (Result, SizeOf(Longint));
    Longint((@Result[1])^) := DW;
  end;

var
  Params: PBreakStringArray;
  ParamNameFound: array[Low(ParamNames)..High(ParamNames)] of Boolean;
  NewRegistryEntry: PSetupRegistryEntry;
  P: Integer;
  AData: String;
begin
  New (Params);
  try
    FillChar (Params^, SizeOf(Params^), 0);
    FillChar (ParamNameFound, SizeOf(ParamNameFound), 0);
    BreakString (PChar(Line), Params, LineNumber);

    NewRegistryEntry := AllocMem(SizeOf(TSetupRegistryEntry));
    try
      with NewRegistryEntry^ do begin
        MinVersion := SetupHeader^.MinVersion;

        for P := Low(Params^) to High(Params^) do
          with Params^[P] do begin
            if ParamName = '' then Break;
            case CompareParamName(LineNumber, Params^[P], ParamNames, ParamNameFound) of
              -1: AbortCompileOnLineFmt (LineNumber, SCompilerParamUnknownParam, [Params^[P].ParamName]);
              0: while True do
                   case ExtractFlag(ParamData, Flags) of
                     -2: Break;
                     -1: AbortCompileParamError (LineNumber, SCompilerParamUnknownFlag2, ParamCommonFlags);
                     0: Include (Options, roCreateValueIfDoesntExist);
                     1: Include (Options, roUninsDeleteValue);
                     2: Include (Options, roUninsDeleteEntireKey);
                     3: Include (Options, roUninsDeleteEntireKeyIfEmpty);
                     4: Include (Options, roUninsClearValue);
                     5: Include (Options, roPreserveStringType);
                     6: Include (Options, roDeleteKey);
                     7: Include (Options, roDeleteValue);
                     8: Include (Options, roNoError);
                     9: Include (Options, roDontCreateKey);
                   end;
              1: begin
                   ParamData := Uppercase(Trim(ParamData));
                   if ParamData = 'HKCR' then
                     RootKey := HKEY_CLASSES_ROOT
                   else
                   if ParamData = 'HKCU' then
                     RootKey := HKEY_CURRENT_USER
                   else
                   if ParamData = 'HKLM' then
                     RootKey := HKEY_LOCAL_MACHINE
                   else
                   if ParamData = 'HKU' then
                     RootKey := HKEY_USERS
                   else
                   if ParamData = 'HKCC' then
                     RootKey := HKEY_CURRENT_CONFIG
                   else
                     AbortCompileParamError (LineNumber, SCompilerParamInvalid2, ParamRegistryRoot);
                 end;
              2: begin
                   if (ParamData <> '') and (ParamData[1] = '\') then
                     AbortCompileParamError (LineNumber, SCompilerParamNoPrecedingBackslash, ParamRegistrySubkey);
                   Subkey := ParamData;
                 end;
              3: begin
                   ParamData := Uppercase(Trim(ParamData));
                   if ParamData = 'NONE' then
                     Typ := rtNone
                   else if ParamData = 'STRING' then
                     Typ := rtString
                   else if ParamData = 'EXPANDSZ' then
                     Typ := rtExpandString
                   else if ParamData = 'MULTISZ' then
                     Typ := rtMultiString
                   else if ParamData = 'DWORD' then
                     Typ := rtDWord
                   else if ParamData = 'BINARY' then
                     Typ := rtBinary
                   else
                     AbortCompileParamError (LineNumber, SCompilerParamInvalid2, ParamRegistryValueType);
                 end;
              4: ValueName := ParamData;
              5: AData := ParamData;
              6: if not StrToVersionNumbers(ParamData, MinVersion) then
                   AbortCompileParamError (LineNumber, SCompilerParamInvalid2, ParamCommonMinVersion);
              7: if not StrToVersionNumbers(ParamData, OnlyBelowVersion) then
                   AbortCompileParamError (LineNumber, SCompilerParamInvalid2, ParamCommonOnlyBelowVersion);
            end;
          end;

        if not ParamNameFound[1] then
          AbortCompileParamError (LineNumber, SCompilerParamNotSpecified, ParamRegistryRoot);
        if not ParamNameFound[2] then
          AbortCompileParamError (LineNumber, SCompilerParamNotSpecified, ParamRegistrySubkey);
        if (roUninsDeleteEntireKey in Options) and
           (roUninsDeleteEntireKeyIfEmpty in Options) then
          AbortCompileOnLineFmt (LineNumber, SCompilerParamErrorBadCombo2,
            [ParamCommonFlags, 'uninsdeletekey', 'uninsdeletekeyifempty']);
        if (roUninsDeleteEntireKey in Options) and
           (roUninsClearValue in Options) then
          AbortCompileOnLineFmt (LineNumber, SCompilerParamErrorBadCombo2,
            [ParamCommonFlags, 'uninsclearvalue', 'uninsdeletekey']);
        if (roUninsDeleteValue in Options) and
           (roUninsDeleteEntireKey in Options) then
          AbortCompileOnLineFmt (LineNumber, SCompilerParamErrorBadCombo2,
            [ParamCommonFlags, 'uninsdeletevalue', 'uninsdeletekey']);
        if (roUninsDeleteValue in Options) and
           (roUninsClearValue in Options) then
          AbortCompileOnLineFmt (LineNumber, SCompilerParamErrorBadCombo2,
            [ParamCommonFlags, 'uninsdeletevalue', 'uninsclearvalue']);

        case Typ of
          rtString, rtExpandString, rtMultiString:
            ValueData := AData;
          rtDWord:
            ValueData := ConvertDWordString(AData);
          rtBinary:
            ValueData := ConvertBinaryString(AData);
        end;

        CheckDirConst (Subkey, LineNumber, MinVersion, []);
        CheckDirConst (ValueName, LineNumber, MinVersion, []);
        case Typ of
          rtString, rtExpandString:
            CheckDirConst (ValueData, LineNumber, MinVersion, [acOldData]);
          rtMultiString:
            CheckDirConst (ValueData, LineNumber, MinVersion, [acOldData, acBreak]);
        end;
      end;
    except
      SEFreeRec (NewRegistryEntry, SetupRegistryEntryStrings);
      raise;
    end;
    RegistryEntries.Add (NewRegistryEntry);
  finally
    Dispose (Params);
  end;
end;

procedure TSetupCompiler.EnumDelete (const Line: PChar; const Ext, LineNumber: Integer);
const
  ParamDeleteType = 'Type';
  ParamDeleteName = 'Name';
  ParamNames: array[0..3] of TParamInfo = (
    (Name: ParamDeleteType; Flags: []),
    (Name: ParamDeleteName; Flags: [piNoEmpty]),
    (Name: ParamCommonMinVersion; Flags: []),
    (Name: ParamCommonOnlyBelowVersion; Flags: []));
  Types: array[TSetupDeleteType] of PChar = (
    'files', 'filesandordirs', 'dirifempty');
var
  Params: PBreakStringArray;
  ParamNameFound: array[Low(ParamNames)..High(ParamNames)] of Boolean;
  NewDeleteEntry: PSetupDeleteEntry;
  P: Integer;
  Valid: Boolean;
  J: TSetupDeleteType;
begin
  New (Params);
  try
    FillChar (Params^, SizeOf(Params^), 0);
    FillChar (ParamNameFound, SizeOf(ParamNameFound), 0);
    BreakString (PChar(Line), Params, LineNumber);

    NewDeleteEntry := AllocMem(SizeOf(TSetupDeleteEntry));
    try
      with NewDeleteEntry^ do begin
        MinVersion := SetupHeader^.MinVersion;

        for P := Low(Params^) to High(Params^) do
          with Params^[P] do begin
            if ParamName = '' then Break;
            case CompareParamName(LineNumber, Params^[P], ParamNames, ParamNameFound) of
              -1: AbortCompileOnLineFmt (LineNumber, SCompilerParamUnknownParam, [Params^[P].ParamName]);
              0: begin
                   ParamData := Trim(ParamData);
                   Valid := False;
                   for J := Low(J) to High(J) do
                     if StrIComp(Types[J], StringAsPChar(ParamData)) = 0 then begin
                       DeleteType := J;
                       Valid := True;
                       Break;
                     end;
                   if not Valid then
                     AbortCompileParamError (LineNumber, SCompilerParamInvalid2, ParamDeleteType);
                 end;
              1: Name := ParamData;
              2: if not StrToVersionNumbers(ParamData, MinVersion) then
                   AbortCompileParamError (LineNumber, SCompilerParamInvalid2, ParamCommonMinVersion);
              3: if not StrToVersionNumbers(ParamData, OnlyBelowVersion) then
                   AbortCompileParamError (LineNumber, SCompilerParamInvalid2, ParamCommonOnlyBelowVersion);
            end;
          end;

        if not ParamNameFound[0] then
          AbortCompileParamError (LineNumber, SCompilerParamNotSpecified, ParamDeleteType);
        if not ParamNameFound[1] then
          AbortCompileParamError (LineNumber, SCompilerParamNotSpecified, ParamDeleteName);

        CheckDirConst (Name, LineNumber, MinVersion, []);
      end;
    except
      SEFreeRec (NewDeleteEntry, SetupDeleteEntryStrings);
      raise;
    end;
    if Ext = 0 then
      InstallDeleteEntries.Add (NewDeleteEntry)
    else
      UninstallDeleteEntries.Add (NewDeleteEntry);
  finally
    Dispose (Params);
  end;
end;

type
  PExtraData = ^TExtraData;
  TExtraData = record
    SourceFile, DestFile: TFileStream;
  end;

function ReadProc (var Buf; MaxBytes: Cardinal; ExtraData: Longint): Cardinal; far;
var
  Bytes: Cardinal;
begin
  with PExtraData(ExtraData)^ do begin
    Bytes := SourceFile.Size - SourceFile.Position;
    if Bytes > MaxBytes then Bytes := MaxBytes;
    SourceFile.ReadBuffer (Buf, Bytes);
    Result := Bytes;
  end;
end;

function WriteProc (var Buf; BufSize: Cardinal; ExtraData: Longint): Cardinal; far;
begin
  PExtraData(ExtraData)^.DestFile.WriteBuffer (Buf, BufSize);
  Result := BufSize;
end;

procedure TSetupCompiler.EnumFiles (const Line: PChar; const Ext, LineNumber: Integer);

  function GetVersionInfo (Filename: String; var VersionInfo: TVSFixedFileInfo): Boolean;
  var
    VersionSize: Integer;
    VersionHandle: DWORD;
    VersionBuf: PChar;
    VerInfo: PVSFixedFileInfo;
    VerInfoSize: UINT;
  begin
    Result := False;

    VersionSize := GetFileVersionInfoSize(StringAsPChar(Filename),
      VersionHandle);
    if VersionSize <> 0 then begin
      GetMem (VersionBuf, VersionSize);
      try
        if GetFileVersionInfo(StringAsPChar(Filename), VersionHandle, VersionSize, VersionBuf) then begin
          if VerQueryValue(VersionBuf, '\', Pointer(VerInfo), VerInfoSize) then begin
            VersionInfo := VerInfo^;
            Result := True;
          end;
        end;
      finally
        FreeMem (VersionBuf, VersionSize);
      end;
    end;
  end;

  function EscapeBraces (const S: String): String;
  { Changes all '{' to '{{' }
  var
    I: Integer;
  begin
    Result := S;
    I := 1;
    while I <= Length(Result) do begin
      if Result[I] = '{' then begin
        Insert ('{', Result, I);
        Inc (I);
      end;
      Inc (I);
    end;
  end;

const
  ParamFilesSource = 'Source';
  ParamFilesDestDir = 'DestDir';
  ParamFilesDestName = 'DestName';
  ParamFilesCopyMode = 'CopyMode';
  ParamFilesAttribs = 'Attribs';
  ParamFilesFontInstall = 'FontInstall';
  ParamNames: array[0..8] of TParamInfo = (
    (Name: ParamCommonFlags; Flags: []),
    (Name: ParamFilesSource; Flags: [piNoEmpty, piNoQuotes]),
    (Name: ParamFilesDestDir; Flags: [piNoEmpty, piNoQuotes]),
    (Name: ParamFilesDestName; Flags: [piNoEmpty, piNoQuotes]),
    (Name: ParamFilesCopyMode; Flags: []),
    (Name: ParamFilesAttribs; Flags: []),
    (Name: ParamFilesFontInstall; Flags: [piNoEmpty]),
    (Name: ParamCommonMinVersion; Flags: []),
    (Name: ParamCommonOnlyBelowVersion; Flags: []));
  Flags: array[0..13] of PChar = (
    'confirmoverwrite', 'uninsneveruninstall', 'isreadme', 'regserver',
    'sharedfile', 'restartreplace', 'deleteafterinstall',
    'comparetimestampalso', 'fontisnttruetype', 'regtypelib', 'external',
    'skipifsourcedoesntexist', 'overwritereadonly', 'onlyifdestfileexists');
  CopyModes: array[TSetupFileCopyMode] of PChar = (
    'normal', 'onlyifdoesntexist', 'alwaysoverwrite',
    'alwaysskipifsameorolder');
  AttribsFlags: array[0..2] of PChar = (
    'readonly', 'hidden', 'system');
var
  Params: PBreakStringArray;
  ParamNameFound: array[Low(ParamNames)..High(ParamNames)] of Boolean;
  NewFileEntry, OldFileEntry: PSetupFileEntry;
  NewFileLocationEntry: PSetupFileLocationEntry;
  P: Integer;
  VersionInfo: TVSFixedFileInfo;
  J: TSetupFileCopyMode;
  ValidCopyMode: Boolean;
  SourceWildcard, SourceFile, ADestDir, ADestName, AInstallFontName: String;
  ExternalFile, SourceIsWildcard: Boolean;
  SearchRec: TSearchRec;
begin
  CallIdleProc;

  New (Params);
  try
    FillChar (Params^, SizeOf(Params^), 0);
    FillChar (ParamNameFound, SizeOf(ParamNameFound), 0);
    if Ext = 0 then
      BreakString (PChar(Line), Params, LineNumber);

    NewFileEntry := nil;
    try
      NewFileEntry := AllocMem(SizeOf(TSetupFileEntry));
      with NewFileEntry^ do begin
        MinVersion := SetupHeader^.MinVersion;
        CopyMode := cmNormal;

        ADestName := '';
        ADestDir := '';
        AInstallFontName := '';
        ExternalFile := False;

        case Ext of
          0: for P := Low(Params^) to High(Params^) do
               with Params^[P] do begin
                 if ParamName = '' then Break;
                 case CompareParamName(LineNumber, Params^[P], ParamNames, ParamNameFound) of
                   -1: AbortCompileOnLineFmt (LineNumber, SCompilerParamUnknownParam, [Params^[P].ParamName]);
                   0: while True do
                        case ExtractFlag(ParamData, Flags) of
                          -2: Break;
                          -1: AbortCompileParamError (LineNumber, SCompilerParamUnknownFlag2, ParamCommonFlags);
                          0: Include (Options, foConfirmOverwrite);
                          1: Include (Options, foUninsNeverUninstall);
                          2: Include (Options, foIsReadmeFile);
                          3: Include (Options, foRegisterServer);
                          4: Include (Options, foSharedFile);
                          5: Include (Options, foRestartReplace);
                          6: Include (Options, foDeleteAfterInstall);
                          7: Include (Options, foCompareTimeStampAlso);
                          8: Include (Options, foFontIsntTrueType);
                          9: Include (Options, foRegisterTypeLib);
                          10: ExternalFile := True;
                          11: Include (Options, foSkipIfSourceDoesntExist);
                          12: Include (Options, foOverwriteReadOnly);
                          13: Include (Options, foOnlyIfDestFileExists);
                        end;
                   1: SourceWildcard := ParamData;
                   2: ADestDir := ParamData;
                   3: begin
                        if Pos('\', ParamData) <> 0 then
                          AbortCompileParamError (LineNumber, SCompilerParamNoBackslash, ParamFilesDestName);
                        ADestName := ParamData;
                      end;
                   4: begin
                        ParamData := Trim(ParamData);
                        ValidCopyMode := False;
                        for J := Low(J) to High(J) do
                          if StrIComp(CopyModes[J], StringAsPChar(ParamData)) = 0 then begin
                            CopyMode := J;
                            ValidCopyMode := True;
                            Break;
                          end;
                        if not ValidCopyMode then
                          AbortCompileParamError (LineNumber, SCompilerParamInvalid2, ParamFilesCopyMode);
                      end;
                   5: while True do
                        case ExtractFlag(ParamData, AttribsFlags) of
                          -2: Break;
                          -1: AbortCompileParamError (LineNumber, SCompilerParamUnknownFlag2, ParamFilesAttribs);
                          0: Attribs := Attribs or faReadOnly;
                          1: Attribs := Attribs or faHidden;
                          2: Attribs := Attribs or faSysFile;
                        end;
                   6: AInstallFontName := ParamData;
                   7: if not StrToVersionNumbers(ParamData, MinVersion) then
                        AbortCompileParamError (LineNumber, SCompilerParamInvalid2, ParamCommonMinVersion);
                   8: if not StrToVersionNumbers(ParamData, OnlyBelowVersion) then
                        AbortCompileParamError (LineNumber, SCompilerParamInvalid2, ParamCommonOnlyBelowVersion);
                 end;
               end;
          1: begin
               SourceWildcard := 'compiler:UNINST.E32';
               FileType := ftUninstExe;
               CopyMode := cmAlwaysSkipIfSameOrOlder;
               Options := [foOverwriteSameVersion];
             end;
          2: begin
               SourceWildcard := 'compiler:REGSVR.E32';
               FileType := ftRegSvrExe;
               Options := [foUninsNeverUninstall];
             end;
        end;

        if Ext = 0 then begin
          if not ParamNameFound[1] then
            AbortCompileParamError (LineNumber, SCompilerParamNotSpecified, ParamFilesSource);
          if not ParamNameFound[2] then
            AbortCompileParamError (LineNumber, SCompilerParamNotSpecified, ParamFilesDestDir);
        end;

        if (ADestDir = '{tmp}') or (Copy(ADestDir, 1, 4) = '{tmp}\') then
          Include (Options, foDeleteAfterInstall);
        if foDeleteAfterInstall in Options then begin
          if foIsReadmeFile in Options then
            AbortCompileOnLineFmt (LineNumber, SCompilerFilesTmpBadFlag,
              ['isreadme']);
          if foRestartReplace in Options then
            AbortCompileOnLineFmt (LineNumber, SCompilerFilesTmpBadFlag,
              ['restartreplace']);
          if foUninsNeverUninstall in Options then
            AbortCompileOnLineFmt (LineNumber, SCompilerFilesTmpBadFlag,
              ['uninsneveruninstall']);
          if foRegisterServer in Options then
            AbortCompileOnLineFmt (LineNumber, SCompilerFilesTmpBadFlag,
              ['regserver']);
          if foRegisterTypeLib in Options then
            AbortCompileOnLineFmt (LineNumber, SCompilerFilesTmpBadFlag,
              ['regtypelib']);
          if foSharedFile in Options then
            AbortCompileOnLineFmt (LineNumber, SCompilerFilesTmpBadFlag,
              ['sharedfile']);
          Include (Options, foUninsNeverUninstall);
        end;

        if (foRestartReplace in Options) and (foRegisterServer in Options) and
           MinVersionLessThan4(MinVersion) then
          AbortCompileOnLineFmt (LineNumber, SCompilerParamErrorComboUsed +
            SNewLine2 + MinVersionErrorMessage(True, $400, $400),
            [ParamCommonFlags, 'restartreplace', 'registerserver']);
        if (shAlwaysRestart in SetupHeader^.Options) or (foRestartReplace in Options) then
          HasRestart := True;
        if (foRegisterServer in Options) or (foRegisterTypeLib in Options) then
          HasRegSvr := True;

        if AInstallFontName <> '' then begin
          if not(foFontIsntTrueType in Options) then
            AInstallFontName := AInstallFontName + ' (TrueType)';
          InstallFontName := AInstallFontName;
        end;

        SourceIsWildcard := (Ext = 0) and ((Pos('*', SourceWildcard) <> 0)
          or (Pos('?', SourceWildcard) <> 0));
        if ExternalFile then begin
          if Pos('\', SourceWildcard) = 0 then
            AbortCompileOnLine (LineNumber, SCompilerFilesSourcePathNotSpecified);
          CheckDirConst (SourceWildcard, LineNumber, MinVersion, []);
        end;
        if (ADestName <> '') and SourceIsWildcard then
          AbortCompileOnLine (LineNumber, SCompilerFilesDestNameCantBeSpecified);
        CheckDirConst (ADestDir, LineNumber, MinVersion, []);
        CheckDirConst (ADestName, LineNumber, MinVersion, []);
        if not ExternalFile then begin
          SourceWildcard := PrependSourceDirName(SourceWildcard);
          if (Ext = 0) and not SourceIsWildcard and not FileExists(SourceWildcard) then
            AbortCompileOnLineFmt (LineNumber, SCompilerSourceFileDoesntExist, [SourceWildcard]);
        end;
      end;

      if ExternalFile then
        SourceIsWildcard := False;  { don't do wildcard expansion at compile time }
      if not SourceIsWildcard or
         (FindFirst(SourceWildcard, 0, SearchRec) = 0) then begin
        try
          while True do begin
            if SourceIsWildcard then begin
              SourceFile := ExtractFilePath(SourceWildcard) + SearchRec.Name;
              NewFileEntry^.DestName := AddBackslash(ADestDir) + EscapeBraces(SearchRec.Name);
            end
            else begin
              SourceFile := SourceWildcard;
              if Ext = 0 then begin
                if ADestName = '' then begin
                  if not ExternalFile then
                    NewFileEntry^.DestName := AddBackslash(ADestDir) + EscapeBraces(ExtractFileName(SourceWildcard))
                  else
                    NewFileEntry^.DestName := AddBackslash(ADestDir) + ExtractFileName(SourceWildcard);
                end
                else begin
                  NewFileEntry^.DestName := AddBackslash(ADestDir) + ADestName;
                    { ^ user is already required to escape '{' in DestName }
                  Include (NewFileEntry^.Options, foCustomDestName);
                end;
              end
              else
                NewFileEntry^.DestName := '';
            end;

            NewFileLocationEntry := nil;
            if not ExternalFile then begin
              if not DontMergeDuplicateFiles then
                for P := 0 to FileLocationEntryFilenames.Count-1 do
                  { See if the source filename is already in the list of files to
                    be compressed. If so, don't duplicate it. }
                  if {$IFDEF IS_D3} AnsiCompareFileName {$ELSE} AnsiCompareText {$ENDIF}
                     (FileLocationEntryFilenames[P], SourceFile) = 0 then begin
                    NewFileLocationEntry := FileLocationEntries[P];
                    NewFileEntry^.LocationEntry := P;
                    Break;
                  end;
              if NewFileLocationEntry = nil then begin
                NewFileLocationEntry := AllocMem(SizeOf(TSetupFileLocationEntry));
                FileLocationEntries.Add (NewFileLocationEntry);
                FileLocationEntryFilenames.Add (SourceFile);
                NewFileEntry^.LocationEntry := FileLocationEntries.Count-1;
              end;
            end
            else begin
              NewFileEntry^.SourceFilename := SourceWildcard;
              NewFileEntry^.LocationEntry := -1;
            end;

            { Read version info }
            if not ExternalFile and (NewFileEntry^.CopyMode <> cmAlwaysOverwrite) and
               (NewFileLocationEntry^.Flags * [foVersionInfoValid, foVersionInfoNotValid] = []) then begin
              AddStatus (Format(SCompilerStatusFilesVerInfo, [SourceFile]));
              if GetVersionInfo(SourceFile, VersionInfo) then begin
                NewFileLocationEntry^.FileVersionMS := VersionInfo.dwFileVersionMS;
                NewFileLocationEntry^.FileVersionLS := VersionInfo.dwFileVersionLS;
                Include (NewFileLocationEntry^.Flags, foVersionInfoValid);
              end
              else
                Include (NewFileLocationEntry^.Flags, foVersionInfoNotValid);
            end;

            FileEntries.Add (NewFileEntry);

            if not SourceIsWildcard or (FindNext(SearchRec) <> 0) then
              Break;

            OldFileEntry := NewFileEntry;
            NewFileEntry := nil;
            NewFileEntry := AllocMem(SizeOf(TSetupFileEntry));
            SEDuplicateRec (OldFileEntry, NewFileEntry,
              SizeOf(TSetupFileEntry), SetupFileEntryStrings);
          end;
        finally
          if SourceIsWildcard then
            FindClose (SearchRec);
        end;
      end
      else
        AbortCompileOnLineFmt (LineNumber, SCompilerFilesWildcardNotMatched, [SourceWildcard]);
    except
      SEFreeRec (NewFileEntry, SetupFileEntryStrings);
      raise;
    end;
  finally
    Dispose (Params);
  end;
end;

procedure UpdateTimeStamp (H: THandle);
var
  FT: TFileTime;
begin
  GetSystemTimeAsFileTime (FT);
  SetFileTime (H, nil, nil, @FT);
end;

procedure TSetupCompiler.EnumRun (const Line: PChar; const Ext, LineNumber: Integer);
const
  ParamRunFilename = 'Filename';
  ParamRunParameters = 'Parameters';
  ParamRunWorkingDir = 'WorkingDir';
  ParamRunRunOnceId = 'RunOnceId';
  ParamNames: array[0..6] of TParamInfo = (
    (Name: ParamCommonFlags; Flags: []),
    (Name: ParamRunFilename; Flags: [piNoEmpty, piNoQuotes]),
    (Name: ParamRunParameters; Flags: []),
    (Name: ParamRunWorkingDir; Flags: []),
    (Name: ParamRunRunOnceId; Flags: []),
    (Name: ParamCommonMinVersion; Flags: []),
    (Name: ParamCommonOnlyBelowVersion; Flags: []));
  Flags: array[0..5] of PChar = (
    'nowait', 'waituntilidle', 'shellexec', 'skipifdoesntexist',
    'runminimized', 'runmaximized');
var
  Params: PBreakStringArray;
  ParamNameFound: array[Low(ParamNames)..High(ParamNames)] of Boolean;
  NewRunEntry: PSetupRunEntry;
  P: Integer;
begin
  New (Params);
  try
    FillChar (Params^, SizeOf(Params^), 0);
    FillChar (ParamNameFound, SizeOf(ParamNameFound), 0);
    BreakString (PChar(Line), Params, LineNumber);

    NewRunEntry := AllocMem(SizeOf(TSetupRunEntry));
    try
      with NewRunEntry^ do begin
        MinVersion := SetupHeader^.MinVersion;
        ShowCmd := SW_SHOWNORMAL;

        for P := Low(Params^) to High(Params^) do
          with Params^[P] do begin
            if ParamName = '' then Break;
            case CompareParamName(LineNumber, Params^[P], ParamNames, ParamNameFound) of
              -1: AbortCompileOnLineFmt (LineNumber, SCompilerParamUnknownParam, [Params^[P].ParamName]);
              0: while True do
                   case ExtractFlag(ParamData, Flags) of
                     -2: Break;
                     -1: AbortCompileParamError (LineNumber, SCompilerParamUnknownFlag2, ParamCommonFlags);
                     0: Wait := rwNoWait;
                     1: Wait := rwWaitUntilIdle;
                     2: Include (Options, roShellExec);
                     3: Include (Options, roSkipIfDoesntExist);
                     4: ShowCmd := SW_SHOWMINNOACTIVE;
                     5: ShowCmd := SW_SHOWMAXIMIZED;
                   end;
              1: Name := ParamData;
              2: Parameters := ParamData;
              3: WorkingDir := ParamData;
              4: begin
                   if (Ext = 0) and (ParamData <> '') then
                     AbortCompileOnLine (LineNumber, SCompilerRunCantUseRunOnceId);
                   RunOnceId := ParamData;
                 end;
              5: if not StrToVersionNumbers(ParamData, MinVersion) then
                   AbortCompileParamError (LineNumber, SCompilerParamInvalid2, ParamCommonMinVersion);
              6: if not StrToVersionNumbers(ParamData, OnlyBelowVersion) then
                   AbortCompileParamError (LineNumber, SCompilerParamInvalid2, ParamCommonOnlyBelowVersion);
            end;
          end;

        CheckDirConst (Name, LineNumber, MinVersion, []);
        CheckDirConst (Parameters, LineNumber, MinVersion, []);
        CheckDirConst (WorkingDir, LineNumber, MinVersion, []);
        CheckDirConst (RunOnceId, LineNumber, MinVersion, []);
      end;
    except
      SEFreeRec (NewRunEntry, SetupRunEntryStrings);
      raise;
    end;
    if Ext = 0 then
      RunEntries.Add (NewRunEntry)
    else
      UninstallRunEntries.Add (NewRunEntry);
  finally
    Dispose (Params);
  end;
end;

procedure TSetupCompiler.EnumMessages (const Line: PChar; const Ext, LineNumber: Integer);
var
  P, M: PChar;
  I {, L}, LineNum: Integer;
  N: String;
begin
  LineNum := LineNumber;
  P := StrScan(Line, '=');
  if P = nil then
    AbortCompileOnLine (LineNum, SCompilerMessagesMissingEquals);
  SetLength (N, P - Line);
  Move (Line^, N[1], Length(N));
  I := GetEnumValue(TypeInfo(TSetupMessageID), 'msg' + N);
  if I = -1 then
    AbortCompileOnLineFmt (LineNum, SCompilerMessagesNotRecognized, [N]);
  M := StrNew(P + 1);
  { Replace %n with actual CR/LF characters }
  P := M;
  while True do begin
    P := StrPos(P, '%n');
    if P = nil then Break;
    P[0] := #13;
    P[1] := #10;
    Inc (P, 2);
  end;
  { Replace %% with % }
  {}(* this isn't used anymore
  P := M;
  while True do begin
    P := StrPos(P, '%%');
    if P = nil then Break;
    Inc (P);
    L := StrLen(P);
    Move (P^, (P-1)^, L);
    P[L-1] := #0;
  end;*)
  if Assigned(Messages[TSetupMessageID(I)]) then begin
    StrDispose (Messages[TSetupMessageID(I)]);
    Messages[TSetupMessageID(I)] := nil;
  end;
  Messages[TSetupMessageID(I)] := M;
end;

procedure TSetupCompiler.ReadMessages;
var
  S, Filename, SaveAbortFilename: String;
  P: Integer;
  F: TextFile;
  I: TSetupMessageID;
begin
  { Read in all the message files }
  S := MessagesFile;
  while True do begin
    P := Pos(',', S);
    if P = 0 then P := Maxint;
    Filename := Trim(Copy(S, 1, P-1));
    if Filename = '' then
      Break;
    Filename := PrependSourceDirName(Filename);
    AddStatus (Format(SCompilerStatusReadingInFile, [Filename]));
    SaveAbortFilename := AbortFilename;
    AssignFile (F, Filename);
    FileMode := fmOpenRead or fmShareDenyWrite;  Reset (F);
    try
      AbortFilename := Filename;
      EnumIniSection (EnumMessages, 'Messages', 1, False, F);
      CallIdleProc;
    finally
      AbortFilename := SaveAbortFilename;
      CloseFile (F);
    end;
    Delete (S, 1, P);
  end;
  { Then read the [Messages] section }
  AddStatus (SCompilerStatusReadingInScriptMsgs);
  EnumIniSection (EnumMessages, 'Messages', 0, False, TextFile(nil^));
  CallIdleProc;
  { Check messages }
  for I := Low(Messages) to High(Messages) do
    if Messages[I] = nil then
      if (I >= msg_CopyrightFont) and (I <= msg_TitleFont) then
        Messages[I] := StrNew('')
      else
        AbortCompileFmt (SCompilerMessagesMissingMessage,
          [Copy(GetEnumName(TypeInfo(TSetupMessageID), Ord(I)), 4, Maxint)]);
          { ^ Copy(..., 4, Maxint) is to skip past "msg" }
  CallIdleProc;
end;

procedure TSetupCompiler.Compile;
  procedure EmptyOutputDir;
    procedure DelFile (const Filename: String);
    begin
      if DeleteFile(OutputDir + Filename) then
        AddStatus (Format(SCompilerStatusDeletingPrevious, [Filename]));
    end;
  var
    SR: TSearchRec;
    Ext: String;
    I: Integer;
    {FoundFiles,} AllNumbers: Boolean;
  begin
    { Delete SETUP.* if they existed in the output directory }
    DelFile (OutputBaseFilename + '.exe');
    DelFile ('SETUP.TMP');
    DelFile ('SETUP.MSG');
    if FindFirst(OutputDir + OutputBaseFilename + '.*', 0, SR) = 0 then begin
      repeat
        if SR.Attr and faDirectory = 0 then begin
          Ext := ExtractFileExt(SR.Name);
          Delete (Ext, 1, 1);
          AllNumbers := False;
          for I := 1 to Length(Ext) do
            if Ext[I] in ['0'..'9'] then
              AllNumbers := True
            else begin
              AllNumbers := False;
              Break;
            end;
          if AllNumbers then
            DelFile (SR.Name);
        end;
      until FindNext(SR) <> 0;
      FindClose (SR);
    end;

    (*
    { Make sure output dir is empty of other files }
    FoundFiles := False;
    if FindFirst(OutputDir + '*.*', faAnyFile, SR) = 0 then begin
      { must check result first due to Delphi bug }
      repeat
        if SR.Attr and faDirectory = 0 then begin
          FoundFiles := True;
          Break;
        end;
      until FindNext(SR) <> 0;
      FindClose (SR);
    end;
    if FoundFiles then
      AbortCompile (SCompilerOutputNotEmpty2);
    *)
  end;
  procedure ReadTextFile (const Filename: String; var Text: String);
  var
    F: File;
    TextSize: Cardinal;
  begin
    AssignFile (F, Filename);
    FileMode := fmOpenRead or fmShareDenyWrite;  Reset (F, 1);
    try
      TextSize := FileSize(F);
      //if TextSize > 32000 then
      //  AbortCompile (SCompilerFileTooBig);
      SetLength (Text, TextSize);
      BlockRead (F, Text[1], TextSize);
    finally
      CloseFile (F);
    end;
  end;
  procedure FreeListItems (const List: TList; const NumStrings: Integer);
  var
    I: Integer;
  begin
    for I := List.Count-1 downto 0 do begin
      SEFreeRec (List[I], NumStrings);
      List.Delete (I);
    end;
  end;
  procedure FreeMessages;
  var
    I: TSetupMessageID;
  begin
    for I := High(Messages) downto Low(Messages) do begin
      StrDispose (Messages[I]);
      Messages[I] := nil;
    end;
  end;
type
  PCopyBuffer = ^TCopyBuffer;
  TCopyBuffer = array[0..32767] of Char;
var
  SetupFile: File;
  ConvertFile, ExeFile: File;
  LicenseText, InfoBeforeText, InfoAfterText: String;
  WizardImage: TMemoryStream;

  SetupPrg: String;

  SetupLdrExeHeader: TSetupLdrExeHeader;
  SetupLdrOffsetTable: TSetupLdrOffsetTable;
  SizeOfExe, SizeOfHeaders: Longint;
  DiskHdr: TDiskHeader;

  function WriteSetup0 (var F: File): Longint;
    procedure WriteWizardImage (var Data: TDeflateBlockWriteData);
    type
      PBuffer = ^TBuffer;
      TBuffer = array[0..8191] of Byte;
    var
      Buf: PBuffer;
      BytesLeft, Bytes: Longint;
    begin
      New (Buf);
      try
        WizardImage.Seek (0, soFromBeginning);
        BytesLeft := WizardImage.Size;
        DeflateBlockWrite (Data, BytesLeft, SizeOf(BytesLeft));
        while BytesLeft > 0 do begin
          Bytes := BytesLeft;
          if Bytes > SizeOf(Buf^) then Bytes := SizeOf(Buf^);
          WizardImage.ReadBuffer (Buf^, Bytes);
          DeflateBlockWrite (Data, Buf^, Bytes);
          Dec (BytesLeft, Bytes);
        end;
      finally
        Dispose (Buf);
      end;
    end;
  var
    Pos: Longint;
    J: Integer;
    Data: TDeflateBlockWriteData;
  begin
    Pos := FilePos(F);

    BlockWrite (F, SetupID, SizeOf(SetupID));

    SetupHeader^.NumDirEntries := DirEntries.Count;
    SetupHeader^.NumFileEntries := FileEntries.Count;
    SetupHeader^.NumFileLocationEntries := FileLocationEntries.Count;
    SetupHeader^.NumIconEntries := IconEntries.Count;
    SetupHeader^.NumIniEntries := IniEntries.Count;
    SetupHeader^.NumRegistryEntries := RegistryEntries.Count;
    SetupHeader^.NumInstallDeleteEntries := InstallDeleteEntries.Count;
    SetupHeader^.NumUninstallDeleteEntries := UninstallDeleteEntries.Count;
    SetupHeader^.NumRunEntries := RunEntries.Count;
    SetupHeader^.NumUninstallRunEntries := UninstallRunEntries.Count;
    SetupHeader^.LicenseText := LicenseText;
    SetupHeader^.InfoBeforeText := InfoBeforeText;
    SetupHeader^.InfoAfterText := InfoAfterText;

    DeflateBlockWriteBegin (F, CompressLevel, Data);
    try
      SEDeflateBlockWrite (Data, SetupHeader^, SizeOf(TSetupHeader),
        SetupHeaderStrings);

      WriteWizardImage (Data);

      for J := 0 to DirEntries.Count-1 do
        SEDeflateBlockWrite (Data, DirEntries[J]^, SizeOf(TSetupDirEntry),
          SetupDirEntryStrings);
      for J := 0 to FileEntries.Count-1 do
        SEDeflateBlockWrite (Data, FileEntries[J]^, SizeOf(TSetupFileEntry),
          SetupFileEntryStrings);
      for J := 0 to IconEntries.Count-1 do
        SEDeflateBlockWrite (Data, IconEntries[J]^, SizeOf(TSetupIconEntry),
          SetupIconEntryStrings);
      for J := 0 to IniEntries.Count-1 do
        SEDeflateBlockWrite (Data, IniEntries[J]^, SizeOf(TSetupIniEntry),
          SetupIniEntryStrings);
      for J := 0 to RegistryEntries.Count-1 do
        SEDeflateBlockWrite (Data, RegistryEntries[J]^, SizeOf(TSetupRegistryEntry),
          SetupRegistryEntryStrings);
      for J := 0 to InstallDeleteEntries.Count-1 do
        SEDeflateBlockWrite (Data, InstallDeleteEntries[J]^, SizeOf(TSetupDeleteEntry),
          SetupDeleteEntryStrings);
      for J := 0 to UninstallDeleteEntries.Count-1 do
        SEDeflateBlockWrite (Data, UninstallDeleteEntries[J]^, SizeOf(TSetupDeleteEntry),
          SetupDeleteEntryStrings);
      for J := 0 to RunEntries.Count-1 do
        SEDeflateBlockWrite (Data, RunEntries[J]^, SizeOf(TSetupRunEntry),
          SetupRunEntryStrings);
      for J := 0 to UninstallRunEntries.Count-1 do
        SEDeflateBlockWrite (Data, UninstallRunEntries[J]^, SizeOf(TSetupRunEntry),
          SetupRunEntryStrings);

      DeflateBlockWriteEnd (Data);
    except
      DeflateBlockWriteCancel (Data);
      raise;
    end;

    if not DiskSpanning then
      DeflateBlockWriteBegin (F, CompressLevel, Data)
    else
      DeflateBlockWriteBegin (F, Z_NO_COMPRESSION, Data);
      { ^ When disk spanning is enabled, the Setup Compiler requires that
        FileLocationEntries be a fixed size, so Z_NO_COMPRESSION is used so
        that FileLocationEntries are not compressed }
    try
      for J := 0 to FileLocationEntries.Count-1 do
        DeflateBlockWrite (Data, FileLocationEntries[J]^, SizeOf(TSetupFileLocationEntry));
      DeflateBlockWriteEnd (Data);
    except
      DeflateBlockWriteCancel (Data);
      raise;
    end;

    Result := FilePos(F) - Pos;
  end;
  function CreateSetup0File: Longint;
  var
    F: File;
  begin
    AssignFile (F, OutputDir + 'SETUP.0');
    FileMode := fmOpenWrite or fmShareExclusive;  Rewrite (F, 1);
    try
      Result := WriteSetup0(F);
    finally
      CloseFile (F);
    end;
  end;
  procedure WriteSetupMsg (var F: File);
  type
    PMessageLengths = ^TMessageLengths;
    TMessageLengths = array[TSetupMessageID] of Smallint;
  var
    Data: TDeflateBlockWriteData;
    ZeroPad: Longint;
    Lengths: PMessageLengths;
    I: TSetupMessageID;
    Len: Integer;
    Header: TMessagesHeader;
    TotalMessagesSize: Longint;
  begin
    ZeroPad := 0;
    New (Lengths);
    try
      { Prepare the header }
      Header.CRCMessages := Longint($FFFFFFFF);  { updated below }
      TotalMessagesSize := 0;
      for I := Low(Messages) to High(Messages) do begin
        Len := StrLen(Messages[I]) + 1;
        Header.CRCMessages := UpdateCRC32(Header.CRCMessages, Messages[I]^,
          Len);
        Lengths^[I] := Len;
        Inc (TotalMessagesSize, Len);
      end;
      Header.CRCMessages := Header.CRCMessages xor Longint($FFFFFFFF);
      Header.NumMessages := Ord(High(Messages)) - Ord(Low(Messages)) + 1;
      Header.Padding := (SizeOf(MessagesHdrID) + SizeOf(Header) + SizeOf(Lengths^)) and 3;
      if Header.Padding <> 0 then
        Header.Padding := 4 - Header.Padding;
      Header.TotalSize := SizeOf(MessagesHdrID) + SizeOf(Header) + SizeOf(Lengths^) + Header.Padding + TotalMessagesSize;
      Header.NotTotalSize := not Header.TotalSize;
      Header.CRCLengths := GetCRC32(Lengths^, SizeOf(Lengths^));

      { Write it all }
      DeflateBlockWriteBegin (F, CompressLevel, Data);
      try
        DeflateBlockWrite (Data, MessagesHdrID, SizeOf(MessagesHdrID));
        DeflateBlockWrite (Data, Header, SizeOf(Header));
        DeflateBlockWrite (Data, Lengths^, SizeOf(Lengths^));
        if Header.Padding <> 0 then
          DeflateBlockWrite (Data, ZeroPad, Header.Padding);
        for I := Low(Messages) to High(Messages) do
          DeflateBlockWrite (Data, Messages[I]^, Lengths^[I]);
        DeflateBlockWriteEnd (Data);
      except
        DeflateBlockWriteCancel (Data);
        raise;
      end;
    finally
      Dispose (Lengths);
    end;
  end;
  procedure CreateSetupMsgFile;
  var
    F: File;
  begin
    AssignFile (F, OutputDir + 'SETUP.MSG');
    FileMode := fmOpenWrite or fmShareExclusive;  Rewrite (F, 1);
    try
      WriteSetupMsg (F);
    finally
      CloseFile (F);
    end;
  end;
  function RoundToNearestClusterSize (const L: Longint): Longint;
  begin
    Result := (L div DiskClusterSize) * DiskClusterSize;
    if L mod DiskClusterSize <> 0 then
      Inc (Result, DiskClusterSize);
  end;
  procedure SplitIntoDisks (const BytesToReserveOnFirstDisk: Longint);
  var
    Buf: PCopyBuffer;
    CurDisk: Integer;
    Src, Dst: TFileStream;
    DiskBytesLeft, BytesToCopy, NumBytes: Longint;
    J: Integer;
    DiskHeader: TDiskHeader;
    procedure EndDisk;
    begin
      if Assigned(Dst) then begin
        DiskHeader.TotalSize := Dst.Size;
        Dst.Position := SizeOf(DiskID);
        Dst.WriteBuffer (DiskHeader, SizeOf(DiskHeader));
        Dst.Free;
        Dst := nil;
      end;
    end;
    procedure NewDisk;
    begin
      EndDisk;
      Inc (CurDisk);
      AddStatus (Format(SCompilerStatusCreatingDisk, [CurDisk]));
      CallIdleProc;
      Dst := TFileStream.Create(OutputDir + OutputBaseFilename + '.' + IntToStr(CurDisk), fmCreate);
      Dst.WriteBuffer (DiskID, SizeOf(DiskID));
      DiskHeader.TotalSize := 0;
      Dst.WriteBuffer (DiskHeader, SizeOf(DiskHeader));
      DiskBytesLeft := DiskTotalBytes - (SizeOf(DiskID) + SizeOf(DiskHeader));
    end;
  begin
    Src := TFileStream.Create(TempFilename, fmOpenRead);
    try
      Src.Position := SizeOf(DiskID) + SizeOf(DiskHeader);
      Dst := nil;
      Buf := nil;
      try
        New (Buf);
        CurDisk := 0;
        NewDisk;
        Dec (DiskBytesLeft, BytesToReserveOnFirstDisk);
        if DiskBytesLeft < 0 then
          AbortCompile (SCompilerNotEnoughSpaceOnFirstDisk);
        for J := 0 to FileLocationEntries.Count-1 do
          with PSetupFileLocationEntry(FileLocationEntries[J])^ do begin
            if DiskBytesLeft < SizeOf(ZLIBID)+1 then
              NewDisk;
            FirstDisk := CurDisk;
            StartOffset := Dst.Position;
            BytesToCopy := SizeOf(ZLIBID) + CompressedSize;
            while True do begin
              NumBytes := BytesToCopy;
              if NumBytes > SizeOf(TCopyBuffer) then NumBytes := SizeOf(TCopyBuffer);
              if NumBytes > DiskBytesLeft then NumBytes := DiskBytesLeft;
              Src.ReadBuffer (Buf^, NumBytes);
              Dst.WriteBuffer (Buf^, NumBytes);
              Dec (BytesToCopy, NumBytes);
              Dec (DiskBytesLeft, NumBytes);
              if BytesToCopy = 0 then
                Break;
              if DiskBytesLeft = 0 then
                NewDisk;
            end;
            LastDisk := CurDisk;
          end;
      finally
        if Assigned(Buf) then
          Dispose (Buf);
        EndDisk;
      end;
    finally
      Src.Free;
      DeleteFile (TempFilename);
    end;
  end;
  procedure CompressFilesIntoTempFile;
  var
    I: Integer;
    ExtraData: TExtraData;
    Date2: TFileTime;
  begin
    for I := 0 to FileLocationEntries.Count-1 do
      with PSetupFileLocationEntry(FileLocationEntries[I])^ do begin
        AddStatus (Format(SCompilerStatusFilesCompressing, [FileLocationEntryFilenames[I]]));
        ExtraData.DestFile := TempFile;
        ExtraData.SourceFile := TFileStream.Create(FileLocationEntryFilenames[I],
          fmOpenRead or fmShareDenyWrite);
        try
          FirstDisk := 1;
          LastDisk := 1;
          StartOffset := TempFile.Position - TempFileOffset;
          TempFile.WriteBuffer (ZLIBID, SizeOf(ZLIBID));
          OriginalSize := ExtraData.SourceFile.Size;
          CustomDeflateData (ReadProc, WriteProc, Longint(@ExtraData), CompressLevel,
            CompressedSize, Adler);
          GetFileTime (ExtraData.SourceFile.Handle, nil, nil, @Date2);
          FileTimeToLocalFileTime (Date2, Date);
        finally
          ExtraData.SourceFile.Free;
        end;
      end;
  end;
var
  I: Integer;
begin
  AbortLineNumber := 0;

  WizardImage := nil;
  HasRestart := False;
  HasRegSvr := False;

  try
    SetupHeader := AllocMem(SizeOf(TSetupHeader));
    FillChar (SetupHeader^, SizeOf(TSetupHeader), 0);

    { Initialize defaults }
    OutputDir := 'Output';
    OutputBaseFilename := 'setup';
    CompressLevel := 7;
    UseSetupLdr := True;
    DiskTotalBytes := 1457664;
    DiskClusterSize := 512;
    ReserveBytes := 0;
    with SetupHeader^.MinVersion do begin
      WinVersion := $04000000;
      NTVersion := $04000000;
    end;
    SetupHeader^.Options := [shCreateAppDir, shUninstallable, shWindowVisible,
      shWindowStartMaximized, shWindowShowCaption, shWindowResizable,
      shCreateUninstallRegKey, shUsePreviousAppDir, shUsePreviousGroup,
      shUpdateUninstallLogAppName];
    SetupHeader^.UninstallFilesDir := '{app}';
    SetupHeader^.BackColor := clBlue;
    SetupHeader^.BackColor2 := clBlack;
    BackSolid := False;
    SetupHeader^.WizardImageBackColor := clTeal;
    WizardImageFile := 'compiler:WIZIMAGE.BMP';
    MessagesFile := 'compiler:DEFAULT.ISL';

    { Read [Setup] section }
    EnumIniSection (EnumSetup, 'Setup', 0, True, TextFile(nil^));
    CallIdleProc;

    { Verify settings set in [Setup] section }
    SetupHeader^.BaseFilename := OutputBaseFilename;
    if SetupDirectiveLines[ssAppName] = 0 then
      AbortCompileFmt (SCompilerEntryMissing2, ['Setup', 'AppName']);
    if SetupDirectiveLines[ssAppVerName] = 0 then
      AbortCompileFmt (SCompilerEntryMissing2, ['Setup', 'AppVerName']);
    if SetupHeader^.AppId = '' then
      SetupHeader^.AppId := SetupHeader^.AppName;
    CheckDirConst (SetupHeader^.DefaultDirName, SetupDirectiveLines[ssDefaultDirName],
      SetupHeader^.MinVersion, []);
    if SetupHeader^.DefaultDirName = '' then begin
      if shCreateAppDir in SetupHeader^.Options then
        AbortCompileFmt (SCompilerEntryMissing2, ['Setup', 'DefaultDirName'])
      else
        SetupHeader^.DefaultDirName := '?ERROR?';
    end;
    if MinVersionLessThan4(SetupHeader^.MinVersion) and
       ContainsBadPMChars(SetupHeader^.DefaultDirName) then
      AbortCompile (SCompilerBadCharInDefaultDirName32);
    if SetupHeader^.DefaultGroupName = '' then
      SetupHeader^.DefaultGroupName := '(Default)';
    CheckDirConst (SetupHeader^.UninstallDisplayIcon, SetupDirectiveLines[ssUninstallDisplayIcon],
      SetupHeader^.MinVersion, []);
    CheckDirConst (SetupHeader^.UninstallFilesDir, SetupDirectiveLines[ssUninstallFilesDir],
      SetupHeader^.MinVersion, []);
    if BackSolid then
      SetupHeader^.BackColor2 := SetupHeader^.BackColor;
    if not DiskSpanning then begin
      DiskTotalBytes := High(Longint);
      DiskClusterSize := 1;
      ReserveBytes := 0;
    end;

    CompilerDir := AddBackslash(ExpandFileName(CompilerDir));
    SourceDir := AddBackslash(ExpandFileName(SourceDir));
    if IsRelativePath(OutputDir) then
      OutputDir := SourceDir + OutputDir;
    OutputDir := ExpandFileName(OutputDir);
    if not DirExists(OutputDir) then
      MkDir (OutputDir);
    OutputDir := AddBackslash(OutputDir);
    { Clear any SETUP.* files out of the output directory first. Raise
      an error if any files still remain }
    EmptyOutputDir;

    { Read text files }
    if LicenseFile <> '' then begin
      AddStatus (Format(SCompilerStatusReadingFile, ['LicenseFile']));
      ReadTextFile (PrependSourceDirName(LicenseFile), LicenseText);
    end;
    if InfoBeforeFile <> '' then begin
      AddStatus (Format(SCompilerStatusReadingFile, ['InfoBeforeFile']));
      ReadTextFile (PrependSourceDirName(InfoBeforeFile), InfoBeforeText);
    end;
    if InfoAfterFile <> '' then begin
      AddStatus (Format(SCompilerStatusReadingFile, ['InfoAfterFile']));
      ReadTextFile (PrependSourceDirName(InfoAfterFile), InfoAfterText);
    end;
    CallIdleProc;

    { Read wizard image }
    AddStatus (Format(SCompilerStatusReadingFile, ['WizardImage']));
    AddStatus (Format(SCompilerStatusReadingInFile, [PrependSourceDirName(WizardImageFile)]));
    WizardImage := TMemoryStream.Create;
    WizardImage.LoadFromFile (PrependSourceDirName(WizardImageFile));

    { Read messages }
    AddStatus (SCompilerStatusReadingMessages);
    ReadMessages;

    { Read [Dirs] section }
    EnumIniSection (EnumDirs, 'Dirs', 0, True, TextFile(nil^));
    CallIdleProc;

    { Read [Icons] section }
    EnumIniSection (EnumIcons, 'Icons', 0, True, TextFile(nil^));
    CallIdleProc;

    { Read [INI] section }
    EnumIniSection (EnumINI, 'INI', 0, True, TextFile(nil^));
    CallIdleProc;

    { Read [Registry] section }
    EnumIniSection (EnumRegistry, 'Registry', 0, True, TextFile(nil^));
    CallIdleProc;

    { Read [InstallDelete] section }
    EnumIniSection (EnumDelete, 'InstallDelete', 0, True, TextFile(nil^));
    CallIdleProc;

    { Read [UninstallDelete] section }
    EnumIniSection (EnumDelete, 'UninstallDelete', 1, True, TextFile(nil^));
    CallIdleProc;

    { Read [Run] section }
    EnumIniSection (EnumRun, 'Run', 0, True, TextFile(nil^));
    CallIdleProc;

    { Read [UninstallRun] section }
    EnumIniSection (EnumRun, 'UninstallRun', 1, True, TextFile(nil^));
    CallIdleProc;

    { Read [Files] section }
    if shUninstallable in SetupHeader^.Options then
      EnumFiles ('', 1, 0{ignored});
    EnumIniSection (EnumFiles, 'Files', 0, True, TextFile(nil^));
    if HasRestart and HasRegSvr then
      { include REGSVR.E32 }
      EnumFiles ('', 2, 0{ignored});
    CallIdleProc;

    { Create setup files }
    AddStatus (SCompilerStatusCreateSetupFiles);

    ExeFilename := OutputDir + OutputBaseFilename + '.exe';
    SetupPrg := CompilerDir;
    if not UseSetupLdr then
      SetupPrg := SetupPrg + 'SETUP.E32'
    else
      SetupPrg := SetupPrg + 'SETUPLDR.E32';
    if not FileCopy(SetupPrg, ExeFilename, False, fmOpenRead or fmShareDenyWrite) then
      AbortCompileFmt (SCompilerCopyError2, [SetupPrg, ExeFilename]);
    FileSetAttr (ExeFilename, faArchive);  { if there was a read-only attribute, remove it }
    AssignFile (SetupFile, ExeFilename);
    FileMode := fmOpenRead or fmShareDenyWrite;  Reset (SetupFile, 1);
    try
      SizeOfExe := FileSize(SetupFile);
    finally
      CloseFile (SetupFile);
    end;
    CallIdleProc;
    try
      TempFile := nil;
      try
        if UseSetupLdr and not DiskSpanning then begin
          TempFilename := ExeFilename;
          TempFile := TFileStream.Create(TempFilename, fmOpenReadWrite or fmShareExclusive);
          TempFile.Seek (0, soFromEnd);
          TempFileOffset := TempFile.Position;
        end
        else begin
          if not DiskSpanning then
            TempFilename := OutputDir + OutputBaseFilename + '.1'
          else
            TempFilename := OutputDir + 'SETUP.TMP';
          TempFile := TFileStream.Create(TempFilename, fmCreate);
          TempFileOffset := 0;
        end;

        TempFile.WriteBuffer (DiskID, SizeOf(DiskID));
        DiskHdr.TotalSize := 0;
        TempFile.WriteBuffer (DiskHdr, SizeOf(DiskHdr));

        CompressFilesIntoTempFile;

        TempFile.Position := TempFileOffset + SizeOf(DiskID);
        DiskHdr.TotalSize := TempFile.Size - TempFileOffset;
        TempFile.WriteBuffer (DiskHdr, SizeOf(DiskHdr));
      finally
        TempFile.Free;
        TempFile := nil;
      end;
      CallIdleProc;

      if not UseSetupLdr then begin
        { Create SETUP.MSG in output dir }
        CreateSetupMsgFile;

        { Create SETUP.0 in output dir }
        SizeOfHeaders := CreateSetup0File;

        if DiskSpanning then begin
          { Split up SETUP.1 into separate disks }
          SplitIntoDisks (RoundToNearestClusterSize(SizeOfExe) +
            RoundToNearestClusterSize(SizeOfHeaders) +
            RoundToNearestClusterSize(ReserveBytes));
          { SplitIntoDisks modifies setup header data, so go back and
            rewrite it }
          if CreateSetup0File <> SizeOfHeaders then
            { Make sure new and old size match. No reason why they
              shouldn't but check just in case }
            AbortCompile (SCompilerSetup0Mismatch);
        end;
      end
      else begin
        AssignFile (ExeFile, ExeFilename);
        FileMode := fmOpenReadWrite or fmShareExclusive;  Reset (ExeFile, 1);
        try
          Seek (ExeFile, FileSize(ExeFile));

          { Move the data from SETUP.E?? into the SETUP.EXE, and write
            header data }
          SetupLdrOffsetTable.ID := SetupLdrOffsetTableID;
          SetupLdrOffsetTable.OffsetEXE := FilePos(ExeFile);
          AssignFile (ConvertFile, CompilerDir + 'SETUP.E32');
          FileMode := fmOpenRead or fmShareDenyWrite;  Reset (ConvertFile, 1);
          try
            SetupLdrOffsetTable.UncompressedSizeEXE := FileSize(ConvertFile);
            DeflateData (ConvertFile, SetupLdrOffsetTable.UncompressedSizeEXE, True,
              ExeFile, -1, True,
              CompressLevel, SetupLdrOffsetTable.CompressedSizeEXE, SetupLdrOffsetTable.AdlerEXE);
          finally
            CloseFile (ConvertFile);
          end;
          SetupLdrOffsetTable.OffsetMsg := FilePos(ExeFile);
          WriteSetupMsg (ExeFile);
          SetupLdrOffsetTable.Offset0 := FilePos(ExeFile);
          SizeOfHeaders := WriteSetup0(ExeFile);
          SetupLdrOffsetTable.TotalSize := FileSize(ExeFile) + SizeOf(SetupLdrOffsetTable);
          if DiskSpanning then begin
            SetupLdrOffsetTable.Offset1 := 0;
            { Split up SETUP.TMP into separate disks }
            SplitIntoDisks (RoundToNearestClusterSize(SetupLdrOffsetTable.TotalSize) +
              RoundToNearestClusterSize(ReserveBytes));
            { SplitIntoDisks modifies setup header data, so go back and
              rewrite it }
            Seek (ExeFile, SetupLdrOffsetTable.Offset0);
            if WriteSetup0(ExeFile) <> SizeOfHeaders then
              { Make sure new and old size match. No reason why they
                shouldn't but check just in case }
              AbortCompile (SCompilerSetup0Mismatch);
          end
          else
            SetupLdrOffsetTable.Offset1 := SizeOfExe;

          { Write SetupLdrExeHeader and SetupLdrOffsetTable to SETUP.EXE }
          with SetupLdrExeHeader do begin
            ID := SetupLdrExeHeaderID;
            OffsetTableOffset := FileSize(ExeFile);
            NotOffsetTableOffset := not OffsetTableOffset;
          end;
          Seek (ExeFile, SetupLdrExeHeaderOffset);
          BlockWrite (ExeFile, SetupLdrExeHeader, SizeOf(SetupLdrExeHeader));
          Seek (ExeFile, SetupLdrExeHeader.OffsetTableOffset);
          BlockWrite (ExeFile, SetupLdrOffsetTable, SizeOf(SetupLdrOffsetTable));

          { For some reason, on Win95 the date/time of the EXE sometimes
            doesn't get updated after it's been written to so it has to
            manually set it. (I don't get it!!) }
          UpdateTimeStamp (TFileRec(ExeFile).Handle);
        finally
          CloseFile (ExeFile);
        end;
      end;
    except
      DeleteFile (ExeFilename);
      if DiskSpanning then
        DeleteFile (TempFilename);
      raise;
    end;
    CallIdleProc;

    { Done }
    AddStatus ('');
    for I := 0 to WarningsList.Count-1 do
      AddStatus (SCompilerStatusWarning + WarningsList[I]);
    asm jmp @1; db 0,'Inno Setup Compiler, Copyright (C) 1998-2000 Jordan Russell',0; @1: end;
    { Note: Removing or modifying the copyright text is a violation of the
      Inno Setup license agreement; see LICENSE.TXT. }
  finally
    WarningsList.Clear;
    { Free all the data }
    SEFreeRec (SetupHeader, SetupHeaderStrings);
    WizardImage.Free;
    FreeListItems (DirEntries, SetupDirEntryStrings);
    FreeListItems (FileEntries, SetupFileEntryStrings);
    FreeListItems (FileLocationEntries, SetupFileLocationEntryStrings);
    FreeListItems (IconEntries, SetupIconEntryStrings);
    FreeListItems (IniEntries, SetupIniEntryStrings);
    FreeListItems (RegistryEntries, SetupRegistryEntryStrings);
    FreeListItems (InstallDeleteEntries, SetupDeleteEntryStrings);
    FreeListItems (UninstallDeleteEntries, SetupDeleteEntryStrings);
    FreeListItems (RunEntries, SetupRunEntryStrings);
    FreeListItems (UninstallRunEntries, SetupRunEntryStrings);
    FileLocationEntryFilenames.Clear;
    FreeMessages;
  end;
end;


{ Interface functions }

function ISCompileScript (const Params: TCompileScriptParams;
  const PropagateExceptions: Boolean): Integer;
var
  SetupCompiler: TSetupCompiler;
  Data: TCompilerCallbackData;
  S: String;
begin
  if (Params.Size <> SizeOf(Params)) or
     not Assigned(Params.CallbackProc) then begin
    Result := isceInvalidParam;
    Exit;
  end;
  SetupCompiler := TSetupCompiler.Create(nil);
  try
    SetupCompiler.AppData := Params.AppData;
    SetupCompiler.CallbackProc := Params.CallbackProc;
    if Assigned(Params.CompilerPath) then
      SetupCompiler.CompilerDir := Params.CompilerPath
    else
      SetupCompiler.CompilerDir := ExtractFilePath(GetSelfFilename);
    SetupCompiler.SourceDir := Params.SourcePath;

    Result := isceNoError;
    try
      SetupCompiler.Compile;
    except
      on E: Exception do begin
        Result := isceCompileFailure;
        Data.ErrorMsg := nil;
        Data.ErrorFilename := nil;
        Data.ErrorLine := 0;
        if not(E is EAbort) then begin
          S := AddPeriod(E.Message);
          Data.ErrorMsg := PChar(S);
        end;
        if E is EISCompileError then begin
          if EISCompileError(E).Filename <> '' then
            Data.ErrorFilename := PChar(EISCompileError(E).Filename);
          Data.ErrorLine := EISCompileError(E).LineNumber;
        end;
        Params.CallbackProc (iscbNotifyError, Data, Params.AppData);
        if PropagateExceptions then
          raise;
        Exit;
      end;
    end;
    Data.OutputExeFilename := PChar(SetupCompiler.ExeFilename);
    Params.CallbackProc (iscbNotifySuccess, Data, Params.AppData);
  finally
    SetupCompiler.Free;
  end;
end;

function ISGetVersion: PCompilerVersionInfo;
const
  Ver: TCompilerVersionInfo =
   (Title: SetupTitle; Version: SetupVersion; BinVersion: SetupBinVersion);
begin
  Result := @Ver;
end;

end.
