unit CompForm;

{
  Inno Setup
  Copyright (C) 1998-2001 Jordan Russell
  For conditions of distribution and use, see LICENSE.TXT.

  Compiler form
}

{x$DEFINE STATICCOMPILER}
{ For debugging purposes, remove the 'x' to have it link the compiler code
  into this program and not depend on ISCmplr.dll. }

interface

uses
  WinTypes, WinProcs, SysUtils, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, ExtCtrls, Menus,
  Buttons;

const
  CompilerIDEVersion = 'a';

  WM_StartCommandLineCompile = WM_USER + $1000;

  MRUListMaxCount = 8;

type
  TCompileForm = class(TForm)
    OpenDialog: TOpenDialog;
    MainMenu1: TMainMenu;
    FMenu: TMenuItem;
    FNew: TMenuItem;
    FOpen: TMenuItem;
    FSave: TMenuItem;
    FSaveAs: TMenuItem;
    N1: TMenuItem;
    FCompile: TMenuItem;
    N2: TMenuItem;
    FExit: TMenuItem;
    EMenu: TMenuItem;
    EUndo: TMenuItem;
    N3: TMenuItem;
    ECut: TMenuItem;
    ECopy: TMenuItem;
    EPaste: TMenuItem;
    EDelete: TMenuItem;
    N4: TMenuItem;
    ESelectAll: TMenuItem;
    N5: TMenuItem;
    EWordWrap: TMenuItem;
    VMenu: TMenuItem;
    EFind: TMenuItem;
    EFindNext: TMenuItem;
    EReplace: TMenuItem;
    Help1: TMenuItem;
    HDoc: TMenuItem;
    N6: TMenuItem;
    HAbout: TMenuItem;
    Memo: TMemo;
    SaveDialog: TSaveDialog;
    FMRUSep: TMenuItem;
    VCompilerStatus: TMenuItem;
    FindDialog: TFindDialog;
    ReplaceDialog: TReplaceDialog;
    StatusPanel: TPanel;
    Status: TListBox;
    SplitPanel: TPanel;
    ToolbarPanel: TPanel;
    SpeedButton1: TSpeedButton;
    Bevel1: TBevel;
    SpeedButton2: TSpeedButton;
    SpeedButton3: TSpeedButton;
    SpeedButton4: TSpeedButton;
    SpeedButton5: TSpeedButton;
    HWebsite: TMenuItem;
    VToolbar: TMenuItem;
    N7: TMenuItem;
    VOptions: TMenuItem;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure FExitClick(Sender: TObject);
    procedure FOpenClick(Sender: TObject);
    procedure EUndoClick(Sender: TObject);
    procedure EMenuClick(Sender: TObject);
    procedure ECutClick(Sender: TObject);
    procedure ECopyClick(Sender: TObject);
    procedure EPasteClick(Sender: TObject);
    procedure EDeleteClick(Sender: TObject);
    procedure FSaveClick(Sender: TObject);
    procedure ESelectAllClick(Sender: TObject);
    procedure FNewClick(Sender: TObject);
    procedure FSaveAsClick(Sender: TObject);
    procedure EWordWrapClick(Sender: TObject);
    procedure HDocClick(Sender: TObject);
    procedure FCompileClick(Sender: TObject);
    procedure FMenuClick(Sender: TObject);
    procedure FMRUClick(Sender: TObject);
    procedure VCompilerStatusClick(Sender: TObject);
    procedure HAboutClick(Sender: TObject);
    procedure EFindClick(Sender: TObject);
    procedure FindDialogFind(Sender: TObject);
    procedure EReplaceClick(Sender: TObject);
    procedure ReplaceDialogReplace(Sender: TObject);
    procedure EFindNextClick(Sender: TObject);
    procedure SplitPanelMouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Integer);
    procedure VMenuClick(Sender: TObject);
    procedure HWebsiteClick(Sender: TObject);
    procedure VToolbarClick(Sender: TObject);
    procedure VOptionsClick(Sender: TObject);
  private
    { Private declarations }
    Version, Filename: String;
    MRUMenuItems: array[0..MRUListMaxCount-1] of TMenuItem;
    MRUList: TStringList;
    Options: record
      TestAfterCompile: Boolean;
      MakeBackups: Boolean;
    end;
    procedure AddToMRUList (const AFilename: String);
    function AskToSaveModifiedFile: Boolean;
    procedure CompileFile (const AFilename: String;
      const ReadFromFile, PromptToTest: Boolean);
    procedure NewFile;
    procedure OpenFile (AFilename: String);
    function ProcessPaintMessages: Boolean;
    function SaveFile (const SaveAs: Boolean): Boolean;
    function SearchMemo (const SearchString: String;
      Options: TFindOptions): Boolean;
    procedure SetStatusPanelVisible (const AVisible: Boolean);
    procedure StatusMessage (S: String; const AddTime: Boolean);
    procedure UpdateCaption;
    procedure UpdateStatusPanelHeight (H: Integer);
    procedure WMStartCommandLineCompile (var Message: TMessage); message WM_StartCommandLineCompile;
  public
    { Public declarations }
  end;

var
  CompileForm: TCompileForm;

  CommandLineFilename: String;
  CommandLineCompile: Boolean;

implementation

uses
  Clipbrd, ShellApi, IniFiles, Registry,
  CmnFunc, CmnFunc2, CompMsgs,
  {$IFDEF STATICCOMPILER} Compile, {$ENDIF}
  CompInt, CompOptions;

{$R *.DFM}

const
  HistoryListSize = 8;

procedure TCompileForm.FormCreate(Sender: TObject);
var
  Ini: TRegIniFile;
  S: String;
  I: Integer;
  NewItem: TMenuItem;
begin
  {$IFNDEF STATICCOMPILER}
  Version := ISDllGetVersion.Version;
  {$ELSE}
  Version := ISGetVersion.Version;
  {$ENDIF}

  { For some reason, if AutoScroll=False is set on the form Delphi ignores the
    'poDefault' Position setting }
  AutoScroll := False;

  { Append 'Del' to the end of the Delete item. Don't actually use Del as
    the shortcut key so that the Del key still works when the menu item is
    disabled because there is no selection. }
  EDelete.Caption := EDelete.Caption + #9 + ShortCutToText(VK_DELETE);

  MRUList := TStringList.Create;
  for I := 0 to High(MRUMenuItems) do begin
    NewItem := TMenuItem.Create(Self);
    NewItem.OnClick := FMRUClick;
    FMenu.Insert (FMenu.IndexOf(FMRUSep), NewItem);
    MRUMenuItems[I] := NewItem;
  end;

  UpdateCaption;
  {ScriptFileLabel.Caption := SCompilerScriptFileLabel;
  ScriptBrowseButton.Caption := SCompilerScriptBrowseButton;
  StatusLabel.Caption := SCompilerStatusLabel;
  StartButton.Caption := SCompilerStartButton;
  ExitButton.Caption := SCompilerExitButton;}
  OpenDialog.Filter := SCompilerOpenFilter;
  SaveDialog.Filter := SCompilerOpenFilter;

  if CommandLineCompile then
    PostMessage (Handle, WM_StartCommandLineCompile, 0, 0)
  else begin
    Ini := TRegIniFile.Create('Software\Jordan Russell\Inno Setup');
    try
      { Don't localize! }
      for I := 0 to High(MRUMenuItems) do begin
        S := Ini.ReadString('ScriptFileHistoryNew', 'History' + IntToStr(I), '');
        if S <> '' then MRUList.Add (S);
      end;
      ToolbarPanel.Visible := Ini.ReadBool('Options', 'ShowToolbar', True);
      Options.TestAfterCompile := Ini.ReadBool('Options', 'TestAfterCompile', True);
      Options.MakeBackups := Ini.ReadBool('Options', 'MakeBackups', False);
    finally
      Ini.Free;
    end;
    if CommandLineFilename <> '' then
      OpenFile (CommandLineFilename);
  end;
end;

procedure TCompileForm.FormDestroy(Sender: TObject);
var
  Ini: TRegIniFile;
  I: Integer;
  S: String;
begin
  if not CommandLineCompile then begin
    Ini := TRegIniFile.Create('Software\Jordan Russell\Inno Setup');
    try
      { Don't localize! }
      for I := 0 to High(MRUMenuItems) do begin
        if I < MRUList.Count then
          S := MRUList[I]
        else
          S := '';
        Ini.WriteString ('ScriptFileHistoryNew', 'History' + IntToStr(I),
          S {work around Delphi 2 bug:} + #0);
      end;
      Ini.WriteBool ('Options', 'ShowToolbar', ToolbarPanel.Visible);
      Ini.WriteBool ('Options', 'TestAfterCompile', Options.TestAfterCompile);
      Ini.WriteBool ('Options', 'MakeBackups', Options.MakeBackups);
    finally
      Ini.Free;
    end;
  end;

  MRUList.Free;
end;

procedure TCompileForm.FormCloseQuery(Sender: TObject;
  var CanClose: Boolean);
begin
  CanClose := AskToSaveModifiedFile;
end;

procedure TCompileForm.UpdateCaption;
var
  NewCaption: String;
begin
  NewCaption := Filename;
  if NewCaption = '' then NewCaption := 'Untitled';
  NewCaption := NewCaption + ' - ' + SCompilerFormCaption + ' ' + Version;
  Caption := NewCaption;
  Application.Title := NewCaption;
end;

procedure TCompileForm.NewFile;
begin
  Memo.Clear;
  Memo.Modified := False;
  Filename := '';
  UpdateCaption;
end;

procedure TCompileForm.OpenFile (AFilename: String);
begin
  AFilename := ExpandFileName(AFilename);
  AddToMRUList (AFilename);
  try
    Memo.Lines.LoadFromFile (AFilename);
  except
    on EInvalidOperation do begin
      if MsgBox('Script file is too large to open in the Inno Setup editor.'#13#10#13#10 +
         'Would you like it to be compiled?', SCompilerFormCaption, mbError,
         MB_YESNO) = ID_YES then begin
        NewFile;
        CompileFile (AFilename, True, True);
      end;
      Exit;
    end;
  end;
  Memo.Modified := False;
  Filename := AFilename;
  UpdateCaption;
end;

function TCompileForm.SaveFile (const SaveAs: Boolean): Boolean;

  procedure SaveTo (const FN: String);
  var
    BackupFN: String;
  begin
    if Options.MakeBackups and NewFileExists(FN) then begin
      BackupFN := ChangeFileExt(FN, '.~is');  {m}
      DeleteFile (BackupFN);
      if not RenameFile(FN, BackupFN) then
        raise Exception.Create('Error creating backup file. Could not save file');
    end;
    Memo.Lines.SaveToFile (FN);
  end;

begin
  Result := False;
  if SaveAs or (Filename = '') then begin
    SaveDialog.Filename := Filename;
    if not SaveDialog.Execute then Exit;
    SaveTo (SaveDialog.Filename);
    Filename := SaveDialog.Filename;
    UpdateCaption;
  end
  else
    SaveTo (Filename);
  Memo.Modified := False;
  Result := True;
  AddToMRUList (Filename);
end;

function TCompileForm.AskToSaveModifiedFile: Boolean;
var
  FileTitle: String;
begin
  Result := True;
  if Memo.Modified then begin
    FileTitle := Filename;
    if FileTitle = '' then FileTitle := 'Untitled';
    case MsgBox('The text in the ' + FileTitle + ' file has changed.'#13#10#13#10 +
       'Do you want to save the changes?', SCompilerFormCaption, mbError,
       MB_YESNOCANCEL) of
      ID_YES: Result := SaveFile(False);
      ID_NO: ;
    else
      Result := False;
    end;
  end;
end;

procedure TCompileForm.AddToMRUList (const AFilename: String);
var
  I: Integer;
begin
  I := 0;
  while I < MRUList.Count do begin
    if CompareText(MRUList[I], AFilename) = 0 then
      MRUList.Delete (I)
    else
      Inc (I);
  end;
  MRUList.Insert (0, AFilename);
  while MRUList.Count > High(MRUMenuItems)+1 do
    MRUList.Delete (MRUList.Count-1);
end;

procedure TCompileForm.StatusMessage (S: String; const AddTime: Boolean);
var
  DC: HDC;
  Size: TSize;
begin
  if AddTime then
    S := S + '  [' + TimeToStr(Time) + ']';
  with Status do begin
    try
      TopIndex := Items.Add(S);
    except
      on EOutOfResources do begin
        Clear;
        SendMessage (Handle, LB_SETHORIZONTALEXTENT, 0, 0);
        Items.Add (SCompilerStatusReset);
        TopIndex := Items.Add(S);
      end;
    end;
    DC := GetDC(0);
    try
      SelectObject (DC, Font.Handle);
      GetTextExtentPoint (DC, PChar(S), Length(S), Size);
    finally
      ReleaseDC (0, DC);
    end;
    Inc (Size.cx, 5);
    if Size.cx > SendMessage(Handle, LB_GETHORIZONTALEXTENT, 0, 0) then
      SendMessage (Handle, LB_SETHORIZONTALEXTENT, Size.cx, 0);
  end;
end;

type
  PAppData = ^TAppData;
  TAppData = record
    Form: TCompileForm;
    ScriptFile: ^TextFile;
    CurLineNumber: Integer;
    CurLine: String;
    OutputExe: String;
    ErrorMsg: String;
    ErrorFilename: String;
    ErrorLine: Integer;
    Aborted: Boolean;
  end;

function CompilerCallbackProc (Code: Integer; var Data: TCompilerCallbackData;
  AppData: Longint): Integer; stdcall;
var
  Len, BufSize: Integer;
begin
  Result := iscrSuccess;
  with PAppData(AppData)^ do
    case Code of
      iscbReadScript:
        if ScriptFile = nil then begin
          if Data.Reset then
            CurLineNumber := 0;
          if CurLineNumber < Form.Memo.Lines.Count then begin
            BufSize := 4096;
            while True do begin
              SetLength (CurLine, BufSize);
              Cardinal((@CurLine[1])^) := BufSize + 1;
              Len := SendMessage(Form.Memo.Handle, EM_GETLINE, CurLineNumber,
                Longint(@CurLine[1]));
              if Len < BufSize then begin
                SetLength (CurLine, Len);
                Data.LineRead := PChar(CurLine);
                Inc (CurLineNumber);
                Break;
              end;
              { Increase buffer size if it couldn't fit entire line }
              Inc (BufSize, 4096);
            end;
          end;
        end
        else begin
          if Data.Reset then
            Reset (ScriptFile^);
          if not Eof(ScriptFile^) then begin
            Readln (ScriptFile^, CurLine);
            Data.LineRead := PChar(CurLine);
          end;
        end;
      iscbNotifyStatus:
        Form.StatusMessage (Data.StatusMsg, False);
      iscbNotifyIdle:
        if Form.ProcessPaintMessages then
          Result := iscrRequestAbort;
      iscbNotifySuccess:
        OutputExe := Data.OutputExeFilename;
      iscbNotifyError:
        begin
          if Assigned(Data.ErrorMsg) then
            ErrorMsg := Data.ErrorMsg
          else
            Aborted := True;
          ErrorFilename := Data.ErrorFilename;
          ErrorLine := Data.ErrorLine;
        end;
    end;
end;

function TCompileForm.ProcessPaintMessages: Boolean;
{ Dispatches all pending WM_PAINT messages. In effect, this is like an
  'UpdateWindow' on all visible windows }
var
  Msg: TMsg;
begin
  Result := False;
  while PeekMessage(Msg, 0, WM_KEYFIRST, WM_KEYLAST, PM_REMOVE) do begin
    if Msg.message = WM_QUIT then begin
      { Repost WM_QUIT messages }
      PostQuitMessage (Msg.WParam);
      Exit;
    end;
    if (Msg.message = WM_KEYDOWN) and (Msg.wParam = VK_ESCAPE) then begin
      Result := True;
      Break;
    end;
  end;
  while PeekMessage(Msg, 0, WM_PAINT, WM_PAINT, PM_NOREMOVE) do begin
    case Integer(GetMessage(Msg, 0, WM_PAINT, WM_PAINT)) of
      -1: Break; { if GetMessage failed }
      0: begin
           { Repost WM_QUIT messages }
           PostQuitMessage (Msg.WParam);
           Break;
         end;
    end;
    DispatchMessage (Msg);
  end;
end;

procedure TCompileForm.CompileFile (const AFilename: String;
  const ReadFromFile, PromptToTest: Boolean);
var
  F: TextFile;
  SourcePath, S: String;
  Params: TCompileScriptParams;
  AppData: TAppData;
  I: Integer;
begin
  if ReadFromFile then begin
    AssignFile (F, AFilename);
    FileMode := fmOpenRead or fmShareDenyWrite;  Reset (F);
  end;
  try
    Screen.Cursor := crHourglass;
    Enabled := False;
    Status.Clear;
    SendMessage (Status.Handle, LB_SETHORIZONTALEXTENT, 0, 0);
    SetStatusPanelVisible (True);

    SourcePath := ExtractFilePath(AFilename);
    FillChar (Params, SizeOf(Params), 0);
    Params.Size := SizeOf(Params);
    Params.CompilerPath := nil;
    Params.SourcePath := PChar(SourcePath);
    Params.CallbackProc := CompilerCallbackProc;
    Pointer(Params.AppData) := @AppData;

    AppData.Form := Self;
    if ReadFromFile then
      AppData.ScriptFile := @F
    else
      AppData.ScriptFile := nil;
    AppData.CurLineNumber := 0;
    AppData.Aborted := False;

    StatusMessage (SCompilerStatusStarting, True);
    StatusMessage ('', False);
    {$IFNDEF STATICCOMPILER}
    if ISDllCompileScript(Params) <> isceNoError then begin
    {$ELSE}
    if ISCompileScript(Params, False) <> isceNoError then begin
    {$ENDIF}
      StatusMessage (SCompilerStatusErrorAborted, False);
      Enabled := True;
      if (AppData.ScriptFile = nil) and (AppData.ErrorLine > 0) then begin
        { Move the caret to the line number the error occured on }
        I := SendMessage(Memo.Handle, EM_LINEINDEX, AppData.ErrorLine-1, 0);
        SendMessage (Memo.Handle, EM_SETSEL, I, I);
        SendMessage (Memo.Handle, EM_SCROLLCARET, 0, 0);
      end;
      if not AppData.Aborted then begin
        S := '';
        if AppData.ErrorFilename <> '' then
          S := 'File: ' + AppData.ErrorFilename + SNewLine2;
        if AppData.ErrorLine > 0 then
          S := S + Format('Line %d:' + SNewLine, [AppData.ErrorLine]);
        S := S + AppData.ErrorMsg;
        MsgBox (S, 'Compiler Error', mbCriticalError, MB_OK)
      end
      else
        MsgBox ('Compile aborted.', 'Compiler', mbError, MB_OK);
      Exit;
    end;
    StatusMessage (SCompilerStatusFinished, True);

    Enabled := True;
    if PromptToTest and Options.TestAfterCompile then begin
      if MsgBoxFmt(SCompilerSuccessfulMessage2,
         [RemoveBackslashUnlessRoot(ExtractFilePath(AppData.OutputExe))],
         SCompilerSuccessfulTitle, mbInformation,
         MB_YESNO or MB_DEFBUTTON2) = ID_YES then
        if ShellExecute(Handle, nil, StringAsPChar(AppData.OutputExe), '',
           StringAsPChar(AppData.OutputExe), SW_SHOW) <= 32 then
          raise Exception.Create(SCompilerExecuteSetupError);
    end;
  finally
    Enabled := True;
    Screen.Cursor := crDefault;
    if ReadFromFile then
      CloseFile (F);
  end;
end;

procedure TCompileForm.FMenuClick(Sender: TObject);
var
  I: Integer;
begin
  FMRUSep.Visible := MRUList.Count <> 0;
  for I := 0 to High(MRUMenuItems) do
    with MRUMenuItems[I] do begin
      if I < MRUList.Count then begin
        Visible := True;
        Caption := '&' + IntToStr(I+1) + ' ' + MRUList[I];
      end
      else
        Visible := False;
    end;
end;

procedure TCompileForm.FNewClick(Sender: TObject);
begin
  if not AskToSaveModifiedFile then Exit;
  NewFile;
end;

procedure TCompileForm.FOpenClick(Sender: TObject);
begin
  OpenDialog.Filename := '';
  if not AskToSaveModifiedFile or not OpenDialog.Execute then
    Exit;
  OpenFile (OpenDialog.Filename);
end;

procedure TCompileForm.FSaveClick(Sender: TObject);
begin
  SaveFile (False);
end;

procedure TCompileForm.FSaveAsClick(Sender: TObject);
begin
  SaveFile (True);
end;

procedure TCompileForm.FCompileClick(Sender: TObject);
begin
  if Filename = '' then begin
    if MsgBox('The file must be saved before compiling.'#13#10#13#10 +
       'Do you want to save it now?', SCompilerFormCaption, mbError,
       MB_YESNO) <> ID_YES then
      Exit;
    if not SaveFile(False) then
      Exit;
  end;
  if Memo.WordWrap then begin
    { When word wrap is on, EM_GETLINE thinks text wrapped from a previous
      line is a new line. I'm not sure how to work around that, so for now
      don't allow compiling when word wrap is on. }
    MsgBox ('You must uncheck Word Wrap on the Edit menu before compiling.',
      '', mbError, MB_OK);
    Exit;
  end;
  CompileFile (Filename, False, True);
end;

procedure TCompileForm.FMRUClick(Sender: TObject);
var
  I: Integer;
begin
  if not AskToSaveModifiedFile then Exit;
  for I := 0 to High(MRUMenuItems) do
    if MRUMenuItems[I] = Sender then begin
      OpenFile (MRUList[I]);
      Break;
    end;
end;

procedure TCompileForm.FExitClick(Sender: TObject);
begin
  Close;
end;

procedure TCompileForm.EMenuClick(Sender: TObject);
begin
  EUndo.Enabled := Memo.Modified;
  ECut.Enabled := Memo.SelLength <> 0;
  ECopy.Enabled := Memo.SelLength <> 0;
  EPaste.Enabled := Clipboard.HasFormat(CF_TEXT);
  EDelete.Enabled := Memo.SelLength <> 0;
  EWordWrap.Checked := Memo.WordWrap;
end;

procedure TCompileForm.EUndoClick(Sender: TObject);
begin
  SendMessage (Memo.Handle, WM_UNDO, 0, 0);
end;

procedure TCompileForm.ECutClick(Sender: TObject);
begin
  Memo.CutToClipboard;
end;

procedure TCompileForm.ECopyClick(Sender: TObject);
begin
  Memo.CopyToClipboard;
end;

procedure TCompileForm.EPasteClick(Sender: TObject);
begin
  Memo.PasteFromClipboard;
end;

procedure TCompileForm.EDeleteClick(Sender: TObject);
begin
  Memo.ClearSelection;
end;

procedure TCompileForm.ESelectAllClick(Sender: TObject);
begin
  Memo.SelectAll;
end;

procedure TCompileForm.EWordWrapClick(Sender: TObject);
begin
  Memo.WordWrap := not Memo.WordWrap;
  if Memo.WordWrap then
    Memo.ScrollBars := ssVertical
  else
    Memo.ScrollBars := ssBoth;
end;

procedure TCompileForm.VMenuClick(Sender: TObject);
begin
  VToolbar.Checked := ToolbarPanel.Visible;
  VCompilerStatus.Checked := StatusPanel.Visible;
end;

procedure TCompileForm.VToolbarClick(Sender: TObject);
begin
  ToolbarPanel.Visible := not ToolbarPanel.Visible;
end;

procedure TCompileForm.SetStatusPanelVisible (const AVisible: Boolean);
begin
  if StatusPanel.Visible <> AVisible then begin
    SplitPanel.Visible := AVisible;
    StatusPanel.Visible := AVisible;
    if AVisible then
      StatusPanel.Top := ClientHeight;
  end;
end;

procedure TCompileForm.VCompilerStatusClick(Sender: TObject);
begin
  SetStatusPanelVisible (not StatusPanel.Visible);
end;

procedure TCompileForm.HDocClick(Sender: TObject);
begin
  Application.HelpCommand (HELP_FINDER, 0);
end;

procedure TCompileForm.HWebsiteClick(Sender: TObject);
begin
  ShellExecute (Application.Handle, 'open', 'http://www.innosetup.com/', nil,
    nil, SW_SHOW);
end;

procedure TCompileForm.HAboutClick(Sender: TObject);
begin
  { Removing the About box or modifying the text inside it is a violation of the
    Inno Setup license agreement; see LICENSE.TXT. However, adding additional
    lines to the About box is permitted. }
  MsgBox ('Inno Setup Compiler version ' +
    Version + ' (' + CompilerIDEVersion + ')' + SNewLine +
    'Copyright (C) 1998-2001 Jordan Russell. All rights reserved.' + SNewLine2 +
    'Inno Setup home page:' + SNewLine +
    'http://www.innosetup.com/' + SNewLine2 +
    'Refer to LICENSE.TXT for conditions of distribution and use.',
    'About Inno Setup', mbInformation, MB_OK);
end;

procedure TCompileForm.WMStartCommandLineCompile (var Message: TMessage);
var
  Code: Integer;
begin
  UpdateStatusPanelHeight (ClientHeight);
  Code := 0;
  try
    try
      CompileFile (CommandLineFilename, True, False);
    except
      on E: Exception do begin
        Code := 2;
        Application.HandleException (E);
      end;
    end;
  finally
    Halt (Code);
  end;
end;

{ This SearchBuf function is based on the SearchBuf function from the Delphi 1
  demo project "TextDemo". It includes a bug fix, which is noted below. }

function SearchBuf (Buf: PChar; BufLen: Integer; SelStart, SelLength: Integer;
  SearchString: String; Options: TFindOptions): PChar;
const
  { Default word delimiters are any character except the core alphanumerics. }
  WordDelimiters: set of Char = [#0..#255] - ['a'..'z', 'A'..'Z', '0'..'9'];
var
  SearchCount, I: Integer;
  C: Char;
  Direction: Integer;
  CharMap: array[Char] of Char;

  function FindNextWordStart (var BufPtr: PChar): Boolean;
  var
    OrigBufPtr: PChar;
  begin
    OrigBufPtr := BufPtr;
                          { (True XOR N) is equivalent to (not N) }
                          { (False XOR N) is equivalent to (N)    }
     { When Direction is forward (1), skip non delimiters, then skip delimiters. }
     { When Direction is backward (-1), skip delims, then skip non delims }
    while (SearchCount > 0) and
       ((Direction = 1) xor (BufPtr^ in WordDelimiters)) do begin
      Inc (BufPtr, Direction);
      Dec (SearchCount);
    end;
    while (SearchCount > 0) and
       ((Direction = -1) xor (BufPtr^ in WordDelimiters)) do begin
      Inc (BufPtr, Direction);
      Dec (SearchCount);
    end;
    Result := SearchCount >= 0;
    if (Direction = -1) and (BufPtr^ in WordDelimiters) then begin
      { back up one char, to leave ptr on first non delim }
      Dec (BufPtr, Direction);
      Inc (SearchCount);
    end;
    { Following is to fix a bug in D1 TextDemo's Search.pas, which this code
      is based on. If the memo started with a delimeter (e.g. ";") and you did
      a backwards whole word search, it would get caught in an endless loop }
    if Result and ((Direction = 1) and (BufPtr < OrigBufPtr)) or
       ((Direction = -1) and (BufPtr > OrigBufPtr)) then
      Result := False;
  end;
begin
  Result := nil;
  if BufLen <= 0 then Exit;
  if frDown in Options then begin
    Direction := 1;
    Inc(SelStart, SelLength);  { start search past end of selection }
    SearchCount := BufLen - SelStart - Length(SearchString);
    if SearchCount < 0 then Exit;
    if Longint(SelStart) + SearchCount > BufLen then Exit;
  end
  else begin
    Direction := -1;
    Dec(SelStart, Length(SearchString));
    SearchCount := SelStart;
  end;
  if (SelStart < 0) or (SelStart > BufLen) then Exit;
  Result := @Buf[SelStart];

  { Using a Char map array is faster than calling AnsiUpper on every character }
  for C := Low(CharMap) to High(CharMap) do
    CharMap[C] := C;
  if not(frMatchCase in Options) then begin
    AnsiUpperBuff (@CharMap, SizeOf(CharMap));
    AnsiUpperBuff (@SearchString[1], Length(SearchString));
  end;

  while SearchCount >= 0 do begin
    if frWholeWord in Options then
      if not FindNextWordStart(Result) then Break;
    I := 0;
    while CharMap[Result[I]] = SearchString[I+1] do begin
      Inc(I);
      if I >= Length(SearchString) then begin
        if not(frWholeWord in Options) or
           (SearchCount = 0) or
           (Result[I] in WordDelimiters) then
          Exit;
        Break;
      end;
    end;
    Inc (Result, Direction);
    Dec (SearchCount);
  end;
  Result := nil;
end;

function TCompileForm.SearchMemo (const SearchString: String;
  Options: TFindOptions): Boolean;
var
  Buffer, P: PChar;
  Size: Integer;
  H: THandle;
begin
  Result := False;
  if SearchString = '' then Exit;
  Size := Memo.GetTextLen;
  if (Size = 0) then Exit;
  H := 0;
  if Win32Platform = VER_PLATFORM_WIN32_NT then begin
    { On NT, for speed use EM_GETHANDLE to directly access the edit buffer }
    H := SendMessage(Memo.Handle, EM_GETHANDLE, 0, 0);
    Buffer := LocalLock(H);
  end
  else
    Buffer := StrAlloc(Size + 1);
  try
    if Win32Platform <> VER_PLATFORM_WIN32_NT then
      Memo.GetTextBuf (Buffer, Size + 1);
    P := SearchBuf(Buffer, Size, Memo.SelStart, Memo.SelLength,
      SearchString, Options);
    if P <> nil then begin
      Memo.SelStart := P - Buffer;
      Memo.SelLength := Length(SearchString);
      SendMessage (Memo.Handle, EM_SCROLLCARET, 0, 0);
      Result := True;
    end;
  finally
    if Win32Platform = VER_PLATFORM_WIN32_NT then
      LocalUnlock (H)
    else
      StrDispose (Buffer);
  end;
end;

procedure TCompileForm.EFindClick(Sender: TObject);
begin
  ReplaceDialog.CloseDialog;
  FindDialog.Execute;
end;

procedure TCompileForm.EFindNextClick(Sender: TObject);
begin
  if FindDialog.FindText = '' then
    EFindClick (Sender)
  else
    FindDialogFind (FindDialog);
end;

procedure TCompileForm.FindDialogFind(Sender: TObject);
begin
  { this event handler is shared between FindDialog & ReplaceDialog }
  with Sender as TFindDialog do
    if not SearchMemo(FindText, Options) then
      MsgBoxFmt ('Cannot find "%s"', [FindText], '', mbError, MB_OK);
end;

procedure TCompileForm.EReplaceClick(Sender: TObject);
begin
  FindDialog.CloseDialog;
  ReplaceDialog.Execute;
end;

procedure TCompileForm.ReplaceDialogReplace(Sender: TObject);
begin
  with ReplaceDialog do begin
    if AnsiCompareText(Memo.SelText, FindText) = 0 then
      Memo.SelText := ReplaceText;
    if not SearchMemo(FindText, Options) then
      MsgBoxFmt ('Cannot find "%s"', [FindText], '', mbError, MB_OK)
    else
       if frReplaceAll in Options then begin
        repeat
          Memo.SelText := ReplaceText;
        until not SearchMemo(FindText, Options);
      end;
  end;
end;

procedure TCompileForm.UpdateStatusPanelHeight (H: Integer);
var
  H2: Integer;
begin
  H2 := ClientHeight - 48 - SplitPanel.Height;
  if H > H2 then H := H2;
  if H < 48 then H := 48;
  StatusPanel.Height := H;
end;

procedure TCompileForm.SplitPanelMouseMove(Sender: TObject;
  Shift: TShiftState; X, Y: Integer);
var
  P: TPoint;
begin
  if (ssLeft in Shift) and StatusPanel.Visible then begin
    GetCursorPos (P);
    UpdateStatusPanelHeight (ClientToScreen(Point(0, 0)).Y - P.Y +
      ClientHeight - (SplitPanel.Height div 2));
  end;
end;

(*  Works, but not with word wrap on. To be fixed later.
procedure TCompileForm.MemoKeyPress(Sender: TObject; var Key: Char);

  function CountWhitespace (const S: String): Integer;
  var
    I: Integer;
  begin
    Result := 0;
    for I := 1 to Length(S) do
      if S[I] in [#9, ' '] then
        Inc (Result)
      else
        Break;
  end;

  function ExpandAndCountWhitespace (const S: String): Integer;
  var
    I: Integer;
  begin
    Result := 0;
    for I := 1 to Length(S) do
      case S[I] of
        #9:  Result := (Result + 8) and not 7;
        ' ': Inc (Result);
      else
        Break;
      end;
  end;

var
  CaretIndex, LineNum, LineIndex, X, I, N, N2: Integer;
  S: String;
begin
  case Key of
    ^H: if Memo.SelLength = 0 then begin  { "Backspace unindents" feature }
          SendMessage (Memo.Handle, EM_GETSEL, WPARAM(@CaretIndex), 0);
          LineNum := SendMessage(Memo.Handle, EM_LINEFROMCHAR, CaretIndex, 0);
          LineIndex := SendMessage(Memo.Handle, EM_LINEINDEX, LineNum, 0);
          if (LineIndex >= 0) and (CaretIndex > LineIndex) then begin
            { ^ use normal backspace behavior if we're at the first char. on the line }
            X := CaretIndex - LineIndex;
            S := Memo.Lines[LineNum];
            if CountWhitespace(S) = X then begin
              { ^ is there nothing but whitespace preceding the caret? }
              N := ExpandAndCountWhitespace(S);
              { Find a previous line with less preceding whitespace }
              for I := LineNum-1 downto -1 do begin
                if I <> -1 then
                  N2 := ExpandAndCountWhitespace(Memo.Lines[I])
                else
                  N2 := 0;
                if N2 < N then begin
                  { Found one. Replace whitespace on current line to match. }
                  Key := #0;
                  Memo.SelStart := LineIndex;
                  Memo.SelLength := X;
                  Memo.SelText := StringOfChar(' ', N2);
                  Break;
                end;
              end;
            end;
          end;
        end;
    ^M: begin  { "Auto indent" feature }
          Key := #0;
          SendMessage (Memo.Handle, EM_GETSEL, WPARAM(@CaretIndex), 0);
          LineNum := SendMessage(Memo.Handle, EM_LINEFROMCHAR, CaretIndex, 0);
          for I := LineNum-1 downto 0 do begin
            if Memo.Lines[
          end;
          S := Memo.Lines[LineNum];
          Memo.SelText := #13#10 + Copy(S, 1, CountWhitespace(S));
        end;
  end;
end;
*)

procedure TCompileForm.VOptionsClick(Sender: TObject);
var
  OptionsForm: TOptionsForm;
begin
  OptionsForm := TOptionsForm.Create(Application);
  try
    OptionsForm.TestCheck.Checked := Options.TestAfterCompile;
    OptionsForm.BackupCheck.Checked := Options.MakeBackups;

    if OptionsForm.ShowModal <> mrOK then
      Exit;

    Options.TestAfterCompile := OptionsForm.TestCheck.Checked;
    Options.MakeBackups := OptionsForm.BackupCheck.Checked;
  finally
    OptionsForm.Free;
  end;
end;

end.
