;;; w32-fns.el --- Lisp routines for 32-bit Windows  -*- lexical-binding: t; -*-

;; Copyright (C) 1994, 2001-2021 Free Software Foundation, Inc.

;; Author: Geoff Voelker <voelker@cs.washington.edu>
;; Keywords: internal
;; Package: emacs

;; This file is part of GNU Emacs.

;; GNU Emacs is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs.  If not, see <https://www.gnu.org/licenses/>.

;;; Commentary:


;;; Code:
(require 'w32-vars)

(defvar explicit-shell-file-name)

;;;; Function keys

(declare-function w32-get-locale-info "w32proc.c" (lcid &optional longform))
(declare-function w32-get-valid-locale-ids "w32proc.c" ())

(if (eq system-type 'windows-nt)
    ;; Map all versions of a filename (8.3, longname, mixed case) to the
    ;; same buffer.
    (setq find-file-visit-truename t))

;;;; Shells

(defun w32-shell-name ()
  "Return the name of the shell being used."
  (or (bound-and-true-p shell-file-name)
      (getenv "ESHELL")
      (getenv "SHELL")
      (and (fboundp 'w32-using-nt) (w32-using-nt) "cmd.exe")
      "command.com"))

(defun w32-system-shell-p (shell-name)
  (and shell-name
       (member (downcase (file-name-nondirectory shell-name))
	       w32-system-shells)))

(defun w32-shell-dos-semantics ()
  "Return non-nil if current interactive shell expects MS-DOS shell semantics."
  (or (w32-system-shell-p (w32-shell-name))
      (and (member (downcase (file-name-nondirectory (w32-shell-name)))
		   '("cmdproxy" "cmdproxy.exe"))
	   (w32-system-shell-p (getenv "COMSPEC")))))

(defvar w32-quote-process-args)  ;; defined in w32proc.c

(defun w32-check-shell-configuration ()
  "Check the configuration of shell variables on Windows.
This function is invoked after loading the init files and processing
the command line arguments.  It issues a warning if the user or site
has configured the shell with inappropriate settings."
  (interactive)
  (let ((prev-buffer (current-buffer))
	(buffer (get-buffer-create "*Shell Configuration*"))
	(system-shell))
    (set-buffer buffer)
    (erase-buffer)
    (if (w32-system-shell-p (getenv "ESHELL"))
	(insert (format "Warning! The ESHELL environment variable uses %s.
You probably want to change it so that it uses cmdproxy.exe instead.\n\n"
			(getenv "ESHELL"))))
    (if (w32-system-shell-p (getenv "SHELL"))
	(insert (format "Warning! The SHELL environment variable uses %s.
You probably want to change it so that it uses cmdproxy.exe instead.\n\n"
			(getenv "SHELL"))))
    (if (w32-system-shell-p shell-file-name)
	(insert (format "Warning! shell-file-name uses %s.
You probably want to change it so that it uses cmdproxy.exe instead.\n\n"
			shell-file-name)))
    (if (and (boundp 'explicit-shell-file-name)
	     (w32-system-shell-p explicit-shell-file-name))
	(insert (format "Warning! explicit-shell-file-name uses %s.
You probably want to change it so that it uses cmdproxy.exe instead.\n\n"
			explicit-shell-file-name)))
    (setq system-shell (> (buffer-size) 0))

    ;; Allow user to specify that they really do want to use one of the
    ;; "system" shells, despite the drawbacks, but still warn if
    ;; shell-command-switch doesn't match.
    (if w32-allow-system-shell
	(erase-buffer))

    (cond (system-shell
	   ;; System shells.
	   (if (string-equal "-c" shell-command-switch)
	       (insert "Warning! shell-command-switch is \"-c\".
You should set this to \"/c\" when using a system shell.\n\n"))
	   (if w32-quote-process-args
	       (insert "Warning! w32-quote-process-args is t.
You should set this to nil when using a system shell.\n\n")))
	  ;; Non-system shells.
	  (t
	   (if (string-equal "/c" shell-command-switch)
	       (insert "Warning! shell-command-switch is \"/c\".
You should set this to \"-c\" when using a non-system shell.\n\n"))
	   (if (not w32-quote-process-args)
	       (insert "Warning! w32-quote-process-args is nil.
You should set this to t when using a non-system shell.\n\n"))))
    (if (> (buffer-size) 0)
	(display-buffer buffer)
      (kill-buffer buffer))
    (set-buffer prev-buffer)))

(add-hook 'after-init-hook 'w32-check-shell-configuration)

;;;; Coding-systems, locales, etc.

;; Override setting chosen at startup.
(defun w32-set-default-process-coding-system ()
  ;; Most programs on Windows will accept Unix line endings on input
  ;; (and some programs ported from Unix require it) but most will
  ;; produce DOS line endings on output.
  (setq default-process-coding-system
	'(undecided-dos . undecided-unix))
  ;; Make cmdproxy default to using DOS line endings for input,
  ;; because some Windows programs (including command.com) require it.
  (add-to-list 'process-coding-system-alist
	       '("[cC][mM][dD][pP][rR][oO][xX][yY]"
		 . (undecided-dos . undecided-dos)))
  ;; plink needs DOS input when entering the password.
  (add-to-list 'process-coding-system-alist
	       '("[pP][lL][iI][nN][kK]"
		 . (undecided-dos . undecided-dos))))
(define-obsolete-function-alias 'set-default-process-coding-system
  #'w32-set-default-process-coding-system "26.1")
(add-hook 'before-init-hook #'w32-set-default-process-coding-system)


;;; Basic support functions for managing Emacs's locale setting

(defvar w32-valid-locales nil
  "List of locale ids known to be supported.")

;; This is the brute-force version; an efficient version is now
;; built-in though.
(if (not (fboundp 'w32-get-valid-locale-ids))
    (defun w32-get-valid-locale-ids ()
      "Return list of all valid Windows locale ids."
      (let ((i 65535)
	    locales)
	(while (> i 0)
	  (if (w32-get-locale-info i)
	      (setq locales (cons i locales)))
	  (setq i (1- i)))
	locales)))

(defun w32-list-locales ()
  "List the name and id of all locales supported by Windows."
  (interactive)
  (when (null w32-valid-locales)
    (setq w32-valid-locales (sort (w32-get-valid-locale-ids) #'<)))
  (with-output-to-temp-buffer "*Supported Locales*"
    (princ "LCID\tAbbrev\tFull name\n\n")
    (dolist (locale w32-valid-locales)
      (princ (format "%d\t%s\t%s\n"
		     locale
		     (w32-get-locale-info locale)
		     (w32-get-locale-info locale t))))))

;; The variable source-directory is used to initialize Info-directory-list.
;; However, the common case is that Emacs is being used from a binary
;; distribution, and the value of source-directory is meaningless in that
;; case.  Even worse, source-directory can refer to a directory on a drive
;; on the build machine that happens to be a removable drive on the user's
;; machine.  When this happens, Emacs tries to access the removable drive
;; and produces the abort/retry/ignore dialog.  Since we do not use
;; source-directory, set it to something that is a reasonable approximation
;; on the user's machine.

;;(add-hook 'before-init-hook
;;	  (lambda ()
;;	    (setq source-directory (file-name-as-directory
;;				     (expand-file-name ".." exec-directory)))))

(defun w32-set-system-coding-system (coding-system)
  "Set the coding system used by the Windows system to CODING-SYSTEM.
This is used for things like passing font names with non-ASCII
characters in them to the system.  For a list of possible values of
CODING-SYSTEM, use \\[list-coding-systems].

This function is provided for backward compatibility, since
`w32-system-coding-system' is now an alias for `locale-coding-system'."
  (interactive
   (list (let ((default locale-coding-system))
           (read-coding-system
            (format-prompt "Coding system for system calls" default)
            default))))
  (check-coding-system coding-system)
  (setq locale-coding-system coding-system))
(define-obsolete-function-alias 'set-w32-system-coding-system
  #'w32-set-system-coding-system "26.1")

;; locale-coding-system was introduced to do the same thing as
;; w32-system-coding-system. Use that instead.
(defvaralias 'w32-system-coding-system 'locale-coding-system)

;; Set to a system sound if you want a fancy bell.
(if (fboundp 'set-message-beep)         ; w32fns.c
    (set-message-beep nil))

(defvar w32-charset-info-alist)         ; w32font.c

(defun w32-add-charset-info (xlfd-charset windows-charset codepage)
  "Function to add character sets to display with Windows fonts.
Creates entries in `w32-charset-info-alist'.
XLFD-CHARSET is a string which will appear in the XLFD font name to
identify the character set.  WINDOWS-CHARSET is a symbol identifying
the Windows character set this maps to.  For the list of possible
values, see the documentation for `w32-charset-info-alist'.  CODEPAGE
can be a numeric codepage that Windows uses to display the character
set, t for Unicode output with no codepage translation or nil for 8
bit output with no translation."
  (add-to-list 'w32-charset-info-alist
               (cons xlfd-charset (cons windows-charset codepage))))

(when (boundp 'w32-charset-info-alist)
  ;; The last charset we add becomes the "preferred" charset for the return
  ;; value from x-select-font etc, so list the most important charsets last.
  (w32-add-charset-info "iso8859-14" 'w32-charset-ansi  28604)
  (w32-add-charset-info "iso8859-15" 'w32-charset-ansi  28605)
  (w32-add-charset-info "iso8859-16" 'w32-charset-ansi  28606)
  ;; The following two are included for pattern matching.
  (w32-add-charset-info "jisx0201" 'w32-charset-shiftjis 932)
  (w32-add-charset-info "jisx0208" 'w32-charset-shiftjis 932)
  (w32-add-charset-info "jisx0201-latin" 'w32-charset-shiftjis 932)
  (w32-add-charset-info "jisx0201-katakana" 'w32-charset-shiftjis 932)
  (w32-add-charset-info "jisx0212" 'w32-charset-shiftjis 932)
  (w32-add-charset-info "ksc5601.1989" 'w32-charset-hangeul 949)
  (w32-add-charset-info "ksx1001" 'w32-charset-hangeul 949)
  (w32-add-charset-info "big5" 'w32-charset-chinesebig5 950)
  (w32-add-charset-info "gb2312.1980" 'w32-charset-gb2312 936)
  (w32-add-charset-info "gbk" 'w32-charset-gb2312 936)
  (w32-add-charset-info "ms-symbol" 'w32-charset-symbol nil)
  (w32-add-charset-info "ms-oem" 'w32-charset-oem 437)
  (w32-add-charset-info "ms-oemlatin" 'w32-charset-oem 850)
  (w32-add-charset-info "iso8859-2" 'w32-charset-easteurope 28592)
  (w32-add-charset-info "iso8859-3" 'w32-charset-turkish 28593)
  (w32-add-charset-info "iso8859-4" 'w32-charset-baltic 28594)
  (w32-add-charset-info "iso8859-6" 'w32-charset-arabic 28596)
  (w32-add-charset-info "iso8859-7" 'w32-charset-greek 28597)
  (w32-add-charset-info "iso8859-8" 'w32-charset-hebrew 1255)
  (w32-add-charset-info "iso8859-9" 'w32-charset-turkish 1254)
  (w32-add-charset-info "iso8859-13" 'w32-charset-baltic 1257)
  (w32-add-charset-info "koi8-r" 'w32-charset-russian 20866)
  (w32-add-charset-info "microsoft-cp1251" 'w32-charset-russian 1251)
  (w32-add-charset-info "windows-1251" 'w32-charset-russian 1251)
  (w32-add-charset-info "tis620-2533" 'w32-charset-russian 28595)
  (w32-add-charset-info "iso8859-11" 'w32-charset-thai 874)
  (w32-add-charset-info "windows-1258" 'w32-charset-vietnamese 1258)
  (w32-add-charset-info "viscii" 'w32-charset-vietnamese 1258)
  (w32-add-charset-info "ksc5601.1992" 'w32-charset-johab 1361)
  (w32-add-charset-info "mac-roman" 'w32-charset-mac 10000)
  (w32-add-charset-info "iso10646-1" 'w32-charset-default t)

  ;;   ;; If Unicode Windows charset is not defined, use ansi fonts.
  ;;   (w32-add-charset-info "iso10646-1" 'w32-charset-ansi t))

  ;; Preferred names
  (w32-add-charset-info "big5-0" 'w32-charset-chinesebig5 950)
  (w32-add-charset-info "gb2312.1980-0" 'w32-charset-gb2312 936)
  (w32-add-charset-info "jisx0208-sjis" 'w32-charset-shiftjis 932)
  (w32-add-charset-info "ksc5601.1987-0" 'w32-charset-hangeul 949)
  (w32-add-charset-info "tis620-0" 'w32-charset-thai 874)
  (w32-add-charset-info "iso8859-1" 'w32-charset-ansi 1252))

;;;; Standard filenames

(defun w32-convert-standard-filename (filename)
  "Convert a standard file's name to something suitable for MS-Windows.
This means to guarantee valid names and perhaps to canonicalize
certain patterns.

This function is called by `convert-standard-filename'.

Replace invalid characters and turn Cygwin names into native
names."
  (save-match-data
    (let ((name
	   (if (string-match "\\`/cygdrive/\\([a-zA-Z]\\)/" filename)
               (replace-match "\\1:/" t nil filename)
             (copy-sequence filename)))
	  (start 0))
      ;; leave ':' if part of drive specifier
      (if (and (> (length name) 1)
	       (eq (aref name 1) ?:))
	  (setq start 2))
      ;; destructively replace invalid filename characters with !
      (while (string-match "[?*:<>|\"\000-\037]" name start)
	(aset name (match-beginning 0) ?!)
	(setq start (match-end 0)))
      name)))

;;;; System name and version for emacsbug.el

(declare-function w32-version "w32-win" ())
(declare-function w32-read-registry "w32fns" (root key name))

(defun w32--os-description ()
  "Return a string describing the underlying OS and its version."
  (let* ((w32ver (car (w32-version)))
         (w9x-p (< w32ver 5))
         (key (if w9x-p
                  "SOFTWARE/Microsoft/Windows/CurrentVersion"
                "SOFTWARE/Microsoft/Windows NT/CurrentVersion"))
         (os-name (w32-read-registry 'HKLM key "ProductName"))
         (os-version (if w9x-p
                         (w32-read-registry 'HKLM key "VersionNumber")
                       (let ((vmajor
                              (w32-read-registry 'HKLM key
                                                 "CurrentMajorVersionNumber"))
                             (vminor
                              (w32-read-registry 'HKLM key
                                                 "CurrentMinorVersionNumber")))
                         (if (and vmajor vmajor)
                             (format "%d.%d" vmajor vminor)
                           (w32-read-registry 'HKLM key "CurrentVersion")))))
         (os-csd (w32-read-registry 'HKLM key "CSDVersion"))
         (os-rel (or (w32-read-registry 'HKLM key "ReleaseID")
                     (w32-read-registry 'HKLM key "CSDBuildNumber")
                 "0"))  ; No Release ID before Windows Vista
         (os-build (w32-read-registry 'HKLM key "CurrentBuildNumber"))
         (os-rev (w32-read-registry 'HKLM key "UBR"))
         (os-rev (if os-rev (format "%d" os-rev))))
    (if w9x-p
        (concat
         (if (not (string-match "\\`Microsoft " os-name)) "Microsoft ")
         os-name
         " (v" os-version ")")
      (concat
       (if (not (string-match "\\`Microsoft " os-name)) "Microsoft ")
       os-name      ; Windows 7 Enterprise
       " "
       os-csd       ; Service Pack 1
       (if (and os-csd (> (length os-csd) 0)) " " "")
       "(v"
       os-version "." os-rel "." os-build (if os-rev (concat "." os-rev))
       ")"))))


;;;; Support for build process

;; From autoload.el
(defvar autoload-make-program)
(defvar generated-autoload-file)

(defun w32-batch-update-autoloads ()
  "Like `batch-update-autoloads', but takes the name of the autoloads file
from the command line.

This is required because some Windows build environments, such as MSYS,
munge command-line arguments that include file names to a horrible mess
that Emacs is unable to cope with."
  (let ((generated-autoload-file
	 (expand-file-name (pop command-line-args-left)))
	;; I can only assume the same considerations may apply here...
	(autoload-make-program (pop command-line-args-left)))
    (batch-update-autoloads)))

(defun w32-append-code-lines (orig extra)
  "Append non-empty non-comment lines in the file EXTRA to the file ORIG.

This function saves all buffers and kills the Emacs session, without asking
for any permissions.

This is required because the Windows build environment is not required
to include Sed, which is used by leim/Makefile.in to do the job."
  (with-current-buffer (find-file-noselect orig)
    (goto-char (point-max))
    (insert-file-contents extra)
    (delete-matching-lines "^$\\|^;")
    (save-buffers-kill-emacs t)))

;;; w32-fns.el ends here
