##===----------------------------------------------------------------------===##
#
# Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
# See https://llvm.org/LICENSE.txt for license information.
# SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
#
##===----------------------------------------------------------------------===##
#
# Find OpenMP Target offloading Support for various compilers.
#
##===----------------------------------------------------------------------===##

#[========================================================================[.rst:
FindOpenMPTarget
----------------

Finds OpenMP Target Offloading Support.

This module can be used to detect OpenMP target offloading support in a
compiler. If the compiler support OpenMP Offloading to a specified target, the
flags required to compile offloading code to that target are output for each
target.

This module will automatically include OpenMP support if it was not loaded
already. It does not need to be included separately to get full OpenMP support.

Variables
^^^^^^^^^

The module exposes the components ``NVPTX`` and ``AMDGPU``.  Each of these
controls the various offloading targets to search OpenMP target offloasing
support for.

Depending on the enabled components the following variables will be set:

``OpenMPTarget_FOUND``
  Variable indicating that OpenMP target offloading flags for all requested
  targets have been found.

This module will set the following variables per language in your
project, where ``<device>`` is one of NVPTX or AMDGPU

``OpenMPTarget_<device>_FOUND``
  Variable indicating if OpenMP support for the ``<device>`` was detected.
``OpenMPTarget_<device>_FLAGS``
  OpenMP compiler flags for offloading to ``<device>``, separated by spaces.

For linking with OpenMP code written in ``<device>``, the following
variables are provided:

``OpenMPTarget_<device>_LIBRARIES``
  A list of libraries needed to link with OpenMP code written in ``<lang>``.

Additionally, the module provides :prop_tgt:`IMPORTED` targets:

``OpenMPTarget::OpenMPTarget_<device>``
  Target for using OpenMP offloading to ``<device>``.

If the specific architecture of the target is needed, it can be manually
specified by setting a variable to the desired architecture. Variables can also
be used to override the standard flag searching for a given compiler.

``OpenMPTarget_<device>_ARCH``
  Sets the architecture of ``<device>`` to compile for. Such as `sm_70` for NVPTX
  or `gfx908` for AMDGPU. 

``OpenMPTarget_<device>_DEVICE``
  Sets the name of the device to offload to.

``OpenMPTarget_<device>_FLAGS``
  Sets the compiler flags for offloading to ``<device>``.

#]========================================================================]

# TODO: Support Fortran
# TODO: Support multiple offloading targets by setting the "OpenMPTarget" target
#       to include flags for all components loaded
# TODO: Configure target architecture without a variable (component NVPTX_SM_70)
# TODO: Test more compilers

cmake_policy(PUSH)
cmake_policy(VERSION 3.20.0)

find_package(OpenMP ${OpenMPTarget_FIND_VERSION} REQUIRED)

# Find the offloading flags for each compiler.
function(_OPENMP_TARGET_DEVICE_FLAG_CANDIDATES LANG DEVICE)
  if(NOT OpenMPTarget_${LANG}_FLAGS)
    unset(OpenMPTarget_FLAG_CANDIDATES)

    set(OMPTarget_FLAGS_Clang "-fopenmp-targets=${DEVICE}")
    set(OMPTarget_FLAGS_GNU "-foffload=${DEVICE}=\"-lm -latomic\"")
    set(OMPTarget_FLAGS_XL "-qoffload")
    set(OMPTarget_FLAGS_PGI "-mp=${DEVICE}")
    set(OMPTarget_FLAGS_NVHPC "-mp=${DEVICE}")

    if(DEFINED OMPTarget_FLAGS_${CMAKE_${LANG}_COMPILER_ID})
      set(OpenMPTarget_FLAG_CANDIDATES "${OMPTarget_FLAGS_${CMAKE_${LANG}_COMPILER_ID}}")
    endif()

    set(OpenMPTarget_${LANG}_FLAG_CANDIDATES "${OpenMPTarget_FLAG_CANDIDATES}" PARENT_SCOPE)
  else()
    set(OpenMPTarget_${LANG}_FLAG_CANDIDATES "${OpenMPTarget_${LANG}_FLAGS}" PARENT_SCOPE)
  endif()
endfunction()

# Get the coded name of the device for each compiler.
function(_OPENMP_TARGET_DEVICE_CANDIDATES LANG DEVICE)
  if (NOT OpenMPTarget_${DEVICE}_DEVICE)
    unset(OpenMPTarget_DEVICE_CANDIDATES)

    # Check each supported device.
    if("${DEVICE}" STREQUAL "NVPTX")
      if ("${CMAKE_SIZEOF_VOID_P}" STREQUAL "4")
        set(OMPTarget_DEVICE_Clang "nvptx32-nvidia-cuda")
      else()
        set(OMPTarget_DEVICE_Clang "nvptx64-nvidia-cuda")
      endif()
      set(OMPTarget_DEVICE_GNU "nvptx-none")
      set(OMPTarget_DEVICE_XL "")
      set(OMPTarget_DEVICE_PGI "gpu")
      set(OMPTarget_DEVICE_NVHPC "gpu")

      if(DEFINED OMPTarget_DEVICE_${CMAKE_${LANG}_COMPILER_ID})
        set(OpenMPTarget_DEVICE_CANDIDATES "${OMPTarget_DEVICE_${CMAKE_${LANG}_COMPILER_ID}}")
      endif()
    elseif("${DEVICE}" STREQUAL "AMDGPU")
      set(OMPTarget_DEVICE_Clang "amdgcn-amd-amdhsa")
      set(OMPTarget_DEVICE_GNU "hsa")

      if(DEFINED OMPTarget_DEVICE_${CMAKE_${LANG}_COMPILER_ID})
        set(OpenMPTarget_DEVICE_CANDIDATES "${OMPTarget_DEVICE_${CMAKE_${LANG}_COMPILER_ID}}")
      endif()
    endif()
    set(OpenMPTarget_${LANG}_DEVICE_CANDIDA                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              